<?php
/**
 * Custom Facebook Feed Builder
 *
 * @since 6.0
 */
namespace InstagramFeed\Builder;

use InstagramFeed\Admin\SBI_Callout;
use InstagramFeed\Builder\Tabs\SBI_Styling_Tab;
use InstagramFeed\Builder\SBI_Feed_Saver;
use InstagramFeed\Helpers\Util;


class SBI_Feed_Builder {
	private static $instance;
	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
			return self::$instance;

		}
	}


	/**
	 * Constructor.
	 *
	 * @since 6.0
	 */
	public function __construct() {
		$this->init();
	}

	/**
	 * Init the Builder.
	 *
	 * @since 6.0
	 */
	function init() {
		if ( is_admin() ) {
			add_action( 'admin_menu', array( $this, 'register_menu' ) );
			// add ajax listeners
			SBI_Feed_Saver_Manager::hooks();
			SBI_Source::hooks();
			self::hooks();
		}
	}

	/**
	 * Mostly AJAX related hooks
	 *
	 * @since 6.0
	 */
	public static function hooks() {
		add_action( 'wp_ajax_sbi_dismiss_onboarding', array( 'InstagramFeed\Builder\SBI_Feed_Builder', 'after_dismiss_onboarding' ) );

		add_action( 'wp_ajax_sbi_other_plugins_modal', array( 'InstagramFeed\Builder\SBI_Feed_Builder', 'sb_other_plugins_modal' ) );

	}

	/**
	 * Register Menu.
	 *
	 * @since 6.0
	 */
	public function register_menu() {
		$cap = current_user_can( 'manage_instagram_feed_options' ) ? 'manage_instagram_feed_options' : 'manage_options';
		$cap = apply_filters( 'sbi_settings_pages_capability', $cap );

		$feed_builder = add_submenu_page(
			'sb-instagram-feed',
			__( 'All Feeds', 'instagram-feed' ),
			__( 'All Feeds', 'instagram-feed' ),
			$cap,
			'sbi-feed-builder',
			array( $this, 'feed_builder' ),
			0
		);
		add_action( 'load-' . $feed_builder, array( $this, 'builder_enqueue_admin_scripts' ) );
	}

	/**
	 * Enqueue Builder CSS & Script.
	 *
	 * Loads only for builder pages
	 *
	 * @since 6.0
	 */
	public function builder_enqueue_admin_scripts() {
		if ( get_current_screen() ) :
			$screen = get_current_screen();
			if ( strpos( $screen->id, 'sbi-feed-builder' ) !== false || strpos( $screen->id, 'sbi-setup' ) !== false ) :
				$installed_plugins = get_plugins();

				$newly_retrieved_source_connection_data = SBI_Source::maybe_source_connection_data();
				$license_key                            = get_option( 'sbi_license_key', '' );
				$upgrade_url                            = 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=lite-upgrade-bar';

				$active_extensions = array(
					// Fake
					'feedLayout'       => false,
					'headerLayout'     => false,
					'postStyling'      => false,
					'lightbox'         => false,
					'filtermoderation' => false,
					'shoppablefeed'    => false,

				);

				$sbi_builder = array(
					'ajax_handler'         => admin_url( 'admin-ajax.php' ),
					'pluginType'           => 'free',
					'licenseType'          => sbi_is_pro_version() ? 'pro' : 'free',
					'isSetupPage'			=> strpos( $screen->id, 'sbi-setup' ) !== false ? 'true' : 'false',
					'builderUrl'           => admin_url( 'admin.php?page=sbi-feed-builder' ),
					'setUpUrl'           => admin_url( 'admin.php?page=sbi-setup' ),
					'upgradeUrl'           => $upgrade_url,
					'activeExtensions'     => $active_extensions,
					'pluginUrl'            => trailingslashit( SBI_PLUGIN_URL ),

					'nonce'                => wp_create_nonce( 'sbi-admin' ),
					'admin_nonce'             => wp_create_nonce( 'sbi_admin_nonce' ),

					'adminPostURL'         => admin_url( 'post.php' ),
					'widgetsPageURL'       => admin_url( 'widgets.php' ),
					'themeSupportsWidgets' => current_theme_supports( 'widgets' ),
					'supportPageUrl'       => admin_url( 'admin.php?page=sbi-support' ),
					'genericText'          => self::get_generic_text(),
					'welcomeScreen'        => array(
						'mainHeading'              => __( 'All Feeds', 'instagram-feed' ),
						'createFeed'               => __( 'Create your Feed', 'instagram-feed' ),
						'createFeedDescription'    => __( 'Connect your Instagram account and choose a feed type', 'instagram-feed' ),
						'customizeFeed'            => __( 'Customize your feed type', 'instagram-feed' ),
						'customizeFeedDescription' => __( 'Choose layouts, color schemes, styles and more', 'instagram-feed' ),
						'embedFeed'                => __( 'Embed your feed', 'instagram-feed' ),
						'embedFeedDescription'     => __( 'Easily add the feed anywhere on your website', 'instagram-feed' ),
						'customizeImgPath'         => SBI_BUILDER_URL . 'assets/img/welcome-1.png',
						'embedImgPath'             => SBI_BUILDER_URL . 'assets/img/welcome-2.png',
					),
					'pluginsInfo'          => array(
						'social_wall' => array(
							'installed'    => isset( $installed_plugins['social-wall/social-wall.php'] ) ? true : false,
							'activated'    => is_plugin_active( 'social-wall/social-wall.php' ),
							'settingsPage' => admin_url( 'admin.php?page=sbsw' ),
						)
					),
					'allFeedsScreen'       => array(
						'mainHeading'     => __( 'All Feeds', 'instagram-feed' ),
						'columns'         => array(
							'nameText'      => __( 'Name', 'instagram-feed' ),
							'shortcodeText' => __( 'Shortcode', 'instagram-feed' ),
							'instancesText' => __( 'Instances', 'instagram-feed' ),
							'actionsText'   => __( 'Actions', 'instagram-feed' ),
						),
						'bulkActions'     => __( 'Bulk Actions', 'instagram-feed' ),
						'legacyFeeds'     => array(
							'heading'               => __( 'Legacy Feeds', 'instagram-feed' ),
							'toolTip'               => __( 'What are Legacy Feeds?', 'instagram-feed' ),
							'toolTipExpanded'       => array(
								__( 'Legacy feeds are older feeds from before the version 6 update. You can edit settings for these feeds by using the "Settings" button to the right. These settings will apply to all legacy feeds, just like the settings before version 6, and work in the same way that they used to.', 'instagram-feed' ),
								__( 'You can also create a new feed, which will now have it\'s own individual settings. Modifying settings for new feeds will not affect other feeds.', 'instagram-feed' ),
							),
							'toolTipExpandedAction' => array(
								__( 'Legacy feeds represent shortcodes of old feeds found on your website before <br/>the version 6 update.', 'instagram-feed' ),
								__( 'To edit Legacy feed settings, you will need to use the "Settings" button above <br/>or edit their shortcode settings directly. To delete a legacy feed, simply remove the <br/>shortcode wherever it is being used on your site.', 'instagram-feed' ),
							),
							'show'                  => __( 'Show Legacy Feeds', 'instagram-feed' ),
							'hide'                  => __( 'Hide Legacy Feeds', 'instagram-feed' ),
						),
						'socialWallLinks' => self::get_social_wall_links(),
						'onboarding'      => $this->get_onboarding_text()
					),
					'dialogBoxPopupScreen' => array(
						'deleteSource' => array(
							'heading' =>  __( 'Delete "#"?', 'instagram-feed' ),
							'description' => __( 'This source is being used in a feed on your site. If you delete this source then new posts can no longer be retrieved for these feeds.', 'instagram-feed' ),
						),
						'deleteSourceCustomizer' => array(
							'heading'     => __( 'Delete "#"?', 'instagram-feed' ),
							'description' => __( 'You are going to delete this source. To retrieve it, you will need to add it again. Are you sure you want to continue?', 'instagram-feed' ),
						),
						'deleteSingleFeed'       => array(
							'heading'     => __( 'Delete "#"?', 'instagram-feed' ),
							'description' => __( 'You are going to delete this feed. You will lose all the settings. Are you sure you want to continue?', 'instagram-feed' ),
						),
						'deleteMultipleFeeds'    => array(
							'heading'     => __( 'Delete Feeds?', 'instagram-feed' ),
							'description' => __( 'You are going to delete these feeds. You will lose all the settings. Are you sure you want to continue?', 'instagram-feed' ),
						),
						'backAllToFeed'          => array(
							'heading'       => __( 'Are you Sure?', 'instagram-feed' ),
							'description'   => __( 'Are you sure you want to leave this page, all unsaved settings will be lost, please make sure to save before leaving.', 'instagram-feed' ),
							'customButtons' => array(
								'confirm' => array(
									'text'  => __( 'Save and Exit', 'instagram-feed' ),
									'color' => 'blue',
								),
								'cancel'  => array(
									'text'  => __( 'Exit without Saving', 'instagram-feed' ),
									'color' => 'red',
								),
							),
						),
						'unsavedFeedSources'     => array(
							'heading'       => __( 'You have unsaved changes', 'instagram-feed' ),
							'description'   => __( 'If you exit without saving, all the changes you made will be reverted.', 'instagram-feed' ),
							'customButtons' => array(
								'confirm' => array(
									'text'  => __( 'Save and Exit', 'instagram-feed' ),
									'color' => 'blue'
								),
								'cancel'  => array(
									'text'  => __( 'Exit without Saving', 'instagram-feed' ),
									'color' => 'red'
								)
							)
						)
					),
					'selectFeedTypeScreen' => array(
						'mainHeading'            => __( 'Create an Instagram Feed', 'instagram-feed' ),
						'feedTypeHeading'        => __( 'Select Feed Type', 'instagram-feed' ),
						'mainDescription'        => __( 'Select one or more feed types. You can add or remove them later.', 'instagram-feed' ),
						'updateHeading'          => __( 'Update Feed Type', 'instagram-feed' ),
						'advancedHeading'        => __( 'Advanced Feeds', 'instagram-feed' ),
						'anotherFeedTypeHeading' => __( 'Add Another Source Type', 'instagram-feed' ),
					),
					'mainFooterScreen'     => array(
						'heading'     => sprintf( __( 'Upgrade to the %1$sAll Access Bundle%2$s to get all of our Pro Plugins', 'instagram-feed' ), '<strong>', '</strong>' ),
						'description' => __( 'Includes all Smash Balloon plugins for one low price: Instagram, Facebook, Twitter, YouTube, and Social Wall', 'instagram-feed' ),
						'promo'       => sprintf( __( '%1$sBonus%2$s Lite users get %3$s50&#37; Off%4$s automatically applied at checkout', 'instagram-feed' ), '<span class="sbi-bld-ft-bns">', '</span>', '<strong>', '</strong>' ),
					),
					'embedPopupScreen'     => array(
						'heading'       => __( 'Embed Feed', 'instagram-feed' ),
						'description'   => __( 'Add the unique shortcode to any page, post, or widget:', 'instagram-feed' ),
						'description_2' => current_theme_supports( 'widgets' ) ? __( 'Or use the built in WordPress block or widget', 'instagram-feed' ) : __( 'Or use the built in WordPress block', 'instagram-feed' ),
						'addPage'       => __( 'Add to a Page', 'instagram-feed' ),
						'addWidget'     => __( 'Add to a Widget', 'instagram-feed' ),
						'selectPage'    => __( 'Select Page', 'instagram-feed' ),
					),
					'links'                => self::get_links_with_utm(),
					'pluginsInfo'          => array(
						'social_wall' => array(
							'installed'    => isset( $installed_plugins['social-wall/social-wall.php'] ) ? true : false,
							'activated'    => is_plugin_active( 'social-wall/social-wall.php' ),
							'settingsPage' => admin_url( 'admin.php?page=sbsw' ),
						)
					),
					'selectSourceScreen'   => self::select_source_screen_text(),
					'feedTypes'            => $this->get_feed_types(),
					'advancedFeedTypes'    => $this->get_advanced_feed_types(),
					'socialInfo'           => $this->get_smashballoon_info(),
					'svgIcons'             => $this->builder_svg_icons(),
					'installPluginsPopup'  => $this->install_plugins_popup(),
					'feeds'                => self::get_feed_list(),
					'itemsPerPage'         => SBI_Db::RESULTS_PER_PAGE,
					'feedsCount'           => SBI_Db::feeds_count(),
					'sources'              => self::get_source_list(),
					'sourceConnectionURLs' => SBI_Source::get_connection_urls(),

					'legacyFeeds'          => $this->get_legacy_feed_list(),
					'extensionsPopup'      => array(
						'hashtag'          => array(
							'heading'         => __( 'Upgrade to Pro to get Hashtag Feeds', 'instagram-feed' ),
							'description'     => __( 'Display posts from any public hashtag with an Instagram hashtag feed. Great for pulling in user-generated content associated with your brand, running promotional hashtag campaigns, engaging audiences at events, and more.', 'instagram-feed' ),
							'img'             => '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#cliphashtagpro0_506_22510)"><g clip-path="url(#cliphashtagpro1_506_22510)"><g filter="url(#filterhashtagpro0_ddd_506_22510)"><rect x="169.506" y="31.2762" width="158.634" height="180.446" rx="3.96584" transform="rotate(4 169.506 31.2762)" fill="white"/></g><g clip-path="url(#cliphashtagpro2_506_22510)"><rect x="169.506" y="31.2762" width="158.634" height="158.634" transform="rotate(4 169.506 31.2762)" fill="#0096CC"/></g><path d="M192.49 182.337L188.63 182.067L189.561 178.756C189.585 178.659 189.587 178.557 189.568 178.458C189.549 178.359 189.509 178.266 189.45 178.184C189.391 178.102 189.315 178.034 189.228 177.984C189.14 177.934 189.043 177.904 188.943 177.896L187.552 177.798C187.391 177.783 187.23 177.824 187.097 177.914C186.963 178.004 186.864 178.138 186.818 178.292L185.855 181.873L182.099 181.611L183.029 178.3C183.053 178.204 183.056 178.105 183.038 178.008C183.021 177.911 182.983 177.819 182.927 177.738C182.871 177.657 182.799 177.588 182.715 177.537C182.631 177.486 182.537 177.453 182.439 177.441L181.048 177.343C180.888 177.328 180.727 177.369 180.593 177.459C180.459 177.549 180.361 177.683 180.315 177.837L179.365 181.42L175.192 181.128C175.029 181.113 174.867 181.156 174.733 181.249C174.598 181.341 174.501 181.478 174.457 181.636L174.075 183.007C174.047 183.107 174.042 183.213 174.06 183.316C174.079 183.419 174.12 183.517 174.181 183.602C174.242 183.686 174.322 183.756 174.414 183.806C174.506 183.856 174.608 183.885 174.712 183.89L178.572 184.16L177.043 189.645L172.869 189.353C172.707 189.338 172.544 189.381 172.41 189.474C172.276 189.567 172.179 189.703 172.135 189.861L171.752 191.232C171.724 191.333 171.719 191.438 171.738 191.541C171.756 191.644 171.797 191.742 171.859 191.827C171.92 191.911 171.999 191.981 172.091 192.031C172.183 192.081 172.285 192.11 172.39 192.115L176.278 192.387L175.347 195.698C175.323 195.797 175.321 195.901 175.341 196.002C175.361 196.102 175.403 196.197 175.464 196.279C175.525 196.361 175.603 196.429 175.694 196.478C175.784 196.527 175.884 196.555 175.986 196.56L177.377 196.658C177.532 196.664 177.685 196.619 177.812 196.529C177.938 196.44 178.031 196.31 178.076 196.161L179.053 192.581L182.809 192.844L181.879 196.155C181.855 196.25 181.852 196.35 181.869 196.447C181.887 196.543 181.925 196.635 181.981 196.717C182.036 196.798 182.109 196.866 182.193 196.917C182.277 196.969 182.371 197.001 182.469 197.014L183.86 197.111C184.02 197.126 184.181 197.085 184.315 196.995C184.448 196.905 184.547 196.771 184.593 196.617L185.577 193.037L189.751 193.329C189.913 193.344 190.076 193.301 190.21 193.208C190.344 193.115 190.441 192.979 190.485 192.821L190.861 191.45C190.889 191.349 190.894 191.243 190.875 191.14C190.857 191.037 190.816 190.94 190.754 190.855C190.693 190.77 190.614 190.7 190.522 190.65C190.43 190.6 190.328 190.572 190.223 190.566L186.342 190.295L187.865 184.81L192.038 185.102C192.201 185.116 192.363 185.073 192.498 184.981C192.632 184.888 192.729 184.751 192.773 184.594L193.155 183.223C193.184 183.119 193.189 183.011 193.169 182.906C193.149 182.801 193.106 182.701 193.041 182.616C192.977 182.53 192.894 182.461 192.798 182.412C192.703 182.364 192.597 182.338 192.49 182.337ZM183.56 190.1L179.804 189.838L181.334 184.353L185.09 184.616L183.56 190.1Z" fill="#0068A0"/><rect x="206.604" y="182.953" width="57.5047" height="13.8804" rx="1.98292" transform="rotate(4 206.604 182.953)" fill="#86D0F9"/><g filter="url(#filterhashtagpro1_ddd_506_22510)"><rect x="169.506" y="31.2762" width="158.634" height="180.446" rx="3.96584" transform="rotate(4 169.506 31.2762)" fill="white"/></g><g clip-path="url(#cliphashtagpro3_506_22510)"><rect x="169.506" y="31.2762" width="158.634" height="158.634" transform="rotate(4 169.506 31.2762)" fill="#0096CC"/><circle cx="183.84" cy="95.0859" r="92.2057" fill="#0068A0"/></g><path d="M192.49 182.337L188.63 182.067L189.561 178.756C189.585 178.659 189.587 178.557 189.568 178.458C189.549 178.359 189.509 178.266 189.45 178.184C189.391 178.102 189.315 178.034 189.228 177.984C189.14 177.934 189.043 177.904 188.943 177.896L187.552 177.798C187.391 177.783 187.23 177.824 187.097 177.914C186.963 178.004 186.864 178.138 186.818 178.292L185.855 181.873L182.099 181.611L183.029 178.3C183.053 178.204 183.056 178.105 183.038 178.008C183.021 177.911 182.983 177.819 182.927 177.738C182.871 177.657 182.799 177.588 182.715 177.537C182.631 177.486 182.537 177.453 182.439 177.441L181.048 177.343C180.888 177.328 180.727 177.369 180.593 177.459C180.459 177.549 180.361 177.683 180.315 177.837L179.365 181.42L175.192 181.128C175.029 181.113 174.867 181.156 174.733 181.249C174.598 181.341 174.501 181.478 174.457 181.636L174.075 183.007C174.047 183.107 174.042 183.213 174.06 183.316C174.079 183.419 174.12 183.517 174.181 183.602C174.242 183.686 174.322 183.756 174.414 183.806C174.506 183.856 174.608 183.885 174.712 183.89L178.572 184.16L177.043 189.645L172.869 189.353C172.707 189.338 172.544 189.381 172.41 189.474C172.276 189.567 172.179 189.703 172.135 189.861L171.752 191.232C171.724 191.333 171.719 191.438 171.738 191.541C171.756 191.644 171.797 191.742 171.859 191.827C171.92 191.911 171.999 191.981 172.091 192.031C172.183 192.081 172.285 192.11 172.39 192.115L176.278 192.387L175.347 195.698C175.323 195.797 175.321 195.901 175.341 196.002C175.361 196.102 175.403 196.197 175.464 196.279C175.525 196.361 175.603 196.429 175.694 196.478C175.784 196.527 175.884 196.555 175.986 196.56L177.377 196.658C177.532 196.664 177.685 196.619 177.812 196.529C177.938 196.44 178.031 196.31 178.076 196.161L179.053 192.581L182.809 192.844L181.879 196.155C181.855 196.25 181.852 196.35 181.869 196.447C181.887 196.543 181.925 196.635 181.981 196.717C182.036 196.798 182.109 196.866 182.193 196.917C182.277 196.969 182.371 197.001 182.469 197.014L183.86 197.111C184.02 197.126 184.181 197.085 184.315 196.995C184.448 196.905 184.547 196.771 184.593 196.617L185.577 193.037L189.751 193.329C189.913 193.344 190.076 193.301 190.21 193.208C190.344 193.115 190.441 192.979 190.485 192.821L190.861 191.45C190.889 191.349 190.894 191.243 190.875 191.14C190.857 191.037 190.816 190.94 190.754 190.855C190.693 190.77 190.614 190.7 190.522 190.65C190.43 190.6 190.328 190.572 190.223 190.566L186.342 190.295L187.865 184.81L192.038 185.102C192.201 185.116 192.363 185.073 192.498 184.981C192.632 184.888 192.729 184.751 192.773 184.594L193.155 183.223C193.184 183.119 193.189 183.011 193.169 182.906C193.149 182.801 193.106 182.701 193.041 182.616C192.977 182.53 192.894 182.461 192.798 182.412C192.703 182.364 192.597 182.338 192.49 182.337ZM183.56 190.1L179.804 189.838L181.334 184.353L185.09 184.616L183.56 190.1Z" fill="#0068A0"/><rect x="206.604" y="182.953" width="57.5047" height="13.8804" rx="1.98292" transform="rotate(4 206.604 182.953)" fill="#86D0F9"/><g filter="url(#filterhashtagpro2_ddd_506_22510)"><rect x="61.8896" y="48.4873" width="158.634" height="180.446" rx="3.96584" fill="white"/></g><g clip-path="url(#cliphashtagpro4_506_22510)"><rect x="61.8896" y="48.4873" width="158.634" height="158.634" fill="#F6966B"/><path d="M93.1149 104.67L218.7 230.255H-10.8174L93.1149 104.67Z" fill="#F9BBA0"/></g><path d="M95.3546 197.577H91.4848L92.1821 194.209C92.1992 194.11 92.1946 194.008 92.1687 193.911C92.1427 193.814 92.096 193.723 92.0317 193.646C91.9675 193.568 91.8872 193.506 91.7964 193.462C91.7056 193.419 91.6064 193.395 91.5057 193.394H90.1112C89.95 193.39 89.7925 193.442 89.6653 193.541C89.5382 193.64 89.4494 193.78 89.414 193.937L88.7028 197.577H84.9376L85.6348 194.209C85.6517 194.112 85.6479 194.013 85.6236 193.918C85.5992 193.822 85.555 193.733 85.4937 193.656C85.4325 193.579 85.3556 193.516 85.2682 193.47C85.1807 193.425 85.0847 193.399 84.9864 193.394H83.5919C83.4307 193.39 83.2731 193.442 83.146 193.541C83.0189 193.64 82.93 193.78 82.8946 193.937L82.1974 197.577H78.0138C77.8507 197.574 77.6916 197.628 77.5642 197.73C77.4368 197.832 77.3492 197.975 77.3166 198.135L77.0307 199.529C77.0098 199.632 77.0121 199.738 77.0377 199.839C77.0632 199.941 77.1112 200.035 77.1782 200.115C77.2451 200.196 77.3293 200.26 77.4245 200.303C77.5197 200.347 77.6234 200.368 77.7279 200.366H81.5977L80.4542 205.944H76.2707C76.1076 205.941 75.9485 205.995 75.8211 206.097C75.6937 206.199 75.606 206.342 75.5734 206.502L75.2876 207.897C75.2666 207.999 75.269 208.105 75.2945 208.206C75.3201 208.308 75.3681 208.402 75.435 208.483C75.502 208.563 75.5862 208.627 75.6814 208.671C75.7765 208.714 75.8802 208.735 75.9848 208.733H79.8825L79.1852 212.101C79.1678 212.202 79.1728 212.306 79.2 212.404C79.2271 212.503 79.2757 212.595 79.3423 212.673C79.4088 212.751 79.4918 212.813 79.5851 212.855C79.6785 212.897 79.78 212.918 79.8825 212.917H81.277C81.4322 212.913 81.5816 212.857 81.7016 212.759C81.8216 212.66 81.9053 212.524 81.9394 212.373L82.6645 208.733H86.4297L85.7324 212.101C85.7156 212.198 85.7194 212.297 85.7437 212.393C85.768 212.488 85.8123 212.577 85.8735 212.654C85.9348 212.732 86.0117 212.795 86.0991 212.84C86.1866 212.885 86.2826 212.911 86.3809 212.917H87.7754C87.9366 212.921 88.0941 212.869 88.2213 212.77C88.3484 212.67 88.4372 212.53 88.4727 212.373L89.2048 208.733H93.3883C93.5514 208.737 93.7105 208.683 93.8379 208.581C93.9653 208.479 94.053 208.335 94.0856 208.175L94.3645 206.781C94.3854 206.678 94.383 206.573 94.3575 206.471C94.332 206.37 94.2839 206.275 94.217 206.195C94.15 206.115 94.0658 206.05 93.9707 206.007C93.8755 205.964 93.7718 205.942 93.6672 205.944H89.7765L90.9131 200.366H95.0966C95.2597 200.369 95.4188 200.315 95.5462 200.214C95.6736 200.112 95.7612 199.968 95.7938 199.808L96.0797 198.414C96.1013 198.309 96.0985 198.201 96.0714 198.097C96.0444 197.993 95.9939 197.897 95.9237 197.816C95.8536 197.736 95.7658 197.672 95.6671 197.63C95.5684 197.589 95.4615 197.571 95.3546 197.577ZM86.9875 205.944H83.2223L84.3658 200.366H88.131L86.9875 205.944Z" fill="#FE544F"/><rect x="109.479" y="197.206" width="57.5047" height="13.8804" rx="1.98292" fill="#FCE1D5"/><g filter="url(#filterhashtagpro3_ddd_506_22510)"><rect x="61.8896" y="48.4873" width="158.634" height="180.446" rx="3.96584" fill="white"/></g><g clip-path="url(#cliphashtagpro5_506_22510)"><rect x="61.8896" y="48.4873" width="158.634" height="158.634" fill="#F6966B"/><path d="M93.1149 104.67L218.7 230.255H-10.8174L93.1149 104.67Z" fill="#F9BBA0"/></g><path d="M95.3546 197.577H91.4848L92.1821 194.209C92.1992 194.11 92.1946 194.008 92.1687 193.911C92.1427 193.814 92.096 193.723 92.0317 193.646C91.9675 193.568 91.8872 193.506 91.7964 193.462C91.7056 193.419 91.6064 193.395 91.5057 193.394H90.1112C89.95 193.39 89.7925 193.442 89.6653 193.541C89.5382 193.64 89.4494 193.78 89.414 193.937L88.7028 197.577H84.9376L85.6348 194.209C85.6517 194.112 85.6479 194.013 85.6236 193.918C85.5992 193.822 85.555 193.733 85.4937 193.656C85.4325 193.579 85.3556 193.516 85.2682 193.47C85.1807 193.425 85.0847 193.399 84.9864 193.394H83.5919C83.4307 193.39 83.2731 193.442 83.146 193.541C83.0189 193.64 82.93 193.78 82.8946 193.937L82.1974 197.577H78.0138C77.8507 197.574 77.6916 197.628 77.5642 197.73C77.4368 197.832 77.3492 197.975 77.3166 198.135L77.0307 199.529C77.0098 199.632 77.0121 199.738 77.0377 199.839C77.0632 199.941 77.1112 200.035 77.1782 200.115C77.2451 200.196 77.3293 200.26 77.4245 200.303C77.5197 200.347 77.6234 200.368 77.7279 200.366H81.5977L80.4542 205.944H76.2707C76.1076 205.941 75.9485 205.995 75.8211 206.097C75.6937 206.199 75.606 206.342 75.5734 206.502L75.2876 207.897C75.2666 207.999 75.269 208.105 75.2945 208.206C75.3201 208.308 75.3681 208.402 75.435 208.483C75.502 208.563 75.5862 208.627 75.6814 208.671C75.7765 208.714 75.8802 208.735 75.9848 208.733H79.8825L79.1852 212.101C79.1678 212.202 79.1728 212.306 79.2 212.404C79.2271 212.503 79.2757 212.595 79.3423 212.673C79.4088 212.751 79.4918 212.813 79.5851 212.855C79.6785 212.897 79.78 212.918 79.8825 212.917H81.277C81.4322 212.913 81.5816 212.857 81.7016 212.759C81.8216 212.66 81.9053 212.524 81.9394 212.373L82.6645 208.733H86.4297L85.7324 212.101C85.7156 212.198 85.7194 212.297 85.7437 212.393C85.768 212.488 85.8123 212.577 85.8735 212.654C85.9348 212.732 86.0117 212.795 86.0991 212.84C86.1866 212.885 86.2826 212.911 86.3809 212.917H87.7754C87.9366 212.921 88.0941 212.869 88.2213 212.77C88.3484 212.67 88.4372 212.53 88.4727 212.373L89.2048 208.733H93.3883C93.5514 208.737 93.7105 208.683 93.8379 208.581C93.9653 208.479 94.053 208.335 94.0856 208.175L94.3645 206.781C94.3854 206.678 94.383 206.573 94.3575 206.471C94.332 206.37 94.2839 206.275 94.217 206.195C94.15 206.115 94.0658 206.05 93.9707 206.007C93.8755 205.964 93.7718 205.942 93.6672 205.944H89.7765L90.9131 200.366H95.0966C95.2597 200.369 95.4188 200.315 95.5462 200.214C95.6736 200.112 95.7612 199.968 95.7938 199.808L96.0797 198.414C96.1013 198.309 96.0985 198.201 96.0714 198.097C96.0444 197.993 95.9939 197.897 95.9237 197.816C95.8536 197.736 95.7658 197.672 95.6671 197.63C95.5684 197.589 95.4615 197.571 95.3546 197.577ZM86.9875 205.944H83.2223L84.3658 200.366H88.131L86.9875 205.944Z" fill="#FE544F"/><rect x="109.479" y="197.206" width="57.5047" height="13.8804" rx="1.98292" fill="#FCE1D5"/></g></g><defs><filter id="filterhashtagpro0_ddd_506_22510" x="131.408" y="17.6626" width="221.857" height="242.094" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="11.8975"/><feGaussianBlur stdDeviation="12.889"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.98292"/><feGaussianBlur stdDeviation="1.98292"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_506_22510" result="effect2_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="5.94876"/><feGaussianBlur stdDeviation="5.94876"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_506_22510" result="effect3_dropShadow_506_22510"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_506_22510" result="shape"/></filter><filter id="filterhashtagpro1_ddd_506_22510" x="131.408" y="17.6626" width="221.857" height="242.094" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="11.8975"/><feGaussianBlur stdDeviation="12.889"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.98292"/><feGaussianBlur stdDeviation="1.98292"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_506_22510" result="effect2_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="5.94876"/><feGaussianBlur stdDeviation="5.94876"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_506_22510" result="effect3_dropShadow_506_22510"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_506_22510" result="shape"/></filter><filter id="filterhashtagpro2_ddd_506_22510" x="36.1117" y="34.6069" width="210.19" height="232.002" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="11.8975"/><feGaussianBlur stdDeviation="12.889"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.98292"/><feGaussianBlur stdDeviation="1.98292"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_506_22510" result="effect2_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="5.94876"/><feGaussianBlur stdDeviation="5.94876"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_506_22510" result="effect3_dropShadow_506_22510"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_506_22510" result="shape"/></filter><filter id="filterhashtagpro3_ddd_506_22510" x="36.1117" y="34.6069" width="210.19" height="232.002" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="11.8975"/><feGaussianBlur stdDeviation="12.889"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.98292"/><feGaussianBlur stdDeviation="1.98292"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_506_22510" result="effect2_dropShadow_506_22510"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="5.94876"/><feGaussianBlur stdDeviation="5.94876"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_506_22510" result="effect3_dropShadow_506_22510"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_506_22510" result="shape"/></filter><clipPath id="cliphashtagpro0_506_22510"><rect width="396" height="264" fill="white"/></clipPath><clipPath id="cliphashtagpro1_506_22510"><rect width="530" height="250.308" fill="white" transform="translate(-67 6.84601)"/></clipPath><clipPath id="cliphashtagpro2_506_22510"><path d="M169.275 34.573C169.403 32.7522 170.982 31.3794 172.803 31.5067L324.456 42.1114C326.277 42.2387 327.65 43.818 327.522 45.6387L318.624 172.896L160.377 161.83L169.275 34.573Z" fill="white"/></clipPath><clipPath id="cliphashtagpro3_506_22510"><path d="M169.275 34.573C169.403 32.7522 170.982 31.3794 172.803 31.5067L324.456 42.1114C326.277 42.2387 327.65 43.818 327.522 45.6387L318.624 172.896L160.377 161.83L169.275 34.573Z" fill="white"/></clipPath><clipPath id="cliphashtagpro4_506_22510"><path d="M61.8896 51.7922C61.8896 49.9669 63.3693 48.4873 65.1945 48.4873H217.218C219.044 48.4873 220.523 49.9669 220.523 51.7922V179.36H61.8896V51.7922Z" fill="white"/></clipPath><clipPath id="cliphashtagpro5_506_22510"><path d="M61.8896 51.7922C61.8896 49.9669 63.3693 48.4873 65.1945 48.4873H217.218C219.044 48.4873 220.523 49.9669 220.523 51.7922V179.36H61.8896V51.7922Z" fill="white"/></clipPath></defs></svg>',
							'bullets'         => array(
								'heading' => __( 'And get much more!', 'instagram-feed' ),
								'content' => array(
									__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
									__( 'Powerful visual moderation', 'instagram-feed' ),
									__( 'Comments and Likes', 'instagram-feed' ),
									__( 'Highlight specific posts', 'instagram-feed' ),
									__( 'Multiple layout options', 'instagram-feed' ),
									__( 'Popup photo/video lightbox', 'instagram-feed' ),
									__( 'Instagram Stories', 'instagram-feed' ),
									__( 'Shoppable feeds', 'instagram-feed' ),
									__( 'Pro support', 'instagram-feed' ),
									__( 'Post captions', 'instagram-feed' ),
									__( 'Combine multiple feed types', 'instagram-feed' ),
									__( '30 day money back guarantee', 'instagram-feed' ),
								)
							),
							'buyUrl'          => sprintf( 'https://smashballoon.com/instagram-feed/demo/hashtag?utm_campaign=instagram-free&utm_source=feed-type&utm_medium=hashtag' )
						),
						'tagged'           => array(
							'heading'         => __( 'Upgrade to Pro to get Tagged Posts Feed', 'instagram-feed' ),
							'description'     => __( 'Display posts that you\'ve been tagged in by other users allowing you to increase your audience\'s engagement with your Instagram account.', 'instagram-feed' ),
							'img'             => '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#cliptagged0_506_22672)"><g clip-path="url(#cliptagged1_506_22672)"><g filter="url(#filtertagged0_ddd_506_22672)"><g clip-path="url(#cliptagged2_506_22672)"><rect x="139.63" y="64.4107" width="164.72" height="166.806" rx="4.17014" transform="rotate(2 139.63 64.4107)" fill="white"/><g clip-path="url(#cliptagged3_506_22672)"><path d="M139.131 52.643L305.835 58.4644L300.45 212.666L133.746 206.844L139.131 52.643Z" fill="#59AB46"/><path d="M170.246 102.41L297.611 238.993L56.4177 230.57L170.246 102.41Z" fill="#76C064"/></g><path fill-rule="evenodd" clip-rule="evenodd" d="M157.064 194.828C159.201 194.48 161.393 194.773 163.363 195.671C165.333 196.569 166.993 198.031 168.132 199.872C169.271 201.713 169.839 203.85 169.763 206.014L169.707 207.635C169.671 208.674 169.223 209.656 168.463 210.364C167.703 211.073 166.692 211.451 165.653 211.415C164.615 211.379 163.633 210.931 162.924 210.171C162.792 210.029 162.671 209.879 162.563 209.721C161.492 210.7 160.081 211.22 158.63 211.17C157.162 211.118 155.773 210.486 154.771 209.411C153.769 208.336 153.235 206.907 153.286 205.439C153.337 203.97 153.97 202.582 155.045 201.58C156.119 200.578 157.548 200.043 159.017 200.095C160.486 200.146 161.874 200.779 162.876 201.853C163.878 202.928 164.412 204.357 164.361 205.826L164.304 207.446C164.291 207.84 164.434 208.224 164.703 208.512C164.972 208.8 165.344 208.97 165.738 208.984C166.132 208.998 166.516 208.854 166.804 208.585C167.092 208.316 167.262 207.944 167.276 207.55L167.332 205.929C167.391 204.246 166.95 202.584 166.063 201.152C165.177 199.72 163.887 198.583 162.354 197.885C160.822 197.186 159.117 196.958 157.455 197.229C155.793 197.499 154.249 198.257 153.017 199.405C151.785 200.554 150.922 202.042 150.536 203.681C150.15 205.32 150.259 207.037 150.849 208.614C151.438 210.191 152.483 211.558 153.849 212.542C155.216 213.526 156.844 214.082 158.526 214.141L158.528 214.141C160.021 214.195 161.502 213.853 162.819 213.15C163.412 212.834 164.148 213.058 164.465 213.651C164.781 214.243 164.557 214.98 163.964 215.296C162.267 216.202 160.362 216.641 158.44 216.572L158.484 215.357L158.442 216.572C158.441 216.572 158.441 216.572 158.44 216.572C156.277 216.496 154.185 215.781 152.428 214.516C150.671 213.251 149.328 211.494 148.57 209.466C147.812 207.438 147.672 205.231 148.168 203.123C148.665 201.016 149.775 199.103 151.358 197.626C152.941 196.15 154.927 195.176 157.064 194.828ZM161.93 205.741C161.959 204.917 161.659 204.115 161.097 203.512C160.535 202.909 159.756 202.555 158.932 202.526C158.108 202.497 157.307 202.797 156.704 203.359C156.101 203.921 155.746 204.7 155.717 205.524C155.688 206.348 155.988 207.149 156.55 207.752C157.113 208.355 157.891 208.71 158.715 208.739C159.539 208.767 160.34 208.468 160.943 207.905C161.546 207.343 161.901 206.564 161.93 205.741Z" fill="#59AB46"/><rect x="184.048" y="200.256" width="60.467" height="14.5955" rx="2.08507" transform="rotate(2 184.048 200.256)" fill="#B6DDAD"/></g></g><g filter="url(#filtertagged1_ddd_506_22672)"><g clip-path="url(#cliptagged4_506_22672)"><rect x="73.3418" y="28.4567" width="164.72" height="166.806" rx="4.17014" transform="rotate(-2 73.3418 28.4567)" fill="white"/><g clip-path="url(#cliptagged5_506_22672)"><path d="M72.0225 16.7529L238.726 10.9315L244.111 165.133L77.4073 170.954L72.0225 16.7529Z" fill="#43A6DB"/><circle cx="268.485" cy="149.951" r="96.9557" transform="rotate(-2 268.485 149.951)" fill="#0068A0"/></g><path fill-rule="evenodd" clip-rule="evenodd" d="M99.8296 157.341C101.937 156.845 104.144 156.985 106.172 157.743C108.2 158.501 109.958 159.844 111.223 161.601C112.487 163.358 113.203 165.451 113.279 167.615L113.335 169.235C113.371 170.274 112.994 171.285 112.285 172.045C111.576 172.805 110.594 173.253 109.555 173.289C108.516 173.325 107.506 172.947 106.745 172.238C106.604 172.106 106.473 171.965 106.354 171.815C105.354 172.866 103.983 173.483 102.532 173.534C101.063 173.585 99.6345 173.051 98.5597 172.049C97.485 171.047 96.8524 169.659 96.8011 168.19C96.7498 166.721 97.284 165.292 98.2862 164.218C99.2885 163.143 100.677 162.51 102.145 162.459C103.614 162.408 105.043 162.942 106.117 163.944C107.192 164.946 107.825 166.335 107.876 167.803L107.933 169.424C107.946 169.818 108.116 170.19 108.405 170.459C108.693 170.728 109.076 170.871 109.47 170.858C109.864 170.844 110.237 170.674 110.506 170.386C110.774 170.098 110.918 169.714 110.904 169.32L110.847 167.699C110.789 166.017 110.232 164.389 109.248 163.022C108.265 161.656 106.898 160.611 105.32 160.022C103.743 159.432 102.026 159.323 100.387 159.709C98.7481 160.095 97.2602 160.958 96.1118 162.19C94.9634 163.422 94.2059 164.966 93.9353 166.628C93.6646 168.29 93.8928 169.995 94.5911 171.527C95.2895 173.06 96.4265 174.35 97.8584 175.236C99.2903 176.123 100.953 176.564 102.636 176.505L102.637 176.505C104.13 176.455 105.584 176.011 106.849 175.218C107.418 174.861 108.169 175.033 108.525 175.602C108.882 176.171 108.71 176.922 108.141 177.279C106.511 178.3 104.641 178.871 102.719 178.937L102.678 177.721L102.721 178.937C102.72 178.937 102.72 178.937 102.719 178.937C100.556 179.012 98.419 178.444 96.5783 177.305C94.7373 176.166 93.2754 174.506 92.3776 172.536C91.4798 170.566 91.1863 168.374 91.5343 166.237C91.8823 164.1 92.8562 162.114 94.3327 160.531C95.8093 158.948 97.7222 157.838 99.8296 157.341ZM105.445 167.888C105.416 167.064 105.061 166.286 104.458 165.723C103.856 165.161 103.054 164.861 102.23 164.89C101.406 164.919 100.628 165.274 100.065 165.877C99.5031 166.48 99.2034 167.281 99.2322 168.105C99.261 168.929 99.6158 169.708 100.219 170.27C100.822 170.832 101.623 171.132 102.447 171.103C103.271 171.074 104.05 170.719 104.612 170.116C105.174 169.514 105.474 168.712 105.445 167.888Z" fill="#0068A0"/><rect x="127.129" y="160.874" width="60.467" height="14.5955" rx="2.08507" transform="rotate(-2 127.129 160.874)" fill="#86D0F9"/></g></g></g></g><defs><filter id="filtertagged0_ddd_506_22672" x="106.703" y="49.8152" width="224.653" height="226.664" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="12.5104"/><feGaussianBlur stdDeviation="13.5529"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_506_22672"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="2.08507"/><feGaussianBlur stdDeviation="2.08507"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_506_22672" result="effect2_dropShadow_506_22672"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="6.25521"/><feGaussianBlur stdDeviation="6.25521"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_506_22672" result="effect3_dropShadow_506_22672"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_506_22672" result="shape"/></filter><filter id="filtertagged1_ddd_506_22672" x="46.2359" y="8.11258" width="224.653" height="226.664" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="12.5104"/><feGaussianBlur stdDeviation="13.5529"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_506_22672"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="2.08507"/><feGaussianBlur stdDeviation="2.08507"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_506_22672" result="effect2_dropShadow_506_22672"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="6.25521"/><feGaussianBlur stdDeviation="6.25521"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_506_22672" result="effect3_dropShadow_506_22672"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_506_22672" result="shape"/></filter><clipPath id="cliptagged0_506_22672"><rect width="396" height="264" fill="white"/></clipPath><clipPath id="cliptagged1_506_22672"><rect width="530" height="250.308" fill="white" transform="translate(-67 6.84601)"/></clipPath><clipPath id="cliptagged2_506_22672"><rect x="139.63" y="64.4107" width="164.72" height="166.806" rx="4.17014" transform="rotate(2 139.63 64.4107)" fill="white"/></clipPath><clipPath id="cliptagged3_506_22672"><path d="M139.373 45.6973C139.44 43.7792 141.049 42.2786 142.967 42.3456L302.725 47.9245C304.643 47.9914 306.144 49.6006 306.077 51.5187L301.395 185.576L134.691 179.755L139.373 45.6973Z" fill="white"/></clipPath><clipPath id="cliptagged4_506_22672"><rect x="73.3418" y="28.4567" width="164.72" height="166.806" rx="4.17014" transform="rotate(-2 73.3418 28.4567)" fill="white"/></clipPath><clipPath id="cliptagged5_506_22672"><path d="M71.7785 9.80722C71.7115 7.88913 73.2121 6.27993 75.1302 6.21295L234.888 0.634078C236.806 0.567097 238.415 2.06771 238.482 3.9858L243.164 138.044L76.4599 143.865L71.7785 9.80722Z" fill="white"/></clipPath></defs></svg>',
							'bullets'         => array(
								'heading' => __( 'And get much more!', 'instagram-feed' ),
								'content' => array(
									__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
									__( 'Powerful visual moderation', 'instagram-feed' ),
									__( 'Comments and Likes', 'instagram-feed' ),
									__( 'Highlight specific posts', 'instagram-feed' ),
									__( 'Multiple layout options', 'instagram-feed' ),
									__( 'Popup photo/video lightbox', 'instagram-feed' ),
									__( 'Instagram Stories', 'instagram-feed' ),
									__( 'Shoppable feeds', 'instagram-feed' ),
									__( 'Pro support', 'instagram-feed' ),
									__( 'Post captions', 'instagram-feed' ),
									__( 'Combine multiple feed types', 'instagram-feed' ),
									__( '30 day money back guarantee', 'instagram-feed' ),
								)
							),
							'buyUrl'          => sprintf( 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=feed-type&utm_medium=tagged' )
						),
						'socialwall'       => array(
							// Combine all your social media channels into one Social Wall
							'heading' 		  => '<span class="sb-social-wall">' . __( 'Combine all your social media channels into one', 'instagram-feed' ) .' <span>'. __( 'Social Wall', 'instagram-feed' ).'</span></span>',
							'description' 	  => '<span class="sb-social-wall">' . __( 'A dash of Instagram, a sprinkle of Facebook, a spoonful of Twitter, and a dollop of YouTube, all in the same feed.', 'instagram-feed' ) . '</span>',
							'img'             => '<svg width="397" height="264" viewBox="0 0 397 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0)"><g filter="url(#filter0_ddd)"><rect x="18.957" y="63" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip1)"><path d="M18.957 63H132.769V176.812H18.957V63Z" fill="#0068A0"/><rect x="56.957" y="106" width="105" height="105" rx="9" fill="#005B8C"/></g><path d="M36.0293 165.701C31.4649 165.701 27.7305 169.427 27.7305 174.017C27.7305 178.166 30.7678 181.61 34.7347 182.232V176.423H32.6268V174.017H34.7347V172.183C34.7347 170.1 35.9712 168.954 37.8716 168.954C38.7762 168.954 39.7222 169.112 39.7222 169.112V171.162H38.6766C37.6475 171.162 37.3239 171.801 37.3239 172.456V174.017H39.6309L39.2575 176.423H37.3239V182.232C39.2794 181.924 41.0602 180.926 42.3446 179.419C43.629 177.913 44.3325 175.996 44.3281 174.017C44.3281 169.427 40.5936 165.701 36.0293 165.701Z" fill="#006BFA"/><rect x="53.1016" y="169.699" width="41.2569" height="9.95855" rx="1.42265" fill="#D0D1D7"/><g filter="url(#filter1_ddd)"><rect x="18.957" y="201" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip2)"><path d="M18.957 201H132.769V314.812H18.957V201Z" fill="#EC352F"/><circle cx="23.957" cy="243" r="59" fill="#FE544F"/></g><g filter="url(#filter2_ddd)"><rect x="139.957" y="23" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip3)"><path d="M139.957 23H253.769V136.812H139.957V23Z" fill="#8C8F9A"/><circle cx="127.457" cy="142.5" r="78.5" fill="#D0D1D7"/></g><path d="M157.026 129.493C154.537 129.493 152.553 131.516 152.553 133.967C152.553 136.456 154.537 138.44 157.026 138.44C159.477 138.44 161.5 136.456 161.5 133.967C161.5 131.516 159.477 129.493 157.026 129.493ZM157.026 136.884C155.431 136.884 154.109 135.601 154.109 133.967C154.109 132.372 155.392 131.088 157.026 131.088C158.621 131.088 159.905 132.372 159.905 133.967C159.905 135.601 158.621 136.884 157.026 136.884ZM162.706 129.338C162.706 128.754 162.239 128.287 161.655 128.287C161.072 128.287 160.605 128.754 160.605 129.338C160.605 129.921 161.072 130.388 161.655 130.388C162.239 130.388 162.706 129.921 162.706 129.338ZM165.662 130.388C165.584 128.987 165.273 127.743 164.262 126.731C163.25 125.72 162.005 125.409 160.605 125.331C159.166 125.253 154.848 125.253 153.408 125.331C152.008 125.409 150.802 125.72 149.752 126.731C148.74 127.743 148.429 128.987 148.351 130.388C148.274 131.827 148.274 136.145 148.351 137.585C148.429 138.985 148.74 140.191 149.752 141.241C150.802 142.253 152.008 142.564 153.408 142.642C154.848 142.719 159.166 142.719 160.605 142.642C162.005 142.564 163.25 142.253 164.262 141.241C165.273 140.191 165.584 138.985 165.662 137.585C165.74 136.145 165.74 131.827 165.662 130.388ZM163.795 139.102C163.523 139.88 162.9 140.463 162.161 140.774C160.994 141.241 158.271 141.124 157.026 141.124C155.742 141.124 153.019 141.241 151.891 140.774C151.113 140.463 150.53 139.88 150.219 139.102C149.752 137.974 149.868 135.25 149.868 133.967C149.868 132.722 149.752 129.999 150.219 128.832C150.53 128.093 151.113 127.509 151.891 127.198C153.019 126.731 155.742 126.848 157.026 126.848C158.271 126.848 160.994 126.731 162.161 127.198C162.9 127.47 163.484 128.093 163.795 128.832C164.262 129.999 164.145 132.722 164.145 133.967C164.145 135.25 164.262 137.974 163.795 139.102Z" fill="url(#paint0_linear)"/><rect x="174.102" y="129.699" width="41.2569" height="9.95855" rx="1.42265" fill="#D0D1D7"/><g filter="url(#filter3_ddd)"><rect x="139.957" y="161" width="114" height="109" rx="2.8453" fill="white"/></g><rect x="148.957" y="194" width="91" height="8" rx="1.42265" fill="#D0D1D7"/><rect x="148.957" y="208" width="51" height="8" rx="1.42265" fill="#D0D1D7"/><path d="M164.366 172.062C163.788 172.324 163.166 172.497 162.521 172.579C163.181 172.182 163.691 171.552 163.931 170.794C163.308 171.169 162.618 171.432 161.891 171.582C161.298 170.937 160.466 170.562 159.521 170.562C157.758 170.562 156.318 172.002 156.318 173.779C156.318 174.034 156.348 174.282 156.401 174.514C153.731 174.379 151.353 173.097 149.771 171.154C149.493 171.627 149.336 172.182 149.336 172.767C149.336 173.884 149.898 174.874 150.768 175.437C150.236 175.437 149.741 175.287 149.306 175.062V175.084C149.306 176.644 150.416 177.949 151.886 178.242C151.414 178.371 150.918 178.389 150.438 178.294C150.642 178.934 151.041 179.493 151.579 179.894C152.117 180.295 152.767 180.517 153.438 180.529C152.301 181.43 150.891 181.916 149.441 181.909C149.186 181.909 148.931 181.894 148.676 181.864C150.101 182.779 151.796 183.312 153.611 183.312C159.521 183.312 162.768 178.407 162.768 174.154C162.768 174.012 162.768 173.877 162.761 173.734C163.391 173.284 163.931 172.714 164.366 172.062Z" fill="#1B90EF"/><g filter="url(#filter4_ddd)"><rect x="260.957" y="63" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip4)"><rect x="260.957" y="63" width="113.812" height="113.812" fill="#D72C2C"/><path d="M283.359 103.308L373.461 193.41H208.793L283.359 103.308Z" fill="#DF5757"/></g><path d="M276.37 176.456L280.677 173.967L276.37 171.477V176.456ZM285.963 169.958C286.071 170.348 286.145 170.871 286.195 171.535C286.253 172.199 286.278 172.772 286.278 173.27L286.328 173.967C286.328 175.784 286.195 177.12 285.963 177.975C285.755 178.722 285.274 179.203 284.527 179.411C284.137 179.519 283.423 179.593 282.328 179.643C281.249 179.701 280.262 179.726 279.349 179.726L278.029 179.776C274.552 179.776 272.386 179.643 271.531 179.411C270.784 179.203 270.303 178.722 270.096 177.975C269.988 177.585 269.913 177.062 269.863 176.398C269.805 175.734 269.78 175.162 269.78 174.664L269.73 173.967C269.73 172.149 269.863 170.813 270.096 169.958C270.303 169.212 270.784 168.73 271.531 168.523C271.921 168.415 272.635 168.34 273.73 168.29C274.809 168.232 275.797 168.207 276.71 168.207L278.029 168.158C281.506 168.158 283.672 168.29 284.527 168.523C285.274 168.73 285.755 169.212 285.963 169.958Z" fill="#EB2121"/><rect x="295.102" y="169.699" width="41.2569" height="9.95855" rx="1.42265" fill="#D0D1D7"/><g filter="url(#filter5_ddd)"><rect x="260.957" y="201" width="113.812" height="129.461" rx="2.8453" fill="white"/></g><g clip-path="url(#clip5)"><rect x="260.957" y="201" width="113.812" height="113.812" fill="#59AB46"/><circle cx="374.457" cy="235.5" r="44.5" fill="#468737"/></g><g clip-path="url(#clip6)"><path d="M139.957 228H253.957V296C253.957 296.552 253.509 297 252.957 297H140.957C140.405 297 139.957 296.552 139.957 296V228Z" fill="#0068A0"/><circle cx="227.957" cy="245" r="34" fill="#004D77"/></g></g><defs><filter id="filter0_ddd" x="0.462572" y="53.0414" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter1_ddd" x="0.462572" y="191.041" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter2_ddd" x="121.463" y="13.0414" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter3_ddd" x="121.463" y="151.041" width="150.989" height="145.989" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter4_ddd" x="242.463" y="53.0414" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><filter id="filter5_ddd" x="242.463" y="191.041" width="150.801" height="166.45" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="8.5359"/><feGaussianBlur stdDeviation="9.24723"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.42265"/><feGaussianBlur stdDeviation="1.42265"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.26795"/><feGaussianBlur stdDeviation="4.26795"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/></filter><linearGradient id="paint0_linear" x1="154.502" y1="158.603" x2="191.208" y2="121.133" gradientUnits="userSpaceOnUse"><stop stop-color="white"/><stop offset="0.147864" stop-color="#F6640E"/><stop offset="0.443974" stop-color="#BA03A7"/><stop offset="0.733337" stop-color="#6A01B9"/><stop offset="1" stop-color="#6B01B9"/></linearGradient><clipPath id="clip0"><rect width="396" height="264" fill="white" transform="translate(0.957031)"/></clipPath><clipPath id="clip1"><path d="M18.957 65.3711C18.957 64.0616 20.0186 63 21.3281 63H130.398C131.708 63 132.769 64.0616 132.769 65.3711V156.895H18.957V65.3711Z" fill="white"/></clipPath><clipPath id="clip2"><path d="M18.957 203.371C18.957 202.062 20.0186 201 21.3281 201H130.398C131.708 201 132.769 202.062 132.769 203.371V294.895H18.957V203.371Z" fill="white"/></clipPath><clipPath id="clip3"><path d="M139.957 25.3711C139.957 24.0616 141.019 23 142.328 23H251.398C252.708 23 253.769 24.0616 253.769 25.3711V116.895H139.957V25.3711Z" fill="white"/></clipPath><clipPath id="clip4"><path d="M260.957 65.3711C260.957 64.0616 262.019 63 263.328 63H372.398C373.708 63 374.769 64.0616 374.769 65.3711V156.895H260.957V65.3711Z" fill="white"/></clipPath><clipPath id="clip5"><path d="M260.957 203.371C260.957 202.062 262.019 201 263.328 201H372.398C373.708 201 374.769 202.062 374.769 203.371V294.895H260.957V203.371Z" fill="white"/></clipPath><clipPath id="clip6"><path d="M139.957 228H253.957V296C253.957 296.552 253.509 297 252.957 297H140.957C140.405 297 139.957 296.552 139.957 296V228Z" fill="white"/></clipPath></defs></svg>',
							'demoUrl'         => 'https://smashballoon.com/social-wall/demo/?utm_campaign=instagram-free&utm_source=feed-type&utm_medium=social-wall&utm_content=learn-more',
							'buyUrl'          => sprintf( 'https://smashballoon.com/social-wall/demo/?license_key=%s&upgrade=true&utm_campaign=instagram-free&utm_source=feed-type&utm_medium=social-wall&utm_content=Try Demo', $license_key ),
							'bullets'         => array(
								'heading' => __( 'Upgrade to the All Access Bundle and get:', 'instagram-feed' ),
								'content' => array(
									__( 'Instagram Feed Pro', 'instagram-feed' ),
									__( 'Custom Twitter Feeds Pro', 'instagram-feed' ),
									__( 'YouTube Feeds Pro', 'instagram-feed' ),
									__( 'Custom Facebook Feed Pro', 'instagram-feed' ),
									__( 'All Pro Facebook Extensions', 'instagram-feed' ),
									__( 'Social Wall Pro', 'instagram-feed' ),
								)
							),
						),

						// Other Types
						'feedLayout'       => array(
							'heading'         => __( 'Upgrade to Pro to get Feed Layouts', 'instagram-feed' ),
							'description'     => __( 'Choose from one of our built-in layout options; grid, carousel, masonry, and highlight to allow you to showcase your content in any way you want.', 'instagram-feed' ),
							'img'             => '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g filter="url(#feedLayoutSettingsfilter0_d_541_17506)"><rect x="45" y="67.0205" width="166.67" height="154.67" rx="2.66672" transform="rotate(-3 45 67.0205)" fill="white"/><g clip-path="url(#clip0feedLayoutSettings_541_17506)"><rect width="97.3351" height="82.6682" transform="translate(54.8096 75.8527) rotate(-3)" fill="#FCE1D5"/><circle cx="50.9329" cy="167.516" r="66.0012" transform="rotate(-3 50.9329 167.516)" fill="#F9BBA0"/><circle cx="151.997" cy="121.496" r="56.001" transform="rotate(-3 151.997 121.496)" fill="#FE544F"/></g><rect width="42.6674" height="37.334" transform="translate(160 70.3398) rotate(-3)" fill="#E8E8EB"/><rect width="42.6674" height="37.334" transform="translate(162.372 115.612) rotate(-3)" fill="#DCDDE1"/><rect width="66.6679" height="44.0008" transform="translate(140.779 162.14) rotate(-3)" fill="#D0D1D7"/><rect width="73.3347" height="44.0008" transform="translate(59.5542 166.397) rotate(-3)" fill="#E8E8EB"/></g><g filter="url(#feedLayoutSettingsfilter1_d_541_17506)"><g clip-path="url(#clip1feedLayoutSettings_541_17506)"><rect x="175.256" y="27.1967" width="176.132" height="168.003" rx="2.27031" transform="rotate(5 175.256 27.1967)" fill="white"/><g clip-path="url(#clip2feedLayoutSettings_541_17506)"><rect width="112.002" height="124.647" transform="translate(204.782 52.4471) rotate(5)" fill="#B5E5FF"/><circle cx="199.572" cy="173.295" r="46.9609" transform="rotate(5 199.572 173.295)" fill="#43A6DB"/><circle cx="338.706" cy="133.728" r="64.1417" transform="rotate(5 338.706 133.728)" fill="#86D0F9"/></g><g clip-path="url(#clip3feedLayoutSettings_541_17506)"><rect width="112.002" height="124.647" transform="translate(322.654 62.7597) rotate(5)" fill="#B6DDAD"/><circle cx="317.444" cy="183.608" r="46.9609" transform="rotate(5 317.444 183.608)" fill="#96CE89"/></g><g clip-path="url(#clip4feedLayoutSettings_541_17506)"><rect width="112.002" height="124.647" transform="translate(86.0059 42.0556) rotate(5)" fill="#FCE1D5"/><circle cx="214.51" cy="169.366" r="64.1417" transform="rotate(5 214.51 169.366)" fill="#F9BBA0"/></g></g></g><defs><filter id="feedLayoutSettingsfilter0_d_541_17506" x="36.8814" y="54.8182" width="190.773" height="179.418" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="4.63923"/><feGaussianBlur stdDeviation="4.05932"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.16 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_541_17506"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_541_17506" result="shape"/></filter><filter id="feedLayoutSettingsfilter1_d_541_17506" x="153.701" y="24.2344" width="203.928" height="196.538" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="3.94961"/><feGaussianBlur stdDeviation="3.45591"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.16 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_541_17506"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_541_17506" result="shape"/></filter><clipPath id="clip0feedLayoutSettings_541_17506"><rect width="97.3351" height="82.6682" fill="white" transform="translate(54.8096 75.8527) rotate(-3)"/></clipPath><clipPath id="clip1feedLayoutSettings_541_17506"><rect x="175.256" y="27.1967" width="176.132" height="168.003" rx="2.27031" transform="rotate(5 175.256 27.1967)" fill="white"/></clipPath><clipPath id="clip2feedLayoutSettings_541_17506"><rect width="112.002" height="124.647" fill="white" transform="translate(204.782 52.4471) rotate(5)"/></clipPath><clipPath id="clip3feedLayoutSettings_541_17506"><rect width="112.002" height="124.647" fill="white" transform="translate(322.654 62.7597) rotate(5)"/></clipPath><clipPath id="clip4feedLayoutSettings_541_17506"><rect width="112.002" height="124.647" fill="white" transform="translate(86.0059 42.0556) rotate(5)"/></clipPath></defs></svg>',
							'bullets'         => array(
								'heading' => __( 'And get much more!', 'instagram-feed' ),
								'content' => array(
									__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
									__( 'Powerful visual moderation', 'instagram-feed' ),
									__( 'Comments and Likes', 'instagram-feed' ),
									__( 'Highlight specific posts', 'instagram-feed' ),
									__( 'Multiple layout options', 'instagram-feed' ),
									__( 'Popup photo/video lightbox', 'instagram-feed' ),
									__( 'Instagram Stories', 'instagram-feed' ),
									__( 'Shoppable feeds', 'instagram-feed' ),
									__( 'Pro support', 'instagram-feed' ),
									__( 'Post captions', 'instagram-feed' ),
									__( 'Combine multiple feed types', 'instagram-feed' ),
									__( '30 day money back guarantee', 'instagram-feed' ),
								)
							),
							'buyUrl'          => sprintf( 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=customizer&utm_medium=feed-layouts' )
						),
						'headerLayout'     => array(
							'heading'         => __( 'Get Stories, Followers and Advanced Header Options', 'instagram-feed' ),
							'description'     => __( 'Got stories to tell? We want to help you share them. Display Instagram stories right on your website in a pop-up lightbox to keep your users engaged and on your website for longer.', 'instagram-feed' ),
							'img'             => '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g filter="url(#headerLayoutfilter0_d_543_17792)"><rect x="113.089" y="44.6909" width="219.908" height="169.263" rx="3" transform="rotate(3 113.089 44.6909)" fill="#0096CC"/><circle cx="219.925" cy="94.3318" r="21.9039" transform="rotate(3 219.925 94.3318)" fill="#DCDDE1"/><circle cx="219.925" cy="94.3318" r="23.99" transform="rotate(3 219.925 94.3318)" stroke="url(#paint0_linear_543_17792)" stroke-width="1.04304"/><rect x="148.973" y="123.979" width="134.61" height="11.995" rx="1.33278" transform="rotate(3 148.973 123.979)" fill="#E8E8EB"/><rect x="166.559" y="144.919" width="97.2927" height="11.995" rx="1.33278" transform="rotate(3 166.559 144.919)" fill="#E8E8EB"/><path d="M191.365 180.919C193.326 181.022 194.997 179.517 195.1 177.556C195.203 175.595 193.698 173.923 191.737 173.821C189.776 173.718 188.104 175.223 188.002 177.184C187.899 179.145 189.404 180.816 191.365 180.919ZM191.272 182.694C188.903 182.57 184.111 183.511 183.987 185.871L183.894 187.645L198.091 188.39L198.184 186.615C198.308 184.255 193.641 182.818 191.272 182.694Z" fill="#141B38"/><path d="M211.234 187.974L213.955 188.116L214.643 174.987L211.913 174.844L208.396 177.023L208.268 179.461L211.736 177.362L211.79 177.365L211.234 187.974ZM217.462 188.3L227.006 188.8L227.12 186.617L221.17 186.305L221.176 186.196L224.055 183.828C226.43 181.864 227.239 180.866 227.323 179.274L227.324 179.256C227.441 177.017 225.613 175.334 222.838 175.189C220.063 175.043 217.916 176.673 217.786 179.148L217.782 179.221L220.33 179.354L220.343 179.273C220.442 178.092 221.398 177.257 222.744 177.328C223.909 177.389 224.688 178.196 224.641 179.27L224.64 179.288C224.592 180.198 224.203 180.826 222.428 182.32L217.56 186.417L217.462 188.3ZM232.551 189.091L235.271 189.233L235.959 176.105L233.23 175.962L229.712 178.14L229.585 180.578L233.052 178.479L233.107 178.482L232.551 189.091ZM238.805 189.419L241.462 189.558L241.631 186.328L242.353 185.609L244.937 189.74L248.004 189.901L244.428 184.139L248.333 180.138L245.321 179.98L241.823 183.71L241.768 183.708L242.15 176.429L239.493 176.29L238.805 189.419Z" fill="#141B38"/></g><g filter="url(#headerLayoutfilter1_d_543_17792)"><rect x="88.5157" y="41" width="219.908" height="169.263" rx="3" fill="white"/><circle cx="197.804" cy="84.9817" r="24.6564" stroke="url(#paint1_linear_543_17792)" stroke-width="1.33278"/><rect x="128.499" y="118.301" width="134.61" height="11.995" rx="1.33278" fill="#E8E8EB"/><rect x="147.158" y="138.293" width="97.2927" height="11.995" rx="1.33278" fill="#E8E8EB"/><path d="M173.813 172.945C175.776 172.945 177.367 171.354 177.367 169.391C177.367 167.427 175.776 165.837 173.813 165.837C171.849 165.837 170.259 167.427 170.259 169.391C170.259 171.354 171.849 172.945 173.813 172.945ZM173.813 174.722C171.44 174.722 166.705 175.913 166.705 178.276V180.053H180.921V178.276C180.921 175.913 176.185 174.722 173.813 174.722Z" fill="#141B38"/><path d="M194.025 178.95H196.749V165.803H194.016L190.618 168.163V170.604L193.971 168.327H194.025V178.95ZM200.261 178.95H209.818V176.763H203.86V176.654L206.611 174.139C208.88 172.053 209.636 171.014 209.636 169.42V169.402C209.636 167.161 207.723 165.575 204.944 165.575C202.165 165.575 200.106 167.315 200.106 169.794V169.867H202.657L202.666 169.785C202.703 168.6 203.614 167.716 204.962 167.716C206.128 167.716 206.948 168.482 206.957 169.557V169.575C206.957 170.486 206.602 171.133 204.907 172.718L200.261 177.064V178.95ZM215.371 178.95H218.095V165.803H215.362L211.963 168.163V170.604L215.316 168.327H215.371V178.95ZM221.634 178.95H224.294V175.716L224.977 174.959L227.774 178.95H230.845L226.973 173.383L230.663 169.183H227.647L224.349 173.092H224.294V165.803H221.634V178.95Z" fill="#141B38"/><g clip-path="url(#headerLayoutclip0_543_17792)"><rect x="176" y="63" width="44" height="44" rx="22" fill="#0068A0"/><circle cx="198" cy="80" r="8" fill="#B5E5FF"/><circle cx="198" cy="106" r="15" fill="#B5E5FF"/></g></g><defs><filter id="headerLayoutfilter0_d_543_17792" x="96.2966" y="40.7241" width="244.333" height="196.407" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="2.9751" operator="erode" in="SourceAlpha" result="effect1_dropShadow_543_17792"/><feOffset dy="3.9668"/><feGaussianBlur stdDeviation="5.45435"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_543_17792"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_543_17792" result="shape"/></filter><filter id="headerLayoutfilter1_d_543_17792" x="80.5822" y="37.0332" width="235.775" height="185.13" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="2.9751" operator="erode" in="SourceAlpha" result="effect1_dropShadow_543_17792"/><feOffset dy="3.9668"/><feGaussianBlur stdDeviation="5.45435"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_543_17792"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_543_17792" result="shape"/></filter><linearGradient id="paint0_linear_543_17792" x1="213.028" y1="162.104" x2="314.086" y2="58.9466" gradientUnits="userSpaceOnUse"><stop stop-color="white"/><stop offset="0.147864" stop-color="#F6640E"/><stop offset="0.443974" stop-color="#BA03A7"/><stop offset="0.733337" stop-color="#6A01B9"/><stop offset="1" stop-color="#6B01B9"/></linearGradient><linearGradient id="paint1_linear_543_17792" x1="190.906" y1="152.753" x2="291.964" y2="49.5964" gradientUnits="userSpaceOnUse"><stop stop-color="white"/><stop offset="0.147864" stop-color="#F6640E"/><stop offset="0.443974" stop-color="#BA03A7"/><stop offset="0.733337" stop-color="#6A01B9"/><stop offset="1" stop-color="#6B01B9"/></linearGradient><clipPath id="headerLayoutclip0_543_17792"><rect x="176" y="63" width="44" height="44" rx="22" fill="white"/></clipPath></defs></svg>',
							'bullets'         => array(
								'heading' => __( 'And get much more!', 'instagram-feed' ),
								'content' => array(
									__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
									__( 'Powerful visual moderation', 'instagram-feed' ),
									__( 'Comments and Likes', 'instagram-feed' ),
									__( 'Highlight specific posts', 'instagram-feed' ),
									__( 'Multiple layout options', 'instagram-feed' ),
									__( 'Popup photo/video lightbox', 'instagram-feed' ),
									__( 'Instagram Stories', 'instagram-feed' ),
									__( 'Shoppable feeds', 'instagram-feed' ),
									__( 'Pro support', 'instagram-feed' ),
									__( 'Post captions', 'instagram-feed' ),
									__( 'Combine multiple feed types', 'instagram-feed' ),
									__( '30 day money back guarantee', 'instagram-feed' ),
								)
							),
							'buyUrl'          => sprintf( 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=customizer&utm_medium=header' )
						),

						'postStyling'      => array(
							'heading'         => __( 'Display Captions, Likes, and Comments', 'instagram-feed' ),
							'description'     => __( 'Upgrade to Pro to display post captions below each post and in the lightbox, which can be crawled by search engines to help boost SEO.', 'instagram-feed' ),
							'img'             => '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g filter="url(#hoverstatefilter0_d_557_20473)"><g clip-path="url(#hoverstateclip0_557_20473)"><rect x="196.486" y="41.9882" width="139.542" height="147.818" rx="1.74428" transform="rotate(3 196.486 41.9882)" fill="#59AB46"/><path opacity="0.4" d="M207.272 102.814C208.625 102.884 212.65 103.095 217.927 103.372C224.523 103.718 224.523 103.718 230.104 104.01C235.686 104.303 236.7 104.356 250.4 105.074C264.099 105.792 261.562 105.659 268.666 106.031C275.769 106.403 276.784 106.457 289.976 107.148C300.529 107.701 312.977 108.353 317.882 108.61" stroke="white" stroke-width="6.91919"/><path opacity="0.4" d="M208.185 85.3947C209.538 85.4656 213.563 85.6766 218.84 85.9531C225.436 86.2988 225.436 86.2988 231.017 86.5913C236.599 86.8838 237.613 86.937 251.313 87.6549C265.012 88.3729 262.475 88.2399 269.579 88.6122C276.682 88.9845 277.697 89.0376 290.889 89.729C301.442 90.2821 313.89 90.9345 318.795 91.1915" stroke="white" stroke-width="6.91919"/><path opacity="0.4" d="M206.359 120.232C207.711 120.303 211.732 120.514 217.003 120.79C223.592 121.136 223.592 121.136 229.168 121.428C234.743 121.72 234.236 121.693 247.922 122.411C261.607 123.128 260.593 123.075 268.196 123.473" stroke="white" stroke-width="6.91919"/><path d="M272.181 159.772C272.033 159.764 271.895 159.698 271.796 159.589C271.698 159.479 271.647 159.335 271.654 159.187L271.742 157.521L269.52 157.405C269.225 157.389 268.949 157.258 268.752 157.038C268.554 156.819 268.452 156.53 268.468 156.236L268.817 149.571C268.832 149.277 268.964 149 269.183 148.803C269.403 148.605 269.691 148.503 269.986 148.519L278.872 148.984C279.167 149 279.443 149.132 279.64 149.351C279.838 149.57 279.94 149.859 279.925 150.153L279.575 156.818C279.56 157.113 279.428 157.389 279.209 157.586C278.99 157.784 278.701 157.886 278.406 157.87L275.018 157.693L272.855 159.646C272.739 159.745 272.597 159.794 272.458 159.786L272.181 159.772Z" fill="white"/><path d="M287.454 158.704L289.234 158.798L289.684 150.206L287.898 150.113L285.596 151.538L285.513 153.134L287.782 151.76L287.818 151.762L287.454 158.704ZM291.529 158.918L297.774 159.245L297.849 157.816L293.955 157.612L293.959 157.541L295.843 155.992C297.397 154.706 297.927 154.053 297.981 153.011L297.982 152.999C298.059 151.535 296.863 150.433 295.047 150.338C293.231 150.243 291.826 151.31 291.741 152.929L291.739 152.977L293.406 153.064L293.414 153.011C293.479 152.238 294.104 151.692 294.986 151.738C295.748 151.778 296.257 152.306 296.226 153.009L296.226 153.021C296.195 153.616 295.94 154.027 294.779 155.005L291.593 157.685L291.529 158.918Z" fill="white"/><g clip-path="url(#hoverstateclip1_557_20473)"><path d="M218.664 146.871C219.194 148.86 218.561 150.822 217.366 152.263C216.576 153.241 215.649 154.071 214.709 154.79C213.84 155.518 211.907 156.946 211.413 156.963C210.983 156.857 210.522 156.34 210.19 156.069C208.33 154.495 206.339 152.595 205.384 150.582C204.596 148.656 204.716 146.331 206.172 144.962C208.045 143.443 210.726 143.899 212.006 145.656C212.398 145.198 212.87 144.847 213.42 144.602C215.64 143.848 217.855 144.851 218.664 146.871Z" fill="white"/></g><path d="M226.488 155.509L228.269 155.602L228.719 147.011L226.933 146.918L224.631 148.343L224.548 149.939L226.816 148.565L226.852 148.567L226.488 155.509ZM230.563 155.723L236.809 156.05L236.884 154.621L232.99 154.417L232.994 154.346L234.878 152.797C236.432 151.511 236.961 150.858 237.016 149.816L237.017 149.804C237.093 148.34 235.897 147.238 234.082 147.143C232.266 147.048 230.861 148.114 230.776 149.734L230.773 149.782L232.44 149.869L232.449 149.816C232.513 149.043 233.139 148.497 234.02 148.543C234.782 148.583 235.292 149.111 235.261 149.814L235.26 149.826C235.229 150.421 234.975 150.832 233.813 151.809L230.628 154.49L230.563 155.723ZM242.229 156.334L243.938 156.424L244.021 154.846L245.152 154.905L245.227 153.47L244.096 153.411L244.388 147.832L241.852 147.699L238.116 153.032L238.037 154.532L242.312 154.756L242.229 156.334ZM239.676 153.227L239.678 153.18L242.606 149.04L242.647 149.043L242.42 153.371L239.676 153.227Z" fill="white"/></g></g><g filter="url(#hoverstatefilter1_d_557_20473)"><g clip-path="url(#hoverstateclip2_557_20473)"><rect x="63" y="55.1204" width="139.542" height="164.834" rx="1.74428" fill="white"/><path d="M74.5776 156.85C75.7308 156.82 79.1616 156.769 83.6591 156.797C89.2809 156.832 89.2809 156.876 94.0379 156.85C98.7948 156.823 99.6597 156.771 111.336 156.823C123.012 156.876 120.85 156.876 126.904 156.85C132.958 156.823 133.823 156.81 145.067 156.85C154.062 156.881 164.671 156.877 168.852 156.872" stroke="#DCDDE1" stroke-width="6.91919"/><path d="M74.5776 171.754C75.7308 171.729 79.1616 171.685 83.6591 171.709C89.2809 171.739 89.2809 171.777 94.0379 171.754C98.7948 171.731 98.3623 171.709 110.038 171.754C121.715 171.799 120.85 171.789 127.336 171.783" stroke="#DCDDE1" stroke-width="6.91919"/><g clip-path="url(#hoverstateclip3_557_20473)"><rect x="72.5933" y="52.5039" width="132.565" height="88.9581" fill="#2C324C"/><circle cx="97.5117" cy="88.7693" r="71.0037" fill="#0068A0"/><circle cx="209.41" cy="160.88" r="73.7586" fill="#FE544F"/></g><path d="M143.009 210.272C142.862 210.272 142.72 210.214 142.616 210.109C142.512 210.005 142.453 209.864 142.453 209.716V208.048H140.229C139.934 208.048 139.651 207.93 139.442 207.722C139.233 207.513 139.116 207.23 139.116 206.935V200.262C139.116 199.967 139.233 199.684 139.442 199.475C139.651 199.266 139.934 199.149 140.229 199.149H149.127C149.422 199.149 149.705 199.266 149.913 199.475C150.122 199.684 150.239 199.967 150.239 200.262V206.935C150.239 207.23 150.122 207.513 149.913 207.722C149.705 207.93 149.422 208.048 149.127 208.048H145.734L143.677 210.111C143.565 210.217 143.426 210.272 143.287 210.272H143.009Z" fill="#434960"/><path d="M158.207 208.407H159.989V199.804H158.201L155.977 201.348V202.946L158.171 201.455H158.207V208.407ZM162.287 208.407H168.541V206.976H164.642V206.904L166.443 205.259C167.927 203.894 168.422 203.214 168.422 202.171V202.159C168.422 200.692 167.17 199.655 165.352 199.655C163.533 199.655 162.186 200.793 162.186 202.415V202.463H163.855L163.861 202.409C163.885 201.634 164.481 201.056 165.364 201.056C166.127 201.056 166.663 201.556 166.669 202.26V202.272C166.669 202.868 166.437 203.291 165.328 204.329L162.287 207.173V208.407Z" fill="#434960"/><g clip-path="url(#hoverstateclip4_557_20473)"><path d="M88.8934 200.19C89.5271 202.148 88.9967 204.141 87.879 205.642C87.1412 206.66 86.2595 207.538 85.358 208.305C84.5286 209.077 82.6726 210.605 82.1806 210.647C81.7458 210.564 81.2578 210.072 80.9126 209.819C78.9728 208.344 76.8846 206.551 75.8258 204.591C74.9381 202.708 74.9365 200.38 76.3181 198.937C78.1096 197.322 80.8104 197.637 82.1806 199.325C82.5486 198.847 83.001 198.472 83.5381 198.199C85.7151 197.33 87.9798 198.215 88.8934 200.19Z" fill="#434960"/></g><path d="M97.1573 208.407H98.9399V199.804H97.1514L94.9276 201.348V202.946L97.1216 201.455H97.1573V208.407ZM101.238 208.407H107.492V206.976H103.593V206.904L105.393 205.259C106.878 203.894 107.373 203.214 107.373 202.171V202.159C107.373 200.692 106.121 199.655 104.302 199.655C102.484 199.655 101.137 200.793 101.137 202.415V202.463H102.806L102.812 202.409C102.836 201.634 103.432 201.056 104.314 201.056C105.077 201.056 105.614 201.556 105.62 202.26V202.272C105.62 202.868 105.387 203.291 104.278 204.329L101.238 207.173V208.407ZM112.92 208.407H114.631V206.827H115.764V205.39H114.631V199.804H112.091L108.639 205.324V206.827H112.92V208.407ZM110.207 205.438V205.39L112.914 201.103H112.956V205.438H110.207Z" fill="#434960"/></g></g><path d="M293.823 189.298L293.89 188.027L292.48 187.953L292.546 186.682L289.727 186.534L289.793 185.263L285.564 185.041L285.63 183.77L282.811 183.623L283.077 178.539L281.667 178.465L281.734 177.194L278.914 177.046L278.848 178.317L277.438 178.243L276.838 189.682L275.429 189.608L275.495 188.337L271.266 188.115L270.999 193.199L272.409 193.273L272.276 195.815L273.686 195.889L273.553 198.431L274.962 198.505L274.829 201.047L276.239 201.121L276.039 204.934L291.547 205.746L291.814 200.663L293.224 200.736L293.423 196.923L292.014 196.85L291.814 200.663L290.404 200.589L290.204 204.402L277.516 203.737L277.649 201.195L276.239 201.121L276.372 198.579L274.962 198.505L275.096 195.963L273.686 195.889L273.819 193.347L272.409 193.273L272.609 189.46L275.429 189.608L275.362 190.879L276.772 190.953L276.572 194.766L277.982 194.84L278.848 178.317L281.667 178.465L281.068 189.904L282.478 189.977L282.744 184.894L285.564 185.041L285.297 190.125L286.707 190.199L286.907 186.386L289.727 186.534L289.46 191.618L290.87 191.692L291.07 187.879L292.48 187.953L292.413 189.224L293.823 189.298L293.423 196.923L294.833 196.997L295.233 189.371L293.823 189.298Z" fill="#141B38"/><path d="M292.014 196.85L293.424 196.923L293.823 189.298L292.413 189.224L292.48 187.953L291.07 187.879L290.87 191.692L289.461 191.618L289.727 186.534L286.907 186.386L286.708 190.199L285.298 190.125L285.564 185.041L282.745 184.894L282.478 189.977L281.068 189.904L281.668 178.465L278.848 178.317L277.982 194.84L276.572 194.766L276.772 190.953L275.362 190.879L275.429 189.608L272.609 189.46L272.409 193.273L273.819 193.347L273.686 195.889L275.096 195.963L274.963 198.505L276.372 198.579L276.239 201.121L277.649 201.195L277.516 203.737L290.204 204.402L290.404 200.589L291.814 200.663L292.014 196.85Z" fill="white"/><path fill-rule="evenodd" clip-rule="evenodd" d="M280.869 193.717L282.278 193.791L281.964 199.782L280.555 199.708L280.869 193.717ZM285.098 193.938L283.688 193.864L283.374 199.856L284.784 199.93L285.098 193.938ZM287.917 194.086L286.507 194.012L286.193 200.004L287.603 200.078L287.917 194.086Z" fill="#141B38"/><defs><filter id="hoverstatefilter0_d_557_20473" x="181.831" y="38.5286" width="160.926" height="168.757" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="2.5947" operator="erode" in="SourceAlpha" result="effect1_dropShadow_557_20473"/><feOffset dy="3.45959"/><feGaussianBlur stdDeviation="4.75694"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_557_20473"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_557_20473" result="shape"/></filter><filter id="hoverstatefilter1_d_557_20473" x="56.0808" y="51.6608" width="153.38" height="178.672" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="2.5947" operator="erode" in="SourceAlpha" result="effect1_dropShadow_557_20473"/><feOffset dy="3.45959"/><feGaussianBlur stdDeviation="4.75694"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_557_20473"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_557_20473" result="shape"/></filter><clipPath id="hoverstateclip0_557_20473"><rect x="196.486" y="41.9882" width="139.542" height="147.818" rx="1.74428" transform="rotate(3 196.486 41.9882)" fill="white"/></clipPath><clipPath id="hoverstateclip1_557_20473"><rect width="13.9542" height="13.9542" fill="white" transform="translate(205.154 143.248) rotate(3)"/></clipPath><clipPath id="hoverstateclip2_557_20473"><rect x="63" y="55.1204" width="139.542" height="164.834" rx="1.74428" fill="white"/></clipPath><clipPath id="hoverstateclip3_557_20473"><rect width="139.542" height="86.3416" fill="white" transform="translate(62.9998 55.1204)"/></clipPath><clipPath id="hoverstateclip4_557_20473"><rect width="13.9542" height="13.9542" fill="white" transform="translate(75.21 197.279)"/></clipPath></defs></svg>',
							'bullets'         => array(
								'heading' => __( 'And get much more!', 'instagram-feed' ),
								'content' => array(
									__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
									__( 'Powerful visual moderation', 'instagram-feed' ),
									__( 'Comments and Likes', 'instagram-feed' ),
									__( 'Highlight specific posts', 'instagram-feed' ),
									__( 'Multiple layout options', 'instagram-feed' ),
									__( 'Popup photo/video lightbox', 'instagram-feed' ),
									__( 'Instagram Stories', 'instagram-feed' ),
									__( 'Shoppable feeds', 'instagram-feed' ),
									__( 'Pro support', 'instagram-feed' ),
									__( 'Post captions', 'instagram-feed' ),
									__( 'Combine multiple feed types', 'instagram-feed' ),
									__( '30 day money back guarantee', 'instagram-feed' ),
								)
							),
							'buyUrl'          => sprintf( 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=customizer&utm_medium=posts' )
						),

						'lightbox'         => array(
							'heading'         => __( 'Upgrade to Pro to enable the popup Lightbox', 'instagram-feed' ),
							'description'     => __( 'Allow visitors to view your photos and videos in a beautiful full size lightbox, keeping them on your site for longer to discover more of your content.', 'instagram-feed' ),
							'img'             => '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M50.5998 136.122L48.2638 134.019L39.3134 143.959L49.2538 152.91L51.3572 150.574L43.7685 143.726L50.5998 136.122Z" fill="#8C8F9A"/><path d="M347.201 119.809L345.097 122.145L352.686 128.993L345.855 136.597L348.191 138.7L357.141 128.76L347.201 119.809Z" fill="#8C8F9A"/><g clip-path="url(#lightboxxclip0_557_20899)" filter="url(#lightboxxfilter0_d_557_20899)"><rect width="261.925" height="173.162" transform="translate(62.6831 52.3919) rotate(-3)" fill="white"/><rect x="112.468" y="187.874" width="93.129" height="5.82056" rx="1.45514" transform="rotate(-3 112.468 187.874)" fill="#D0D1D7"/><rect x="113.151" y="200.952" width="53.8402" height="5.82056" rx="1.45514" transform="rotate(-3 113.151 200.952)" fill="#D0D1D7"/><circle cx="94.1753" cy="195.21" r="8.73084" transform="rotate(-3 94.1753 195.21)" fill="#DCDDE1"/><g clip-path="url(#lightboxxclip1_557_20899)"><rect x="62.6812" y="52.3918" width="262.169" height="173.162" transform="rotate(-3 62.6812 52.3918)" fill="#FE544F"/><rect x="191.14" y="20.5734" width="271.58" height="334.479" rx="55.73" transform="rotate(2.99107 191.14 20.5734)" fill="#DCDDE1"/><circle cx="141.741" cy="201.742" r="113.935" transform="rotate(-3 141.741 201.742)" fill="#0096CC"/></g></g><defs><filter id="lightboxxfilter0_d_557_20899" x="53.8243" y="34.2544" width="288.346" height="204.35" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="3.32203" operator="erode" in="SourceAlpha" result="effect1_dropShadow_557_20899"/><feOffset dy="4.42938"/><feGaussianBlur stdDeviation="6.0904"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_557_20899"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_557_20899" result="shape"/></filter><clipPath id="lightboxxclip0_557_20899"><rect width="261.925" height="173.162" fill="white" transform="translate(62.6831 52.3919) rotate(-3)"/></clipPath><clipPath id="lightboxxclip1_557_20899"><rect width="262.15" height="121.608" fill="white" transform="translate(62.6821 52.3919) rotate(-3)"/></clipPath></defs></svg>',
							'bullets'         => array(
								'heading' => __( 'And get much more!', 'instagram-feed' ),
								'content' => array(
									__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
									__( 'Powerful visual moderation', 'instagram-feed' ),
									__( 'Comments and Likes', 'instagram-feed' ),
									__( 'Highlight specific posts', 'instagram-feed' ),
									__( 'Multiple layout options', 'instagram-feed' ),
									__( 'Popup photo/video lightbox', 'instagram-feed' ),
									__( 'Instagram Stories', 'instagram-feed' ),
									__( 'Shoppable feeds', 'instagram-feed' ),
									__( 'Pro support', 'instagram-feed' ),
									__( 'Post captions', 'instagram-feed' ),
									__( 'Combine multiple feed types', 'instagram-feed' ),
									__( '30 day money back guarantee', 'instagram-feed' ),
								)
							),
							'buyUrl'          => sprintf( 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=customizer&utm_medium=lightbox' )
						),

						'filtermoderation' => array(
							'heading'         => __( 'Get Advanced Moderation and Filters with Pro', 'instagram-feed' ),
							'description'     => __( 'Use powerful moderation tools to create feeds of only specific chosen posts, or exclude specific chosen posts. You can also automatically include or exclude posts based on a word or hashtag found in the caption.', 'instagram-feed' ),
							'img'             => '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#moderationfltrclip0_542_16736)"><g filter="url(#moderationfltrfilter0_ddd_542_16736)"><rect x="139.981" y="52.5992" width="162.17" height="179.401" rx="3.96584" fill="white"/></g><path d="M156.198 174.881C157.55 174.847 161.57 174.786 166.841 174.819C173.429 174.86 173.429 174.912 179.003 174.881C184.578 174.85 185.592 174.788 199.275 174.85C212.958 174.912 210.424 174.912 217.519 174.881C224.614 174.85 225.627 174.835 238.804 174.881C249.345 174.918 261.778 174.913 266.677 174.907" stroke="#DCDDE1" stroke-width="8.10851"/><path d="M156.198 194.559C157.55 194.53 161.57 194.478 166.841 194.506C173.429 194.542 173.429 194.586 179.003 194.559C184.578 194.533 184.071 194.506 197.754 194.559C211.437 194.613 210.424 194.6 218.026 194.593" stroke="#DCDDE1" stroke-width="8.10851"/><g clip-path="url(#moderationfltrclip1_542_16736)"><rect x="151.131" y="65.7755" width="139.912" height="88.1801" fill="#8C8F9A"/><circle cx="157.01" cy="165.713" r="48.2051" fill="#EC352F"/><circle cx="295.745" cy="112.805" r="65.8411" fill="#DCDDE1"/></g><circle cx="291.51" cy="58.1738" r="18.7509" fill="#D72C2C"/><path d="M290.886 55.6399L293.028 57.7751V57.667C293.028 57.1294 292.815 56.6137 292.435 56.2336C292.054 55.8534 291.539 55.6399 291.001 55.6399H290.886ZM287.981 56.1804L289.028 57.2278C288.994 57.3697 288.974 57.5116 288.974 57.667C288.974 58.2046 289.188 58.7202 289.568 59.1004C289.948 59.4805 290.464 59.6941 291.001 59.6941C291.15 59.6941 291.298 59.6738 291.44 59.6401L292.488 60.6874C292.035 60.9104 291.535 61.0455 291.001 61.0455C290.105 61.0455 289.246 60.6896 288.612 60.056C287.979 59.4224 287.623 58.563 287.623 57.667C287.623 57.1332 287.758 56.6332 287.981 56.1804ZM284.244 52.4438L285.785 53.9844L286.089 54.2884C284.974 55.1669 284.095 56.3156 283.568 57.667C284.737 60.6333 287.623 62.7348 291.001 62.7348C292.049 62.7348 293.049 62.5321 293.961 62.1672L294.251 62.451L296.224 64.4241L297.083 63.5659L285.102 51.5856L284.244 52.4438ZM291.001 54.2884C291.897 54.2884 292.757 54.6444 293.39 55.278C294.024 55.9116 294.38 56.7709 294.38 57.667C294.38 58.0994 294.292 58.5184 294.136 58.8968L296.116 60.8766C297.13 60.032 297.941 58.9238 298.434 57.667C297.265 54.7006 294.38 52.5992 291.001 52.5992C290.055 52.5992 289.15 52.7681 288.298 53.0722L289.765 54.5249C290.15 54.3763 290.562 54.2884 291.001 54.2884Z" fill="white"/><g filter="url(#moderationfltrfilter1_ddd_542_16736)"><rect x="85.7402" y="31.9814" width="162.17" height="179.401" rx="3.96584" fill="white"/><path d="M101.957 154.263C103.308 154.229 107.329 154.169 112.599 154.201C119.188 154.242 119.188 154.294 124.762 154.263C130.337 154.232 131.35 154.171 145.034 154.232C158.717 154.294 156.183 154.294 163.278 154.263C170.373 154.232 171.386 154.217 184.562 154.263C195.104 154.3 207.537 154.296 212.436 154.289" stroke="#DCDDE1" stroke-width="8.10851"/><path d="M101.957 173.942C103.308 173.912 107.329 173.86 112.599 173.889C119.188 173.924 119.188 173.968 124.762 173.942C130.337 173.915 129.83 173.889 143.513 173.942C157.196 173.995 156.183 173.982 163.784 173.975" stroke="#DCDDE1" stroke-width="8.10851"/><g clip-path="url(#moderationfltrclip2_542_16736)"><rect x="96.8887" y="45.1577" width="139.912" height="88.1801" fill="#2C324C"/><circle cx="125.771" cy="71.4144" r="83.2083" fill="#0068A0"/><circle cx="256.9" cy="155.92" r="86.4368" fill="#FE544F"/></g></g><circle cx="234.383" cy="30.7509" r="18.7509" fill="#0096CC"/><g clip-path="url(#moderationfltrclip3_542_16736)"><path d="M233.874 28.217C233.337 28.217 232.821 28.4306 232.441 28.8108C232.061 29.1909 231.847 29.7065 231.847 30.2442C231.847 30.7818 232.061 31.2974 232.441 31.6775C232.821 32.0577 233.337 32.2713 233.874 32.2713C234.412 32.2713 234.927 32.0577 235.308 31.6775C235.688 31.2974 235.901 30.7818 235.901 30.2442C235.901 29.7065 235.688 29.1909 235.308 28.8108C234.927 28.4306 234.412 28.217 233.874 28.217ZM233.874 33.6227C232.978 33.6227 232.119 33.2667 231.485 32.6331C230.852 31.9995 230.496 31.1402 230.496 30.2442C230.496 29.3481 230.852 28.4888 231.485 27.8552C232.119 27.2216 232.978 26.8656 233.874 26.8656C234.77 26.8656 235.63 27.2216 236.263 27.8552C236.897 28.4888 237.253 29.3481 237.253 30.2442C237.253 31.1402 236.897 31.9995 236.263 32.6331C235.63 33.2667 234.77 33.6227 233.874 33.6227ZM233.874 25.1763C230.496 25.1763 227.61 27.2778 226.441 30.2442C227.61 33.2105 230.496 35.312 233.874 35.312C237.253 35.312 240.138 33.2105 241.307 30.2442C240.138 27.2778 237.253 25.1763 233.874 25.1763Z" fill="white"/></g></g><defs><filter id="moderationfltrfilter0_ddd_542_16736" x="114.203" y="38.7187" width="213.726" height="230.957" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="11.8975"/><feGaussianBlur stdDeviation="12.889"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_542_16736"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.98292"/><feGaussianBlur stdDeviation="1.98292"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_542_16736" result="effect2_dropShadow_542_16736"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="5.94876"/><feGaussianBlur stdDeviation="5.94876"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_542_16736" result="effect3_dropShadow_542_16736"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_542_16736" result="shape"/></filter><filter id="moderationfltrfilter1_ddd_542_16736" x="59.9623" y="18.101" width="213.726" height="230.957" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="11.8975"/><feGaussianBlur stdDeviation="12.889"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_542_16736"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="1.98292"/><feGaussianBlur stdDeviation="1.98292"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/><feBlend mode="normal" in2="effect1_dropShadow_542_16736" result="effect2_dropShadow_542_16736"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="5.94876"/><feGaussianBlur stdDeviation="5.94876"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="effect2_dropShadow_542_16736" result="effect3_dropShadow_542_16736"/><feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow_542_16736" result="shape"/></filter><clipPath id="moderationfltrclip0_542_16736"><path d="M0 0H396V264H0V0Z" fill="white"/></clipPath><clipPath id="moderationfltrclip1_542_16736"><rect x="151.131" y="65.7755" width="139.912" height="88.1801" rx="2" fill="white"/></clipPath><clipPath id="moderationfltrclip2_542_16736"><rect x="96.8887" y="45.1577" width="139.912" height="88.1801" rx="2" fill="white"/></clipPath><clipPath id="moderationfltrclip3_542_16736"><rect width="16.217" height="16.217" fill="white" transform="translate(225.767 22.1356)"/></clipPath></defs></svg>',
							'bullets'         => array(
								'heading' => __( 'And get much more!', 'instagram-feed' ),
								'content' => array(
									__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
									__( 'Powerful visual moderation', 'instagram-feed' ),
									__( 'Comments and Likes', 'instagram-feed' ),
									__( 'Highlight specific posts', 'instagram-feed' ),
									__( 'Multiple layout options', 'instagram-feed' ),
									__( 'Popup photo/video lightbox', 'instagram-feed' ),
									__( 'Instagram Stories', 'instagram-feed' ),
									__( 'Shoppable feeds', 'instagram-feed' ),
									__( 'Pro support', 'instagram-feed' ),
									__( 'Post captions', 'instagram-feed' ),
									__( 'Combine multiple feed types', 'instagram-feed' ),
									__( '30 day money back guarantee', 'instagram-feed' ),
								)
							),
							'buyUrl'          => sprintf( 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=customizer&utm_medium=filters' )
						),

						'shoppablefeed'    => array(
							'heading'         => __( 'Upgrade to Pro to Get Shoppable Feeds', 'instagram-feed' ),
							'description'     => __( 'Automatically link Instagram posts to custom URLs of your choosing by adding the URL in the caption, or manually add links to specific pages or products on your site (or other sites) in a quick and easy way.', 'instagram-feed' ),
							'img'             => '<svg width="396" height="264" viewBox="0 0 396 264" fill="none" xmlns="http://www.w3.org/2000/svg"><g filter="url(#shoppablefeedfilter0_d_557_17550)"><rect x="234.717" y="38.2059" width="143" height="158.889" rx="2" transform="rotate(4 234.717 38.2059)" fill="white"/></g><rect width="143" height="82.0926" transform="translate(234.717 38.2059) rotate(4)" fill="#E8E8EB"/><g filter="url(#shoppablefeedfilter1_dd_557_17550)"><mask id="shoppablefeedmask0_557_17550" style="mask-type:alpha" maskUnits="userSpaceOnUse" x="266" y="53" width="80" height="62"><path fill-rule="evenodd" clip-rule="evenodd" d="M315.226 54.937C315.099 58.5599 310.591 61.343 305.157 61.1533C299.723 60.9635 295.42 57.8727 295.546 54.2498C295.546 54.2498 295.546 54.2497 295.546 54.2497L286.163 53.922C286.057 53.9183 285.951 53.9404 285.856 53.9863L278.402 57.5651C278.37 57.5802 278.351 57.6124 278.352 57.6471C278.35 57.62 278.321 57.6035 278.297 57.6159L266.724 63.5265C266.477 63.6528 266.336 63.921 266.372 64.1964L268.295 78.8772C268.344 79.2535 268.701 79.5097 269.073 79.4357L278.317 77.5986C278.714 77.5198 279.086 77.8146 279.101 78.2185L280.339 112.219C280.352 112.563 280.628 112.839 280.972 112.851L326.982 114.458C327.326 114.47 327.621 114.214 327.657 113.872L331.266 80.0236C331.308 79.6249 331.695 79.3572 332.083 79.4578L341.845 81.9875C342.209 82.0819 342.578 81.8513 342.653 81.4825L345.594 66.9741C345.651 66.6966 345.523 66.4143 345.277 66.2738L333.503 59.5421C333.48 59.5288 333.45 59.5433 333.447 59.57C333.45 59.5361 333.433 59.5032 333.404 59.4863L326.216 55.3957C326.124 55.3432 326.021 55.3139 325.915 55.3102L315.226 54.9369C315.226 54.9369 315.226 54.937 315.226 54.937Z" fill="white"/></mask><g mask="url(#shoppablefeedmask0_557_17550)"><rect x="261.444" y="49.1168" width="94.5192" height="70.8894" transform="rotate(4 261.444 49.1168)" fill="white"/><path fill-rule="evenodd" clip-rule="evenodd" d="M304.293 77.7265C302.782 76.9164 301.23 75.7559 299.747 75.1625C300.196 76.4854 300.555 77.8865 300.931 79.2729C299.943 79.9127 298.808 80.3837 297.719 80.9071C298.549 81.8507 299.776 82.4491 300.728 83.2863C299.853 84.2243 298.197 85.147 297.749 86.0211C299.431 85.9288 301.399 85.7258 302.956 85.7657C303.177 87.5038 303.222 89.3956 303.597 90.9999C304.464 88.9723 305.268 86.8705 306.263 84.99C307.552 85.6316 309.122 86.4139 310.395 86.828C309.575 85.4894 308.84 84.0769 308.131 82.6421C309.383 81.8618 310.648 81.0961 311.86 80.2694C310.247 80.0064 308.572 79.7978 306.872 79.6108C306.724 77.8331 306.942 75.7372 306.638 74.0953C305.915 75.3731 305.086 76.5293 304.293 77.7265ZM304.045 92.3479C303.766 93.2504 304.446 93.6761 304.301 94.2247C303.62 93.9356 303.104 93.7755 302.159 93.8425C302.238 93.1288 302.889 93.0725 302.817 92.1846C292.92 90.3527 294.16 72.3239 304.157 71.908C316.657 71.388 315.549 92.832 304.045 92.3479Z" fill="#FE544F"/><path fill-rule="evenodd" clip-rule="evenodd" d="M306.638 74.0951C306.942 75.7371 306.724 77.833 306.872 79.6107C308.572 79.7977 310.247 80.0062 311.86 80.2693C310.648 81.096 309.383 81.8617 308.131 82.642C308.84 84.0768 309.575 85.4893 310.395 86.8279C309.122 86.4138 307.552 85.6315 306.263 84.9899C305.268 86.8704 304.464 88.9721 303.597 90.9997C303.221 89.3955 303.177 87.5037 302.956 85.7655C301.399 85.7257 299.431 85.9287 297.749 86.021C298.197 85.1469 299.853 84.2242 300.728 83.2861C299.776 82.449 298.549 81.8506 297.719 80.907C298.808 80.3836 299.943 79.9126 300.931 79.2727C300.555 77.8864 300.196 76.4853 299.747 75.1624C301.23 75.7557 302.782 76.9163 304.293 77.7264C305.086 76.5292 305.915 75.3729 306.638 74.0951Z" fill="white"/></g></g><path d="M245.82 168.987C245.767 169.739 245.479 170.369 244.954 170.878C244.429 171.387 243.773 171.678 242.987 171.751L242.896 173.041C242.892 173.11 242.866 173.165 242.819 173.206C242.771 173.247 242.714 173.266 242.645 173.261L241.655 173.192C241.591 173.187 241.538 173.16 241.494 173.11C241.451 173.06 241.432 173.003 241.436 172.939L241.526 171.649C241.207 171.583 240.9 171.485 240.607 171.355C240.313 171.226 240.073 171.1 239.885 170.975C239.697 170.851 239.524 170.721 239.367 170.584C239.209 170.447 239.102 170.347 239.045 170.283C238.988 170.22 238.948 170.173 238.926 170.142C238.85 170.033 238.852 169.932 238.932 169.838L239.757 168.896C239.795 168.849 239.853 168.824 239.932 168.82C240.006 168.815 240.064 168.841 240.104 168.898L240.118 168.914C240.636 169.439 241.209 169.788 241.836 169.96C242.014 170.012 242.194 170.044 242.375 170.057C242.771 170.085 243.127 170.003 243.443 169.813C243.759 169.623 243.93 169.334 243.957 168.945C243.967 168.808 243.939 168.675 243.874 168.547C243.81 168.419 243.735 168.31 243.65 168.22C243.566 168.131 243.429 168.028 243.24 167.914C243.052 167.8 242.896 167.71 242.772 167.644C242.649 167.579 242.459 167.485 242.202 167.364C242.017 167.272 241.871 167.2 241.764 167.148C241.657 167.096 241.511 167.02 241.326 166.921C241.141 166.822 240.994 166.735 240.883 166.66C240.773 166.586 240.641 166.489 240.487 166.37C240.333 166.25 240.21 166.137 240.116 166.029C240.023 165.921 239.925 165.793 239.823 165.645C239.72 165.498 239.643 165.349 239.592 165.2C239.541 165.051 239.501 164.884 239.472 164.699C239.443 164.514 239.436 164.321 239.45 164.12C239.497 163.441 239.779 162.864 240.294 162.386C240.809 161.909 241.456 161.623 242.234 161.529L242.327 160.202C242.331 160.139 242.358 160.085 242.408 160.042C242.458 159.998 242.515 159.979 242.578 159.983L243.569 160.052C243.637 160.057 243.692 160.083 243.733 160.13C243.774 160.178 243.792 160.236 243.787 160.305L243.696 161.602C243.973 161.651 244.24 161.726 244.495 161.828C244.751 161.93 244.958 162.027 245.117 162.12C245.275 162.212 245.424 162.315 245.563 162.429C245.703 162.542 245.793 162.62 245.834 162.662C245.876 162.705 245.91 162.742 245.937 162.773C246.014 162.868 246.02 162.962 245.954 163.056L245.285 164.091C245.24 164.162 245.181 164.197 245.108 164.197C245.038 164.207 244.973 164.185 244.913 164.131C244.9 164.115 244.866 164.083 244.813 164.035C244.76 163.987 244.669 163.915 244.54 163.82C244.412 163.725 244.274 163.636 244.128 163.554C243.981 163.472 243.803 163.396 243.594 163.324C243.386 163.253 243.178 163.21 242.973 163.196C242.508 163.163 242.122 163.242 241.813 163.433C241.505 163.624 241.339 163.886 241.316 164.22C241.307 164.348 241.319 164.468 241.353 164.579C241.388 164.69 241.453 164.797 241.549 164.9C241.645 165.003 241.736 165.09 241.821 165.163C241.907 165.236 242.039 165.321 242.216 165.42C242.394 165.519 242.537 165.596 242.646 165.65C242.756 165.705 242.922 165.784 243.146 165.889C243.398 166.005 243.591 166.097 243.724 166.163C243.857 166.229 244.037 166.328 244.264 166.46C244.49 166.592 244.668 166.709 244.796 166.812C244.924 166.914 245.067 167.048 245.225 167.212C245.383 167.376 245.502 167.541 245.581 167.707C245.661 167.873 245.724 168.067 245.773 168.287C245.821 168.508 245.837 168.741 245.82 168.987Z" fill="#0068A0"/><rect x="240.659" y="143.036" width="74.1481" height="10.5926" transform="rotate(4 240.659 143.036)" fill="#DCDDE1"/><rect x="252.575" y="162.452" width="32.6605" height="10.5926" transform="rotate(4 252.575 162.452)" fill="#BFE8FF"/><rect x="328.66" y="112.025" width="33.5432" height="33.5432" rx="16.7716" transform="rotate(4 328.66 112.025)" fill="#FE544F"/><g clip-path="url(#shoppablefeedclip0_557_17550)"><path d="M338.611 121.57C338.377 121.554 338.147 121.631 337.97 121.784C337.794 121.938 337.685 122.155 337.669 122.389C337.653 122.622 337.73 122.853 337.883 123.03C338.037 123.206 338.254 123.315 338.488 123.331L339.562 123.406L339.756 124.501C339.757 124.514 339.759 124.526 339.762 124.539L340.623 129.404L339.782 130.134C338.595 131.166 339.248 133.118 340.817 133.227L348.377 133.756C348.611 133.772 348.841 133.695 349.018 133.542C349.195 133.388 349.303 133.171 349.319 132.937C349.336 132.704 349.259 132.473 349.105 132.296C348.952 132.12 348.734 132.011 348.5 131.995L340.94 131.466L341.882 130.647L347.682 131.053C347.845 131.064 348.008 131.03 348.153 130.954C348.299 130.877 348.419 130.762 348.503 130.621L351.514 125.522C351.59 125.393 351.632 125.246 351.636 125.096C351.64 124.945 351.605 124.796 351.535 124.663C351.465 124.53 351.363 124.417 351.237 124.335C351.111 124.252 350.966 124.204 350.816 124.193L341.376 123.533L341.18 122.419C341.145 122.226 341.047 122.049 340.901 121.917C340.755 121.786 340.568 121.707 340.372 121.693L338.611 121.57Z" fill="white"/><path d="M349.104 136.019C349.08 136.369 348.917 136.696 348.652 136.926C348.387 137.156 348.041 137.272 347.691 137.248C347.34 137.223 347.014 137.06 346.784 136.795C346.553 136.53 346.438 136.185 346.462 135.834C346.487 135.484 346.649 135.158 346.915 134.928C347.18 134.697 347.525 134.581 347.876 134.606C348.226 134.63 348.552 134.793 348.782 135.058C349.013 135.323 349.129 135.669 349.104 136.019Z" fill="white"/><path d="M340.646 136.755C340.996 136.78 341.342 136.664 341.607 136.433C341.872 136.203 342.035 135.877 342.059 135.527C342.084 135.176 341.968 134.831 341.738 134.566C341.507 134.3 341.181 134.138 340.831 134.113C340.48 134.089 340.135 134.204 339.87 134.435C339.605 134.665 339.442 134.992 339.417 135.342C339.393 135.692 339.509 136.038 339.739 136.303C339.969 136.568 340.296 136.731 340.646 136.755Z" fill="white"/></g><path d="M355.663 146.817L355.732 145.836L354.644 145.76L354.712 144.78L352.537 144.628L352.605 143.647L349.342 143.419L349.411 142.438L347.235 142.286L347.509 138.363L346.422 138.287L346.49 137.307L344.315 137.155L344.246 138.135L343.158 138.059L342.541 146.885L341.454 146.809L341.522 145.828L338.259 145.6L337.985 149.523L339.072 149.599L338.935 151.56L340.023 151.636L339.886 153.597L340.974 153.673L340.836 155.634L341.924 155.711L341.718 158.652L353.684 159.489L353.958 155.567L355.046 155.643L355.252 152.701L354.164 152.625L353.958 155.567L352.87 155.491L352.665 158.432L342.875 157.748L343.012 155.787L341.924 155.711L342.061 153.749L340.974 153.673L341.111 151.712L340.023 151.636L340.16 149.675L339.072 149.599L339.278 146.657L341.454 146.809L341.385 147.789L342.473 147.866L342.267 150.807L343.355 150.883L344.246 138.135L346.422 138.287L345.805 147.113L346.892 147.189L347.167 143.267L349.342 143.419L349.068 147.341L350.156 147.417L350.361 144.475L352.537 144.628L352.263 148.55L353.35 148.626L353.556 145.684L354.644 145.76L354.575 146.741L355.663 146.817L355.252 152.701L356.339 152.777L356.751 146.893L355.663 146.817Z" fill="#141B38"/><path d="M354.164 152.625L355.252 152.701L355.663 146.817L354.576 146.741L354.644 145.76L353.556 145.684L353.351 148.626L352.263 148.55L352.537 144.628L350.362 144.476L350.156 147.417L349.068 147.341L349.342 143.419L347.167 143.267L346.893 147.189L345.805 147.113L346.422 138.287L344.247 138.135L343.355 150.884L342.267 150.807L342.473 147.866L341.385 147.79L341.454 146.809L339.278 146.657L339.073 149.599L340.16 149.675L340.023 151.636L341.111 151.712L340.974 153.673L342.062 153.749L341.924 155.711L343.012 155.787L342.875 157.748L352.665 158.433L352.871 155.491L353.958 155.567L354.164 152.625Z" fill="white"/><path fill-rule="evenodd" clip-rule="evenodd" d="M345.6 150.055L346.687 150.131L346.364 154.754L345.276 154.678L345.6 150.055ZM348.861 150.283L347.774 150.207L347.45 154.83L348.538 154.906L348.861 150.283ZM351.037 150.435L349.949 150.359L349.626 154.982L350.714 155.058L351.037 150.435Z" fill="#141B38"/><g filter="url(#shoppablefeedfilter2_d_557_17550)"><g clip-path="url(#shoppablefeedclip1_557_17550)"><rect x="19.4355" y="57.3804" width="135.359" height="149.741" rx="1.66935" transform="rotate(-4 19.4355 57.3804)" fill="white"/><path d="M40.0569 158.253C41.1801 158.146 44.5242 157.862 48.9146 157.582C54.4026 157.233 54.4056 157.275 59.0454 156.925C63.6853 156.575 64.5256 156.465 75.9223 155.719C87.319 154.974 85.2092 155.121 91.115 154.683C97.0207 154.244 97.8637 154.172 108.838 153.443C117.617 152.86 127.969 152.133 132.047 151.842" stroke="#DCDDE1" stroke-width="6.76797"/><path d="M41.2028 174.638C42.3263 174.535 45.6709 174.258 50.061 173.974C55.5487 173.62 55.5512 173.657 60.1913 173.31C64.8314 172.964 64.4079 172.971 75.8041 172.219C87.2003 171.466 86.3557 171.515 92.6848 171.066" stroke="#DCDDE1" stroke-width="6.76797"/><g clip-path="url(#shoppablefeedclip2_557_17550)"><rect x="28.8848" y="56.2416" width="126" height="85" transform="rotate(-4 28.8848 56.2416)" fill="#2C324C"/><circle cx="55.061" cy="87.8833" r="69.4519" transform="rotate(-4 55.061 87.8833)" fill="#0068A0"/><circle cx="169.165" cy="150.611" r="72.1466" transform="rotate(-4 169.165 150.611)" fill="#FE544F"/><g filter="url(#shoppablefeedfilter3_dd_557_17550)"><mask id="shoppablefeedmask1_557_17550" style="mask-type:alpha" maskUnits="userSpaceOnUse" x="48" y="63" width="85" height="66"><path fill-rule="evenodd" clip-rule="evenodd" d="M99.0497 64.5008C99.3159 68.3075 94.9027 71.7172 89.1926 72.1164C83.4825 72.5157 78.6378 69.7535 78.3716 65.9467C78.3716 65.9467 78.3716 65.9467 78.3715 65.9467L68.5145 66.6359C68.4031 66.6437 68.2952 66.6784 68.2002 66.737L60.7405 71.3345C60.7395 71.3351 60.739 71.3362 60.7391 71.3374C60.739 71.3365 60.7379 71.336 60.7372 71.3365L49.2225 78.8298C48.9775 78.9893 48.8594 79.2856 48.9274 79.5698L52.5559 94.7276C52.6489 95.1162 53.0511 95.3451 53.4327 95.2267L62.9064 92.2867C63.3127 92.1606 63.7354 92.4282 63.7952 92.8495L68.8327 128.308C68.8837 128.667 69.2033 128.925 69.5649 128.9L117.909 125.52C118.27 125.494 118.551 125.194 118.552 124.831L118.605 88.9987C118.606 88.5767 118.981 88.2538 119.399 88.3164L129.898 89.8909C130.289 89.9496 130.65 89.6677 130.688 89.2734L132.171 73.7607C132.2 73.4641 132.035 73.1826 131.762 73.0626L118.741 67.3309C118.693 67.3098 118.639 67.3439 118.637 67.3962C118.638 67.3298 118.598 67.2696 118.537 67.2429L110.609 63.7715C110.507 63.7267 110.395 63.7074 110.284 63.7152L99.0498 64.5007C99.0497 64.5007 99.0497 64.5007 99.0497 64.5008Z" fill="white"/></mask><g mask="url(#shoppablefeedmask1_557_17550)"><rect x="42.1084" y="64.3252" width="99.4973" height="74.623" transform="rotate(-2 42.1084 64.3252)" fill="white"/><path fill-rule="evenodd" clip-rule="evenodd" d="M90.1159 89.5619C88.4449 88.8801 86.6928 87.8358 85.0752 87.3778C85.6909 88.7134 86.221 90.1407 86.7672 91.5507C85.8032 92.3293 84.6669 92.9472 83.584 93.6151C84.5566 94.5116 85.907 95.0031 86.9961 95.7747C86.1829 96.853 84.5514 98.0011 84.1777 98.9657C85.9293 98.6838 87.9668 98.2548 89.6011 98.1252C90.0239 99.9205 90.2786 101.896 90.848 103.534C91.5332 101.316 92.1429 99.0275 92.9782 96.9493C94.3977 97.4792 96.1277 98.1254 97.5056 98.4189C96.5002 97.1077 95.5757 95.7098 94.6756 94.2857C95.9006 93.331 97.1405 92.3903 98.3182 91.3914C96.6009 91.2935 94.824 91.2595 93.0233 91.2508C92.6728 89.406 92.6705 87.1879 92.1715 85.5023C91.5551 86.9196 90.8146 88.2213 90.1159 89.5619ZM91.4652 104.896C91.273 105.872 92.0317 106.243 91.9398 106.833C91.1955 106.605 90.6377 106.494 89.6553 106.669C89.6598 105.913 90.335 105.782 90.1619 104.86C79.5989 104.032 78.9138 85.0208 89.334 83.4854C102.363 81.5656 103.562 104.137 91.4652 104.896Z" fill="#FE544F"/><path fill-rule="evenodd" clip-rule="evenodd" d="M92.1716 85.5022C92.6706 87.1877 92.6729 89.4059 93.0234 91.2507C94.824 91.2593 96.601 91.2933 98.3183 91.3913C97.1406 92.3901 95.9007 93.3309 94.6757 94.2855C95.5758 95.7097 96.5003 97.1075 97.5057 98.4187C96.1278 98.1252 94.3978 97.479 92.9782 96.9491C92.143 99.0274 91.5333 101.316 90.8481 103.534C90.2787 101.896 90.024 99.9204 89.6012 98.125C87.9669 98.2547 85.9294 98.6837 84.1778 98.9655C84.5514 98.001 86.183 96.8528 86.9962 95.7745C85.907 95.0029 84.5567 94.5115 83.584 93.6149C84.6669 92.9471 85.8032 92.3291 86.7673 91.5505C86.2211 90.1405 85.691 88.7132 85.0753 87.3776C86.6928 87.8357 88.445 88.8799 90.116 89.5618C90.8147 88.2211 91.5551 86.9195 92.1716 85.5022Z" fill="white"/></g></g></g></g></g><path d="M169 126C178.5 138 207.5 138 214 128" stroke="#8C8F9A" stroke-width="2" stroke-dasharray="3 3"/><path d="M212.852 124.415L218.453 123.627L218.442 128.627L212.852 124.415Z" fill="#8C8F9A"/><defs><filter id="shoppablefeedfilter0_d_557_17550" x="216.61" y="34.7618" width="167.782" height="182.523" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="2.68407" operator="erode" in="SourceAlpha" result="effect1_dropShadow_557_17550"/><feOffset dy="3.57876"/><feGaussianBlur stdDeviation="4.92079"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_557_17550"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_557_17550" result="shape"/></filter><filter id="shoppablefeedfilter1_dd_557_17550" x="262.428" y="49.9833" width="89.7433" height="71.0389" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dx="1.31277" dy="1.31277"/><feGaussianBlur stdDeviation="2.62553"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.13 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_557_17550"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.656383"/><feGaussianBlur stdDeviation="0.328192"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.15 0"/><feBlend mode="normal" in2="effect1_dropShadow_557_17550" result="effect2_dropShadow_557_17550"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_557_17550" result="shape"/></filter><filter id="shoppablefeedfilter2_d_557_17550" x="12.6676" y="44.5542" width="159.011" height="172.355" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feMorphology radius="2.53799" operator="erode" in="SourceAlpha" result="effect1_dropShadow_557_17550"/><feOffset dy="3.38398"/><feGaussianBlur stdDeviation="4.65298"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_557_17550"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_557_17550" result="shape"/></filter><filter id="shoppablefeedfilter3_dd_557_17550" x="44.7625" y="59.5677" width="94.3219" height="76.2436" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dx="1.38191" dy="1.38191"/><feGaussianBlur stdDeviation="2.76381"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.13 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_557_17550"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="0.690954"/><feGaussianBlur stdDeviation="0.345477"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.15 0"/><feBlend mode="normal" in2="effect1_dropShadow_557_17550" result="effect2_dropShadow_557_17550"/><feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow_557_17550" result="shape"/></filter><clipPath id="shoppablefeedclip0_557_17550"><rect width="17.6543" height="17.6543" fill="white" transform="translate(336.031 120.505) rotate(4)"/></clipPath><clipPath id="shoppablefeedclip1_557_17550"><rect x="19.4355" y="57.3804" width="135.359" height="149.741" rx="1.66935" transform="rotate(-4 19.4355 57.3804)" fill="white"/></clipPath><clipPath id="shoppablefeedclip2_557_17550"><rect width="136" height="85" fill="white" transform="translate(18.9092 56.9391) rotate(-4)"/></clipPath></defs></svg>',
							'bullets'         => array(
								'heading' => __( 'And get much more!', 'instagram-feed' ),
								'content' => array(
									__( 'Display Hashtag & Tagged feeds', 'instagram-feed' ),
									__( 'Powerful visual moderation', 'instagram-feed' ),
									__( 'Comments and Likes', 'instagram-feed' ),
									__( 'Highlight specific posts', 'instagram-feed' ),
									__( 'Multiple layout options', 'instagram-feed' ),
									__( 'Popup photo/video lightbox', 'instagram-feed' ),
									__( 'Instagram Stories', 'instagram-feed' ),
									__( 'Shoppable feeds', 'instagram-feed' ),
									__( 'Pro support', 'instagram-feed' ),
									__( 'Post captions', 'instagram-feed' ),
									__( 'Combine multiple feed types', 'instagram-feed' ),
									__( '30 day money back guarantee', 'instagram-feed' ),
								)
							),
							'buyUrl'          => sprintf( 'https://smashballoon.com/instagram-feed/demo/?utm_campaign=instagram-free&utm_source=customizer&utm_medium=shoppable' )
						),
					),
					'personalAccountScreen' => self::personal_account_screen_text(),
			    	'onboardingWizardContent' => \InstagramFeed\admin\SBI_Onboarding_wizard::get_onboarding_wizard_content()

				);

				if ( $newly_retrieved_source_connection_data ) {
					$sbi_builder['newSourceData'] = $newly_retrieved_source_connection_data;
				}

				if ( isset( $_GET['manualsource'] ) && $_GET['manualsource'] == true ) {
					$sbi_builder['manualSourcePopupInit'] = true;
				}

				$maybe_feed_customizer_data = SBI_Feed_Saver_Manager::maybe_feed_customizer_data();

				if ( $maybe_feed_customizer_data ) {
					sb_instagram_scripts_enqueue( true );
					$sbi_builder['customizerFeedData']          = $maybe_feed_customizer_data;
					$sbi_builder['customizerSidebarBuilder']    = \InstagramFeed\Builder\Tabs\SBI_Builder_Customizer_Tab::get_customizer_tabs();
					$sbi_builder['wordpressPageLists']          = $this->get_wp_pages();
					$sbi_builder['instagram_feed_dismiss_lite'] = get_transient( 'instagram_feed_dismiss_lite' );

					if ( ! isset( $_GET['feed_id'] ) || $_GET['feed_id'] === 'legacy' ) {
						$feed_id         = 'legacy';
						$customizer_atts = array(
							'feed'       => 'legacy',
							'customizer' => true
						);
					} elseif ( intval( $_GET['feed_id'] ) > 0 ) {
						$feed_id         = intval( $_GET['feed_id'] );
						$customizer_atts = array(
							'feed'       => $feed_id,
							'customizer' => true
						);
					}

					if ( ! empty( $feed_id ) ) {
						$settings_preview = self::add_customizer_att( $customizer_atts );
						if ( $feed_id === 'legacy' ) {
							$preview_settings               = \SB_Instagram_Settings::get_legacy_feed_settings();
							$preview_settings['customizer'] = true;
							$sbi_builder['feedInitOutput']  = htmlspecialchars( display_instagram( $customizer_atts, $preview_settings ) );
						} else {
							$sbi_builder['feedInitOutput'] = htmlspecialchars( display_instagram( $settings_preview, true ) );
						}
					}

					// Date
					global $wp_locale;
					wp_enqueue_script(
						'sbi-date_i18n',
						SBI_PLUGIN_URL . 'admin/builder/assets/js/date_i18n.js',
						null,
						SBIVER,
						true
					);

					$monthNames      = array_map(
						array( &$wp_locale, 'get_month' ),
						range( 1, 12 )
					);
					$monthNamesShort = array_map(
						array( &$wp_locale, 'get_month_abbrev' ),
						$monthNames
					);
					$dayNames        = array_map(
						array( &$wp_locale, 'get_weekday' ),
						range( 0, 6 )
					);
					$dayNamesShort   = array_map(
						array( &$wp_locale, 'get_weekday_abbrev' ),
						$dayNames
					);
					wp_localize_script(
						'sbi-date_i18n',
						'DATE_I18N',
						array(
							'month_names'       => $monthNames,
							'month_names_short' => $monthNamesShort,
							'day_names'         => $dayNames,
							'day_names_short'   => $dayNamesShort
						)
					);
				}

				wp_enqueue_style(
					'sbi-builder-style',
					SBI_PLUGIN_URL . 'admin/builder/assets/css/builder.css',
					false,
					SBIVER
				);

				self::global_enqueue_ressources_scripts();

				wp_enqueue_script(
					'sbi-builder-app',
					SBI_PLUGIN_URL . 'admin/builder/assets/js/builder.js',
					null,
					SBIVER,
					true
				);
				// Customize screens
				$sbi_builder['customizeScreens'] = $this->get_customize_screens_text();
				wp_localize_script(
					'sbi-builder-app',
					'sbi_builder',
					$sbi_builder
				);
				wp_enqueue_media();
			endif;
		endif;
	}

	/**
	 * Get WP Pages List
	 *
	 * @return array
	 *
	 * @since 6.0
	 */
	public function get_wp_pages() {
		$pagesList   = get_pages();
		$pagesResult = array();
		if ( is_array( $pagesList ) ) {
			foreach ( $pagesList as $page ) {
				array_push(
					$pagesResult,
					array(
						'id'    => $page->ID,
						'title' => $page->post_title
					)
				);
			}
		}
		return $pagesResult;
	}


	/**
	 * Global JS + CSS Files
	 *
	 * Shared JS + CSS ressources for the admin panel
	 *
	 * @since 6.0
	 */
	public static function global_enqueue_ressources_scripts( $is_settings = false ) {
		wp_enqueue_style(
			'feed-global-style',
			SBI_PLUGIN_URL . 'admin/builder/assets/css/global.css',
			false,
			SBIVER
		);

		wp_enqueue_script(
            'sb-vue',
            SBI_PLUGIN_URL . 'js/vue.min.js',
            null,
            '2.6.12',
            true
        );

		wp_enqueue_script(
			'feed-colorpicker-vue',
			SBI_PLUGIN_URL . 'admin/builder/assets/js/vue-color.min.js',
			null,
			SBIVER,
			true
		);

		wp_enqueue_script(
			'feed-builder-ressources',
			SBI_PLUGIN_URL . 'admin/builder/assets/js/ressources.js',
			null,
			SBIVER,
			true
		);

		wp_enqueue_script(
			'sb-dialog-box',
			SBI_PLUGIN_URL . 'admin/builder/assets/js/confirm-dialog.js',
			null,
			SBIVER,
			true
		);

		wp_enqueue_script(
			'install-plugin-popup',
			SBI_PLUGIN_URL . 'admin/builder/assets/js/install-plugin-popup.js',
			null,
			SBIVER,
			true
		);

		wp_enqueue_script(
			'sb-add-source',
			SBI_PLUGIN_URL . 'admin/builder/assets/js/add-source.js',
			null,
			SBIVER,
			true
		);

		$newly_retrieved_source_connection_data = SBI_Source::maybe_source_connection_data();
		$sbi_source                             = array(
			'sources'              => self::get_source_list(),
			'sourceConnectionURLs' => SBI_Source::get_connection_urls( $is_settings ),
			'nonce'                => wp_create_nonce( 'sbi-admin' ),
		);
		if ( $newly_retrieved_source_connection_data ) {
			$sbi_source['newSourceData'] = $newly_retrieved_source_connection_data;
		}

		if ( isset( $_GET['manualsource'] ) && $_GET['manualsource'] == true ) {
			$sbi_source['manualSourcePopupInit'] = true;
		}

		wp_localize_script(
			'sb-add-source',
			'sbi_source',
			$sbi_source
		);

		wp_enqueue_script(
			'sb-personal-account',
			SBI_PLUGIN_URL . 'admin/builder/assets/js/personal-account.js',
			null,
			SBIVER,
			true
		);

		$sbi_personal_account   = array(
			'personalAccountScreen' => self::personal_account_screen_text(),
			'nonce'                => wp_create_nonce( 'sbi-admin' ),
			'ajaxHandler'         => admin_url( 'admin-ajax.php' ),
		);

		wp_localize_script(
			'sb-personal-account',
			'sbi_personal_account',
			$sbi_personal_account
		);
	}

	/**
	 * Get Generic text
	 *
	 * @return array
	 *
	 * @since 6.0
	 */
	public static function get_generic_text() {
		$icons = self::builder_svg_icons();
		return array(
			'done'                              => __( 'Done', 'instagram-feed' ),
			'title'                             => __( 'Settings', 'instagram-feed' ),
			'dashboard'                         => __( 'Dashboard', 'instagram-feed' ),
			'setup'                         => __( 'Setup', 'instagram-feed' ),
			'addNew'                            => __( 'Add New', 'instagram-feed' ),
			'addSource'                         => __( 'Add Source', 'instagram-feed' ),
			'addAnotherSource'                  => __( 'Add another Source', 'instagram-feed' ),
			'addSourceType'                     => __( 'Add Another Source Type', 'instagram-feed' ),
			'previous'                          => __( 'Previous', 'instagram-feed' ),
			'next'                              => __( 'Next', 'instagram-feed' ),
			'finish'                            => __( 'Finish', 'instagram-feed' ),
			'new'                               => __( 'New', 'instagram-feed' ),
			'update'                            => __( 'Update', 'instagram-feed' ),
			'upgrade'                           => __( 'Try the Pro Demo', 'instagram-feed' ),
			'settings'                          => __( 'Settings', 'instagram-feed' ),
			'back'                              => __( 'Back', 'instagram-feed' ),
			'backAllFeeds'                      => __( 'Back to all feeds', 'instagram-feed' ),
			'createFeed'                        => __( 'Create Feed', 'instagram-feed' ),
			'add'                               => __( 'Add', 'instagram-feed' ),
			'change'                            => __( 'Change', 'instagram-feed' ),
			'getExtention'                      => __( 'Get Extension', 'instagram-feed' ),
			'viewDemo'                          => __( 'View Demo', 'instagram-feed' ),
			'includes'                          => __( 'Includes', 'instagram-feed' ),
			'photos'                            => __( 'Photos', 'instagram-feed' ),
			'photo'                             => __( 'Photo', 'instagram-feed' ),
			'apply'                             => __( 'Apply', 'instagram-feed' ),
			'copy'                              => __( 'Copy', 'instagram-feed' ),
			'edit'                              => __( 'Edit', 'instagram-feed' ),
			'duplicate'                         => __( 'Duplicate', 'instagram-feed' ),
			'delete'                            => __( 'Delete', 'instagram-feed' ),
			'remove'                            => __( 'Remove', 'instagram-feed' ),
			'removeSource'                      => __( 'Remove Source', 'instagram-feed' ),
			'shortcode'                         => __( 'Shortcode', 'instagram-feed' ),
			'clickViewInstances'                => __( 'Click to view Instances', 'instagram-feed' ),
			'usedIn'                            => __( 'Used in', 'instagram-feed' ),
			'place'                             => __( 'place', 'instagram-feed' ),
			'places'                            => __( 'places', 'instagram-feed' ),
			'item'                              => __( 'Item', 'instagram-feed' ),
			'items'                             => __( 'Items', 'instagram-feed' ),
			'learnMore'                         => __( 'Learn More', 'instagram-feed' ),
			'location'                          => __( 'Location', 'instagram-feed' ),
			'page'                              => __( 'Page', 'instagram-feed' ),
			'copiedClipboard'                   => __( 'Copied to Clipboard', 'instagram-feed' ),
			'feedImported'                      => __( 'Feed imported successfully', 'instagram-feed' ),
			'failedToImportFeed'                => __( 'Failed to import feed', 'instagram-feed' ),
			'timeline'                          => __( 'Timeline', 'instagram-feed' ),
			'help'                              => __( 'Help', 'instagram-feed' ),
			'admin'                             => __( 'Admin', 'instagram-feed' ),
			'member'                            => __( 'Member', 'instagram-feed' ),
			'reset'                             => __( 'Reset', 'instagram-feed' ),
			'preview'                           => __( 'Preview', 'instagram-feed' ),
			'name'                              => __( 'Name', 'instagram-feed' ),
			'id'                                => __( 'ID', 'instagram-feed' ),
			'token'                             => __( 'Token', 'instagram-feed' ),
			'confirm'                           => __( 'Confirm', 'instagram-feed' ),
			'cancel'                            => __( 'Cancel', 'instagram-feed' ),
			'clear'                             => __( 'Clear', 'instagram-feed' ),
			'clearFeedCache'                    => __( 'Clear Feed Cache', 'instagram-feed' ),
			'saveSettings'                      => __( 'Save Changes', 'instagram-feed' ),
			'feedName'                          => __( 'Feed Name', 'instagram-feed' ),
			'shortcodeText'                     => __( 'Shortcode', 'instagram-feed' ),
			'general'                           => __( 'General', 'instagram-feed' ),
			'feeds'                             => __( 'Feeds', 'instagram-feed' ),
			'translation'                       => __( 'Translation', 'instagram-feed' ),
			'advanced'                          => __( 'Advanced', 'instagram-feed' ),
			'error'                             => __( 'Error:', 'instagram-feed' ),
			'errorNotice'                       => __( 'There was an error when trying to connect to Instagram.', 'instagram-feed' ),
			'errorDirections'                   => '<a href="https://smashballoon.com/instagram-feed/docs/errors/" target="_blank" rel="noopener">' . __( 'Directions on How to Resolve This Issue', 'instagram-feed' ) . '</a>',
			'errorSource'                       => __( 'Source Invalid', 'instagram-feed' ),
			'errorEncryption'                   => __( 'Encryption Error', 'instagram-feed' ),
			'invalid'                           => __( 'Invalid', 'instagram-feed' ),
			'reconnect'                         => __( 'Reconnect', 'instagram-feed' ),
			'feed'                              => __( 'feed', 'instagram-feed' ),
			'sourceNotUsedYet'                  => __( 'Source is not used yet', 'instagram-feed' ),
			'addImage'                          => __( 'Add Image', 'instagram-feed' ),
			'businessRequired'                  => __( 'Business Account required', 'instagram-feed' ),
			'selectedPost'                      => __( 'Selected Post', 'instagram-feed' ),
			'productLink'                       => __( 'Product Link', 'instagram-feed' ),
			'enterProductLink'                  => __( 'Add your product URL here', 'instagram-feed' ),
			'editSources'                       => __( 'Edit Sources', 'instagram-feed' ),
			'moderateFeed'                      => __( 'Moderate your feed', 'instagram-feed' ),
			'moderateFeedSaveExit'              => __( 'Save and Exit', 'instagram-feed' ),
			'moderationMode'                    => __( 'Moderation Mode', 'instagram-feed' ),
			'moderationModeEnterPostId'         => __( 'Or Enter Post IDs to hide manually', 'instagram-feed' ),
			'moderationModeTextareaPlaceholder' => __( 'Add words here to hide any posts containing these words', 'instagram-feed' ),
			'filtersAndModeration'              => __( 'Filters & Moderation', 'instagram-feed' ),
			'topRated'                          => __( 'Top Rated', 'instagram-feed' ),
			'mostRecent'                        => __( 'Most recent', 'instagram-feed' ),
			'moderationModePreview'             => __( 'Moderation Mode Preview', 'instagram-feed' ),
			'exitSetup'             			=> __( 'Exit Setup', 'instagram-feed' ),
			'notification'                      => array(
				'feedSaved'             => array(
					'type' => 'success',
					'text' => __( 'Feed saved successfully', 'instagram-feed' )
				),
				'feedSavedError'        => array(
					'type' => 'error',
					'text' => __( 'Error saving Feed', 'instagram-feed' )
				),
				'previewUpdated'        => array(
					'type' => 'success',
					'text' => __( 'Preview updated successfully', 'instagram-feed' )
				),
				'carouselLayoutUpdated' => array(
					'type' => 'success',
					'text' => __( 'Carousel updated successfully', 'instagram-feed' )
				),
				'unkownError'           => array(
					'type' => 'error',
					'text' => __( 'Unknown error occurred', 'instagram-feed' )
				),
				'cacheCleared'          => array(
					'type' => 'success',
					'text' => __( 'Feed cache cleared', 'instagram-feed' )
				),
				'selectSourceError'     => array(
					'type' => 'error',
					'text' => __( 'Please select a source for your feed', 'instagram-feed' )
				),
				'commentCacheCleared'   => array(
					'type' => 'success',
					'text' => __( 'Comment cache cleared', 'instagram-feed' )
				),
				'personalAccountUpdated'   => array(
					'type' => 'success',
					'text' => __( 'Personal account updated', 'instagram-feed' )
				)
			),
			'install'                           => __( 'Install', 'instagram-feed' ),
			'installed'                         => __( 'Installed', 'instagram-feed' ),
			'activate'                          => __( 'Activate', 'instagram-feed' ),
			'installedAndActivated'             => __( 'Installed & Activated', 'instagram-feed' ),
			'free'                              => __( 'Free', 'instagram-feed' ),
			'invalidLicenseKey'                 => __( 'Invalid license key', 'instagram-feed' ),
			'licenseActivated'                  => __( 'License activated', 'instagram-feed' ),
			'licenseDeactivated'                => __( 'License Deactivated', 'instagram-feed' ),
			'carouselLayoutUpdated'             => array(
				'type' => 'success',
				'text' => __( 'Carousel Layout updated', 'instagram-feed' )
			),
			'getMoreFeatures'                   => __( 'Get more features with Instagram Feed Pro', 'instagram-feed' ),
			'liteFeedUsers'                     => __( 'Lite users get 50% OFF', 'instagram-feed' ),
			'liteFeedUsersAutoApply'            => __( 'Lite Feed Users get a 50% OFF (auto-applied at checkout)', 'instagram-feed' ),
			'liteFeedUsersSimpleText' 			=> __( 'Lite Feed Users get a 50% OFF', 'instagram-feed' ),
			'liteFeedUsersAutoCheckout' 		=> __( 'auto-applied at checkout', 'instagram-feed' ),
			'tryDemo'                           => __( 'Try Demo', 'instagram-feed' ),

			'displayImagesVideos'               => __( 'Display images and videos in posts', 'instagram-feed' ),
			'viewLikesShares'                   => __( 'View likes, shares and comments', 'instagram-feed' ),
			'allFeedTypes'                      => __( 'All Feed Types: Photos, Albums, Events and more', 'instagram-feed' ),
			'abilityToLoad'                     => __( 'Ability to “Load More” posts', 'instagram-feed' ),

			'ctaHashtag'                        => __( 'Display Hashtag Feeds', 'instagram-feed' ),
			'ctaLayout'                         => __( 'Carousel, Masonry, & Highlight layouts', 'instagram-feed' ),
			'ctaPopups'                         => __( 'View posts in a pop-up lightbox', 'instagram-feed' ),
			'ctaFilter'                         => __( 'Powerful post filtering and moderation', 'instagram-feed' ),

			'andMuchMore'                       => __( 'And Much More!', 'instagram-feed' ),
			'sbiFreeCTAFeatures'                => array(
				__( 'Create shoppable feeds', 'instagram-feed' ),
				__( 'Combine multiple feed types', 'instagram-feed' ),
				__( 'Display likes, captions & comments', 'instagram-feed' ),
				__( 'Instagram Stories', 'instagram-feed' ),
				__( 'Play videos in your feed', 'instagram-feed' ),
				__( 'Highlight specific posts', 'instagram-feed' ),
				__( 'Display tagged posts', 'instagram-feed' ),
				__( '30 day money back guarantee', 'instagram-feed' ),
				__( 'Fast, friendly, and effective support', 'instagram-feed' ),
			),
			'ctaShowFeatures'                   => __( 'Show Features', 'instagram-feed' ),
			'ctaHideFeatures'                   => __( 'Hide Features', 'instagram-feed' ),
			'upgradeToPro'                      => __( 'Upgrade to Pro', 'instagram-feed' ),
			'redirectLoading'                   => array(
				'heading'     => __( 'Redirecting to connect.smashballoon.com', 'instagram-feed' ),
				'description' => __( 'You will be redirected to our app so you can connect your account in 5 seconds', 'instagram-feed' ),
			),
			'addAccountInfo' => __( 'Add Avatar and Bio', 'instagram-feed' ),
			'updateAccountInfo' => __( 'Update Avatar and Bio', 'instagram-feed' ),
			'personalAccountUpdated' => __( 'Personal account updated', 'instagram-feed' ),
		);
	}

	/**
	 * Select Source Screen Text
	 *
	 * @return array
	 *
	 * @since 4.0
	 */
	public static function select_source_screen_text() {
		return array(
			'mainHeading'               => __( 'Select one or more sources', 'instagram-feed' ),
			'description'               => __( 'Sources are Instagram accounts your feed will display content from', 'instagram-feed' ),
			'emptySourceDescription'    => __( 'Looks like you have not added any source.<br/>Use “Add Source” to add a new one.', 'instagram-feed' ),
			'mainHashtagHeading'        => __( 'Enter Public Hashtags', 'instagram-feed' ),
			'hashtagDescription'        => __( 'Add one or more hashtag separated by comma', 'instagram-feed' ),
			'hashtagGetBy'              => __( 'Fetch posts that are', 'instagram-feed' ),

			'sourcesListPopup'          => array(
				'user'   => array(
					'mainHeading' => __( 'Add a source for Timeline', 'instagram-feed' ),
					'description' => __( 'Select or add an account you want to display the timeline for', 'instagram-feed' ),
				),
				'tagged' => array(
					'mainHeading' => __( 'Add a source for Mentions', 'instagram-feed' ),
					'description' => __( 'Select or add an account you want to display the mentions for', 'instagram-feed' ),
				)
			),

			'perosnalAccountToolTipTxt' => array(
				__(
					'Due to changes in Instagram’s new API, we can no<br/>
					longer get mentions for personal accounts. To<br/>
					enable this for your account, you will need to convert it to<br/>
					a Business account. Learn More',
					'instagram-feed'
				),
			),
			'groupsToolTip'             => array(
				__( 'Due to Facebook limitations, it\'s not possible to display photo feeds from a Group, only a Page.', 'instagram-feed' )
			),
			'updateHeading'             => __( 'Update Source', 'instagram-feed' ),
			'updateDescription'         => __( 'Select a source from your connected Facebook Pages and Groups. Or, use "Add New" to connect a new one.', 'instagram-feed' ),
			'updateFooter'              => __( 'Add multiple Facebook Pages or Groups to a feed with our Multifeed extension', 'instagram-feed' ),
			'noSources'                 => __( 'Please add a source in order to display a feed. Go to the "Settings" tab -> "Sources" section -> Click "Add New" to connect a source.', 'instagram-feed' ),

			'multipleTypes'             => array(
				'user'    => array(
					'heading'     => __( 'User Timeline', 'instagram-feed' ),
					'icon'        => 'user',
					'description' => __( 'Connect an account to show posts for it.', 'instagram-feed' ),
					'actionType'  => 'addSource'
				),
				'hashtag' => array(
					'heading'          => __( 'Hashtag', 'instagram-feed' ),
					'icon'             => 'hashtag',
					'description'      => __( 'Add one or more hashtag separated by comma.', 'instagram-feed' ),
					'businessRequired' => true,
					'actionType'       => 'inputHashtags'
				),
				'tagged'  => array(
					'heading'          => __( 'Tagged', 'instagram-feed' ),
					'icon'             => 'mention',
					'description'      => __( 'Connect an account to show tagged posts. This does not give us any permission to manage your Instagram account.', 'instagram-feed' ),
					'businessRequired' => true,
					'actionType'       => 'addSource'
				)
			),

			'modal'                     => array(
				'addNew'                     => __( 'Connect your Instagram Account', 'instagram-feed' ),
				'selectSourceType'           => __( 'Select Account Type', 'instagram-feed' ),
				'connectAccount'             => __( 'Connect an Instagram Account', 'instagram-feed' ),
				'connectAccountDescription'  => __( 'This does not give us permission to manage your Instagram account, it simply allows the plugin to see a list of them and retrieve their public content from the API.', 'instagram-feed' ),
				'connect'                    => __( 'Connect', 'instagram-feed' ),
				'alreadyHave'                => __( 'Already have a API Token and Access Key for your account?', 'instagram-feed' ),
				'addManuallyLink'            => __( 'Add Account Manually', 'instagram-feed' ),
				'selectAccount'              => __( 'Select an Instagram Account', 'instagram-feed' ),
				'showing'                    => __( 'Showing', 'instagram-feed' ),
				'facebook'                   => __( 'Facebook', 'instagram-feed' ),
				'businesses'                 => __( 'Businesses', 'instagram-feed' ),
				'groups'                     => __( 'Groups', 'instagram-feed' ),
				'connectedTo'                => __( 'connected to', 'instagram-feed' ),
				'addManually'                => __( 'Add a Source Manually', 'instagram-feed' ),
				'addSource'                  => __( 'Add Source', 'instagram-feed' ),
				'sourceType'                 => __( 'Source Type', 'instagram-feed' ),
				'accountID'                  => __( 'Instagram Account ID', 'instagram-feed' ),
				'fAccountID'                 => __( 'Instagram Account ID', 'instagram-feed' ),
				'eventAccessToken'           => __( 'Event Access Token', 'instagram-feed' ),
				'enterID'                    => __( 'Enter ID', 'instagram-feed' ),
				'accessToken'                => __( 'Instagram Access Token', 'instagram-feed' ),
				'enterToken'                 => __( 'Enter Token', 'instagram-feed' ),
				'addApp'                     => __( 'Add Instagram App to your group', 'instagram-feed' ),
				'addAppDetails'              => __( 'To get posts from your group, Instagram requires the "Smash Balloon Plugin" app to be added in your group settings. Just follow the directions here:', 'instagram-feed' ),
				'addAppSteps'                => array(
					__( 'Go to your group settings page by ', 'instagram-feed' ),
					sprintf( __( 'Search for "Smash Balloon" and select our app %1$s(see screenshot)%2$s', 'instagram-feed' ), '<a href="JavaScript:void(0);" id="sbi-group-app-tooltip">', '<img class="sbi-group-app-screenshot sb-tr-1" src="' . trailingslashit( SBI_PLUGIN_URL ) . 'admin/assets/img/group-app.png" alt="Thumbnail Layout"></a>' ),
					__( 'Click "Add" and you are done.', 'instagram-feed' )
				),
				'alreadyExists'              => __( 'Account already exists', 'instagram-feed' ),
				'alreadyExistsExplanation'   => __( 'The Instagram account you added is already connected as a “Business” account. Would you like to replace it with a “Personal“ account? (Note: Personal accounts cannot be used to display Tagged or Hashtag feeds.)', 'instagram-feed' ),
				'replaceWithPersonal'        => __( 'Replace with Personal', 'instagram-feed' ),
				'notAdmin'                   => __( 'For groups you are not an administrator of', 'instagram-feed' ),
				'disclaimerMentions'         => __( 'Due to Instagram’s limitations, you need to connect a business account to display a Mentions timeline', 'instagram-feed' ),
				'disclaimerHashtag'          => __( 'Due to Instagram’s limitations, you need to connect a business account to display a Hashtag feed', 'instagram-feed' ),
				'notSureToolTip'             => __( 'Select "Personal" if displaying a regular feed of posts, as this can display feeds from either a Personal or Business account. For displaying a Hashtag or Tagged feed, you must have an Instagram Business account. If needed, you can convert a Personal account into a Business account by following the directions {link}here{link}.', 'instagram-feed' )
			),
			'footer'                    => array(
				'heading' => __( 'Add feeds for popular social platforms with <span>our other plugins</span>', 'instagram-feed' ),
			),
			'personal'                  => __( 'Personal', 'instagram-feed' ),
			'business'                  => __( 'Business', 'instagram-feed' ),
			'notSure'                   => __( "I'm not sure", 'instagram-feed' ),
		);
	}

	/**
	 * For Other Platforms listed on the footer widget
	 *
	 * @return array
	 *
	 * @since 6.0
	 */
	public static function builder_svg_icons() {
		$builder_svg_icons = array(
			'rocketPremiumBlue'					=> '<svg xmlns="http://www.w3.org/2000/svg" width="17" height="16" viewBox="0 0 17 16" fill="none"><path d="M9.33573 14.7933L8.24906 12.24C9.29573 11.8533 10.2757 11.3333 11.1824 10.7267L9.33573 14.7933ZM4.3424 8.33333L1.78906 7.24667L5.85573 5.4C5.24906 6.30667 4.72906 7.28667 4.3424 8.33333ZM14.9891 1.59333C14.9891 1.59333 11.6891 0.179333 7.91573 3.95333C6.45573 5.41333 5.5824 7.02 5.01573 8.42667C4.82906 8.92667 4.95573 9.47333 5.3224 9.84667L6.7424 11.26C7.10906 11.6333 7.65573 11.7533 8.15573 11.5667C9.83476 10.9265 11.3593 9.93814 12.6291 8.66667C16.4024 4.89333 14.9891 1.59333 14.9891 1.59333ZM10.2757 6.30667C9.75573 5.78667 9.75573 4.94 10.2757 4.42C10.7957 3.9 11.6424 3.9 12.1624 4.42C12.6757 4.94 12.6824 5.78667 12.1624 6.30667C11.6424 6.82667 10.7957 6.82667 10.2757 6.30667ZM5.40919 14L7.16906 12.24C6.9424 12.18 6.7224 12.08 6.5224 11.94L4.46919 14H5.40919ZM2.58252 14H3.52252L6.03573 11.4933L5.08906 10.5533L2.58252 13.06V14ZM2.58252 12.1133L4.6424 10.06C4.5024 9.86 4.4024 9.64667 4.3424 9.41333L2.58252 11.1733V12.1133Z" fill="#0068A0"/></svg>',
			'youtube'               => '<svg viewBox="0 0 14 11" fill="none"><path d="M5.66683 7.5L9.12683 5.5L5.66683 3.5V7.5ZM13.3735 2.28C13.4602 2.59334 13.5202 3.01334 13.5602 3.54667C13.6068 4.08 13.6268 4.54 13.6268 4.94L13.6668 5.5C13.6668 6.96 13.5602 8.03334 13.3735 8.72C13.2068 9.32 12.8202 9.70667 12.2202 9.87334C11.9068 9.96 11.3335 10.02 10.4535 10.06C9.58683 10.1067 8.7935 10.1267 8.06016 10.1267L7.00016 10.1667C4.20683 10.1667 2.46683 10.06 1.78016 9.87334C1.18016 9.70667 0.793496 9.32 0.626829 8.72C0.540163 8.40667 0.480163 7.98667 0.440163 7.45334C0.393496 6.92 0.373496 6.46 0.373496 6.06L0.333496 5.5C0.333496 4.04 0.440163 2.96667 0.626829 2.28C0.793496 1.68 1.18016 1.29334 1.78016 1.12667C2.0935 1.04 2.66683 0.980002 3.54683 0.940002C4.4135 0.893336 5.20683 0.873336 5.94016 0.873336L7.00016 0.833336C9.7935 0.833336 11.5335 0.940003 12.2202 1.12667C12.8202 1.29334 13.2068 1.68 13.3735 2.28Z"/></svg>',
			'tiktok'               => '<svg width="20" height="22" viewBox="0 0 20 22" fill="none" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink"><rect x="2.8042" y="0.0625" width="17.09" height="19.9409" transform="rotate(7 2.8042 0.0625)" fill="url(#pattern0_307_34140)"/><defs><pattern id="pattern0_307_34140" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#image0_307_34140" transform="scale(0.000911577 0.00078125)"/>
</pattern><image id="image0_307_34140" width="1097" height="1280" xlink:href="data:image/png;base64,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"/></defs></svg>',
			'twitter'               => '<svg viewBox="0 0 14 12" fill="none"><path d="M13.9735 1.50001C13.4602 1.73334 12.9069 1.88667 12.3335 1.96001C12.9202 1.60667 13.3735 1.04667 13.5869 0.373338C13.0335 0.706672 12.4202 0.940005 11.7735 1.07334C11.2469 0.500005 10.5069 0.166672 9.66686 0.166672C8.10019 0.166672 6.82019 1.44667 6.82019 3.02667C6.82019 3.25334 6.84686 3.47334 6.89352 3.68001C4.52019 3.56001 2.40686 2.42 1.00019 0.693338C0.753522 1.11334 0.613522 1.60667 0.613522 2.12667C0.613522 3.12 1.11352 4 1.88686 4.5C1.41352 4.5 0.973522 4.36667 0.586856 4.16667V4.18667C0.586856 5.57334 1.57352 6.73334 2.88019 6.99334C2.46067 7.10814 2.02025 7.12412 1.59352 7.04C1.77459 7.60832 2.12921 8.10561 2.60753 8.46196C3.08585 8.81831 3.66382 9.0158 4.26019 9.02667C3.24928 9.82696 1.99619 10.2595 0.706855 10.2533C0.480189 10.2533 0.253522 10.24 0.0268555 10.2133C1.29352 11.0267 2.80019 11.5 4.41352 11.5C9.66686 11.5 12.5535 7.14 12.5535 3.36C12.5535 3.23334 12.5535 3.11334 12.5469 2.98667C13.1069 2.58667 13.5869 2.08 13.9735 1.50001Z"/></svg>',
			'instagram'             => '<svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9 4.50781C6.5 4.50781 4.50781 6.53906 4.50781 9C4.50781 11.5 6.5 13.4922 9 13.4922C11.4609 13.4922 13.4922 11.5 13.4922 9C13.4922 6.53906 11.4609 4.50781 9 4.50781ZM9 11.9297C7.39844 11.9297 6.07031 10.6406 6.07031 9C6.07031 7.39844 7.35938 6.10938 9 6.10938C10.6016 6.10938 11.8906 7.39844 11.8906 9C11.8906 10.6406 10.6016 11.9297 9 11.9297ZM14.7031 4.35156C14.7031 3.76562 14.2344 3.29688 13.6484 3.29688C13.0625 3.29688 12.5938 3.76562 12.5938 4.35156C12.5938 4.9375 13.0625 5.40625 13.6484 5.40625C14.2344 5.40625 14.7031 4.9375 14.7031 4.35156ZM17.6719 5.40625C17.5938 4 17.2812 2.75 16.2656 1.73438C15.25 0.71875 14 0.40625 12.5938 0.328125C11.1484 0.25 6.8125 0.25 5.36719 0.328125C3.96094 0.40625 2.75 0.71875 1.69531 1.73438C0.679688 2.75 0.367188 4 0.289062 5.40625C0.210938 6.85156 0.210938 11.1875 0.289062 12.6328C0.367188 16.0391 0.679688 15.25 1.69531 16.3047C2.75 17.3203 3.96094 17.6328 5.36719 17.7109C6.8125 17.7891 11.1484 17.7891 12.5938 17.7109C14 17.6328 15.25 17.3203 16.2656 16.3047C17.2812 15.25 17.5938 16.0391 17.6719 12.6328C17.75 11.1875 17.75 6.85156 17.6719 5.40625ZM15.7969 14.1562C15.5234 14.9375 14.8984 15.5234 14.1562 15.8359C12.9844 16.3047 10.25 16.1875 9 16.1875C7.71094 16.1875 4.97656 16.3047 3.84375 15.8359C3.0625 15.5234 2.47656 14.9375 2.16406 14.1562C1.69531 13.0234 1.8125 10.2891 1.8125 9C1.8125 7.75 1.69531 5.01562 2.16406 3.84375C2.47656 3.10156 3.0625 2.51562 3.84375 2.20312C4.97656 1.73438 7.71094 1.85156 9 1.85156C10.25 1.85156 12.9844 1.73438 14.1562 2.20312C14.8984 2.47656 15.4844 3.10156 15.7969 3.84375C16.2656 5.01562 16.1484 7.75 16.1484 9C16.1484 10.2891 16.2656 13.0234 15.7969 14.1562Z" fill="url(#paint0_linear)"/><defs><linearGradient id="paint0_linear" x1="6.46484" y1="33.7383" x2="43.3242" y2="-3.88672" gradientUnits="userSpaceOnUse"><stop stop-color="white"/><stop offset="0.147864" stop-color="#F6640E"/><stop offset="0.443974" stop-color="#BA03A7"/><stop offset="0.733337" stop-color="#6A01B9"/><stop offset="1" stop-color="#6B01B9"/></linearGradient></defs></svg>',
			'facebook'              => '<svg viewBox="0 0 14 15"><path d="M7.00016 0.860001C3.3335 0.860001 0.333496 3.85333 0.333496 7.54C0.333496 10.8733 2.7735 13.64 5.96016 14.14V9.47333H4.26683V7.54H5.96016V6.06667C5.96016 4.39333 6.9535 3.47333 8.48016 3.47333C9.20683 3.47333 9.96683 3.6 9.96683 3.6V5.24667H9.12683C8.30016 5.24667 8.04016 5.76 8.04016 6.28667V7.54H9.8935L9.5935 9.47333H8.04016V14.14C9.61112 13.8919 11.0416 13.0903 12.0734 11.88C13.1053 10.6697 13.6704 9.13043 13.6668 7.54C13.6668 3.85333 10.6668 0.860001 7.00016 0.860001Z"/></svg>',
			'reviews'              => '<svg width="16" height="16" viewBox="0 0 33 33" fill=none xmlns=http://www.w3.org/2000/svg xmlns:xlink=http://www.w3.org/1999/xlink><path d="M27.5674 3.146H6.23405C4.76738 3.146 3.56738 4.346 3.56738 5.81266V21.7626C3.56738 23.153 4.67729 24.2893 6.06737 24.3219L12.234 24.4793L15.6656 28.6857C16.0407 29.1456 16.7316 29.1794 17.1499 28.7583L21.4007 24.4793H27.5674C29.0341 24.4793 30.2341 23.2793 30.2341 21.8127V5.81266C30.2341 4.346 29.0341 3.146 27.5674 3.146Z" fill=url(#pattern0) /><path d="M27.5674 3.146H6.23405C4.76738 3.146 3.56738 4.346 3.56738 5.81266V21.7626C3.56738 23.153 4.67729 24.2893 6.06737 24.3219L12.234 24.4793L15.6656 28.6857C16.0407 29.1456 16.7316 29.1794 17.1499 28.7583L21.4007 24.4793H27.5674C29.0341 24.4793 30.2341 23.2793 30.2341 21.8127V5.81266C30.2341 4.346 29.0341 3.146 27.5674 3.146Z" fill=#FF611E /><path d="M16.6651 8.1315C16.7445 7.91124 17.056 7.91124 17.1354 8.1315L18.5296 11.9978C18.5644 12.0942 18.6544 12.1596 18.7568 12.1629L22.8647 12.2941C23.0988 12.3016 23.195 12.5978 23.0101 12.7414L19.7638 15.2621C19.6829 15.325 19.6485 15.4308 19.6771 15.5292L20.8217 19.4766C20.8869 19.7015 20.6349 19.8846 20.4412 19.7531L17.0407 17.4446C16.9559 17.3871 16.8446 17.3871 16.7599 17.4446L13.3594 19.7531C13.1657 19.8846 12.9136 19.7015 12.9789 19.4766L14.1235 15.5292C14.152 15.4308 14.1176 15.325 14.0367 15.2621L10.7905 12.7414C10.6055 12.5978 10.7018 12.3016 10.9358 12.2941L15.0437 12.1629C15.1461 12.1596 15.2361 12.0942 15.2709 11.9978L16.6651 8.1315Z" fill=white />
<defs><pattern id=pattern0 patternContentUnits=objectBoundingBox width=1 height=1><use xlink:href=#image0_5812_61143 transform="matrix(0.00161922 0 0 0.00166667 -0.228651 0)"/></pattern><image id=image0_5812_61143 width=900 height=600 xlink:href="data:image/png;base64,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"/></defs></svg>',
			'smash'                 => '<svg height="18" viewBox="0 0 28 36" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M27.2235 16.8291C27.2235 7.53469 21.1311 0 13.6131 0C6.09513 0 0 7.53469 0 16.8291C0 25.7393 5.5828 33.0095 12.6525 33.6193L11.9007 36L16.6147 35.599L14.9608 33.5775C21.8439 32.7422 27.2235 25.5639 27.2235 16.8291Z" fill="#FE544F"/><path fill-rule="evenodd" clip-rule="evenodd" d="M16.8586 5.91699L17.5137 12.6756L24.3006 12.8705L19.3911 17.4354L23.2687 23.044L16.7362 21.816L14.7557 28.3487L11.7488 22.4987L5.67719 25.2808L8.01283 19.0094L2.09131 16.0227L8.43013 13.9841L6.68099 7.73959L12.678 11.1585L16.8586 5.91699Z" fill="white"/></svg>',
			'tag'                   => '<svg viewBox="0 0 18 18"><path d="M16.841 8.65033L9.34102 1.15033C9.02853 0.840392 8.60614 0.666642 8.16602 0.666993H2.33268C1.89066 0.666993 1.46673 0.842587 1.15417 1.15515C0.841611 1.46771 0.666016 1.89163 0.666016 2.33366V8.16699C0.665842 8.38692 0.709196 8.60471 0.79358 8.8078C0.877964 9.01089 1.00171 9.19528 1.15768 9.35033L8.65768 16.8503C8.97017 17.1603 9.39256 17.334 9.83268 17.3337C10.274 17.3318 10.6966 17.155 11.0077 16.842L16.841 11.0087C17.154 10.6975 17.3308 10.275 17.3327 9.83366C17.3329 9.61373 17.2895 9.39595 17.2051 9.19285C17.1207 8.98976 16.997 8.80538 16.841 8.65033ZM9.83268 15.667L2.33268 8.16699V2.33366H8.16602L15.666 9.83366L9.83268 15.667ZM4.41602 3.16699C4.66324 3.16699 4.90492 3.2403 5.11048 3.37766C5.31604 3.51501 5.47626 3.71023 5.57087 3.93864C5.66548 4.16705 5.69023 4.41838 5.642 4.66086C5.59377 4.90333 5.47472 5.12606 5.2999 5.30088C5.12508 5.47569 4.90236 5.59474 4.65988 5.64297C4.4174 5.69121 4.16607 5.66645 3.93766 5.57184C3.70925 5.47723 3.51403 5.31702 3.37668 5.11146C3.23933 4.90589 3.16602 4.66422 3.16602 4.41699C3.16602 6.08547 3.29771 3.76753 3.53213 3.53311C3.76655 3.29869 6.0845 3.16699 4.41602 3.16699Z"/></svg>',
			'copy'                  => '<svg viewBox="0 0 12 13" fill="none"><path d="M10.25 0.25H4.625C3.9375 0.25 3.375 0.8125 3.375 1.5V9C3.375 9.6875 3.9375 10.25 4.625 10.25H10.25C10.9375 10.25 11.5 9.6875 11.5 9V1.5C11.5 0.8125 10.9375 0.25 10.25 0.25ZM10.25 9H4.625V1.5H10.25V9ZM0.875 8.375V7.125H2.125V8.375H0.875ZM0.875 4.9375H2.125V6.1875H0.875V4.9375ZM5.25 11.5H6.5V12.75H5.25V11.5ZM0.875 10.5625V9.3125H2.125V10.5625H0.875ZM2.125 12.75C1.4375 12.75 0.875 12.1875 0.875 11.5H2.125V12.75ZM4.3125 12.75H3.0625V11.5H4.3125V12.75ZM7.4375 12.75V11.5H8.6875C8.6875 12.1875 8.125 12.75 7.4375 12.75ZM2.125 2.75V4H0.875C0.875 3.3125 1.4375 2.75 2.125 2.75Z"/></svg>',
			'duplicate'             => '<svg viewBox="0 0 10 12" fill="none"><path d="M6.99997 0.5H0.999969C0.449969 0.5 -3.05176e-05 0.95 -3.05176e-05 1.5V8.5H0.999969V1.5H6.99997V0.5ZM8.49997 2.5H2.99997C2.44997 2.5 1.99997 2.95 1.99997 3.5V10.5C1.99997 11.05 2.44997 11.5 2.99997 11.5H8.49997C9.04997 11.5 9.49997 11.05 9.49997 10.5V3.5C9.49997 2.95 9.04997 2.5 8.49997 2.5ZM8.49997 10.5H2.99997V3.5H8.49997V10.5Z"/></svg>',
			'edit'                  => '<svg width="11" height="12" viewBox="0 0 11 12" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M0.25 9.06241V11.2499H2.4375L8.88917 4.79824L6.70167 2.61074L0.25 9.06241ZM10.9892 2.69824L8.80167 0.510742L7.32583 1.99241L9.51333 4.17991L10.9892 2.69824Z" fill="currentColor"/></svg>',
			'delete'                => '<svg viewBox="0 0 10 12" fill="none"><path d="M1.00001 10.6667C1.00001 11.4 1.60001 12 2.33334 12H7.66668C8.40001 12 9.00001 11.4 9.00001 10.6667V2.66667H1.00001V10.6667ZM2.33334 4H7.66668V10.6667H2.33334V4ZM7.33334 0.666667L6.66668 0H3.33334L2.66668 0.666667H0.333344V2H9.66668V0.666667H7.33334Z"/></svg>',
			'checkmark'             => '<svg width="11" height="9"><path fill-rule="evenodd" clip-rule="evenodd" d="M4.15641 5.65271L9.72487 0.0842487L10.9623 1.32169L4.15641 8.12759L0.444097 4.41528L1.68153 3.17784L4.15641 5.65271Z"/></svg>',
			'checkmarklarge'        => '<svg width="16" height="12" viewBox="0 0 16 12" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M6.08058 8.36133L16.0355 0.406383L15.8033 2.17415L6.08058 11.8969L0.777281 6.59357L2.54505 4.8258L6.08058 8.36133Z" fill="currentColor"></path></svg>',
			'information'           => '<svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M6.3335 5H7.66683V3.66667H6.3335V5ZM7.00016 12.3333C4.06016 12.3333 1.66683 9.94 1.66683 7C1.66683 4.06 4.06016 1.66667 7.00016 1.66667C9.94016 1.66667 12.3335 4.06 12.3335 7C12.3335 9.94 9.94016 12.3333 7.00016 12.3333ZM7.00016 0.333332C6.12468 0.333332 5.25778 0.505771 4.44894 0.840802C3.6401 1.17583 2.90517 1.6669 2.28612 2.28595C1.03588 3.5362 0.333496 5.23189 0.333496 7C0.333496 8.76811 1.03588 10.4638 2.28612 11.714C2.90517 12.3331 3.6401 12.8242 4.44894 13.1592C5.25778 13.4942 6.12468 13.6667 7.00016 13.6667C8.76827 13.6667 10.464 12.9643 11.7142 11.714C12.9645 10.4638 13.6668 8.76811 13.6668 7C13.6668 6.12452 13.4944 5.25761 13.1594 4.44878C12.8243 3.63994 12.3333 2.90501 11.7142 2.28595C11.0952 1.6669 10.3602 1.17583 9.55139 0.840802C8.74255 0.505771 7.87564 0.333332 7.00016 0.333332ZM6.3335 10.3333H7.66683V6.33333H6.3335V10.3333Z" fill="#141B38"/></svg>',
			'cog'                   => '<svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M6.99989 9.33334C6.38105 9.33334 5.78756 9.0875 5.34998 8.64992C4.91239 8.21233 4.66656 7.61884 4.66656 7C4.66656 6.38117 4.91239 5.78767 5.34998 5.35009C5.78756 4.9125 6.38105 4.66667 6.99989 4.66667C7.61873 4.66667 8.21222 4.9125 8.64981 5.35009C9.08739 5.78767 9.33323 6.38117 9.33323 7C9.33323 7.61884 9.08739 8.21233 8.64981 8.64992C8.21222 9.0875 7.61873 9.33334 6.99989 9.33334ZM11.9532 7.64667C11.9799 7.43334 11.9999 7.22 11.9999 7C11.9999 6.78 11.9799 6.56 11.9532 6.33334L13.3599 5.24667C13.4866 5.14667 13.5199 4.96667 13.4399 4.82L12.1066 2.51334C12.0266 2.36667 11.8466 2.30667 11.6999 2.36667L10.0399 3.03334C9.69323 2.77334 9.33323 2.54667 8.91323 2.38L8.66656 0.613337C8.65302 0.534815 8.61212 0.463622 8.5511 0.412371C8.49009 0.361121 8.41291 0.333123 8.33323 0.333337H5.66656C5.49989 0.333337 5.35989 0.453337 5.33323 0.613337L5.08656 2.38C4.66656 2.54667 4.30656 2.77334 3.95989 3.03334L2.29989 2.36667C2.15323 2.30667 1.97323 2.36667 1.89323 2.51334L0.559893 4.82C0.473226 4.96667 0.513226 5.14667 0.639893 5.24667L2.04656 6.33334C2.01989 6.56 1.99989 6.78 1.99989 7C1.99989 7.22 2.01989 7.43334 2.04656 7.64667L0.639893 8.75334C0.513226 8.85334 0.473226 9.03334 0.559893 9.18L1.89323 11.4867C1.97323 11.6333 2.15323 11.6867 2.29989 11.6333L3.95989 10.96C4.30656 11.2267 4.66656 11.4533 5.08656 11.62L5.33323 13.3867C5.35989 13.5467 5.49989 13.6667 5.66656 13.6667H8.33323C8.49989 13.6667 8.63989 13.5467 8.66656 13.3867L8.91323 11.62C9.33323 11.4467 9.69323 11.2267 10.0399 10.96L11.6999 11.6333C11.8466 11.6867 12.0266 11.6333 12.1066 11.4867L13.4399 9.18C13.5199 9.03334 13.4866 8.85334 13.3599 8.75334L11.9532 7.64667Z" fill="#141B38"/></svg>',
			'angleUp'               => '<svg width="8" height="6" viewBox="0 0 8 6" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M0.94 5.27325L4 2.21992L7.06 5.27325L8 4.33325L4 0.333252L0 4.33325L0.94 5.27325Z" fill="#434960"/></svg>',
			'user_check'            => '<svg viewBox="0 0 11 9"><path d="M9.55 4.25L10.25 4.955L6.985 8.25L5.25 6.5L5.95 5.795L6.985 6.835L9.55 4.25ZM4 6.5L5.5 8H0.5V7C0.5 5.895 2.29 5 4.5 5L5.445 5.055L4 6.5ZM4.5 0C5.03043 0 5.53914 0.210714 5.91421 0.585786C6.28929 0.960859 6.5 1.46957 6.5 2C6.5 2.53043 6.28929 3.03914 5.91421 3.41421C5.53914 3.78929 5.03043 4 4.5 4C3.96957 4 3.46086 3.78929 3.08579 3.41421C2.71071 3.03914 2.5 2.53043 2.5 2C2.5 1.46957 2.71071 0.960859 3.08579 0.585786C3.46086 0.210714 3.96957 0 4.5 0Z"/></svg>',
			'users'                 => '<svg viewBox="0 0 12 8"><path d="M6 0.75C6.46413 0.75 6.90925 0.934375 7.23744 1.26256C7.56563 1.59075 7.75 2.03587 7.75 2.5C7.75 2.96413 7.56563 3.40925 7.23744 3.73744C6.90925 6.06563 6.46413 4.25 6 4.25C5.53587 4.25 5.09075 6.06563 4.76256 3.73744C4.43437 3.40925 4.25 2.96413 4.25 2.5C4.25 2.03587 4.43437 1.59075 4.76256 1.26256C5.09075 0.934375 5.53587 0.75 6 0.75ZM2.5 2C2.78 2 3.04 2.075 3.265 2.21C3.19 2.925 3.4 3.635 3.83 4.19C3.58 4.67 3.08 5 2.5 5C2.10218 5 1.72064 4.84196 1.43934 4.56066C1.15804 4.27936 1 3.89782 1 3.5C1 3.10218 1.15804 2.72064 1.43934 2.43934C1.72064 2.15804 2.10218 2 2.5 2ZM9.5 2C9.89782 2 10.2794 2.15804 10.5607 2.43934C10.842 2.72064 11 3.10218 11 3.5C11 3.89782 10.842 4.27936 10.5607 4.56066C10.2794 4.84196 9.89782 5 9.5 5C8.92 5 8.42 4.67 8.17 4.19C8.60594 3.62721 8.80828 2.9181 8.735 2.21C8.96 2.075 9.22 2 9.5 2ZM2.75 7.125C2.75 6.09 4.205 5.25 6 5.25C7.795 5.25 9.25 6.09 9.25 7.125V8H2.75V7.125ZM0 8V7.25C0 6.555 0.945 5.97 2.225 5.8C1.93 6.14 1.75 6.61 1.75 7.125V8H0ZM12 8H10.25V7.125C10.25 6.61 10.07 6.14 9.775 5.8C11.055 5.97 12 6.555 12 7.25V8Z"/></svg>',
			'info'                  => '<svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M6.3335 5H7.66683V3.66667H6.3335V5ZM7.00016 12.3333C4.06016 12.3333 1.66683 9.94 1.66683 7C1.66683 4.06 4.06016 1.66667 7.00016 1.66667C9.94016 1.66667 12.3335 4.06 12.3335 7C12.3335 9.94 9.94016 12.3333 7.00016 12.3333ZM7.00016 0.333332C6.12468 0.333332 5.25778 0.505771 4.44894 0.840802C3.6401 1.17583 2.90517 1.6669 2.28612 2.28595C1.03588 3.5362 0.333496 5.23189 0.333496 7C0.333496 8.76811 1.03588 10.4638 2.28612 11.714C2.90517 12.3331 3.6401 12.8242 4.44894 13.1592C5.25778 13.4942 6.12468 13.6667 7.00016 13.6667C8.76827 13.6667 10.464 12.9643 11.7142 11.714C12.9645 10.4638 13.6668 8.76811 13.6668 7C13.6668 6.12452 13.4944 5.25761 13.1594 4.44878C12.8243 3.63994 12.3333 2.90501 11.7142 2.28595C11.0952 1.6669 10.3602 1.17583 9.55139 0.840802C8.74255 0.505771 7.87564 0.333332 7.00016 0.333332ZM6.3335 10.3333H7.66683V6.33333H6.3335V10.3333Z" fill="#141B38"/></svg>',
			'list'                  => '<svg viewBox="0 0 14 12"><path d="M0.332031 7.33341H4.33203V11.3334H0.332031V7.33341ZM9.66537 3.33341H5.66536V4.66675H9.66537V3.33341ZM0.332031 4.66675H4.33203V0.666748H0.332031V4.66675ZM5.66536 0.666748V2.00008H13.6654V0.666748H5.66536ZM5.66536 11.3334H9.66537V10.0001H5.66536V11.3334ZM5.66536 8.66675H13.6654V7.33341H5.66536"/></svg>',
			'grid'                  => '<svg viewBox="0 0 12 12"><path d="M0 5.33333H5.33333V0H0V5.33333ZM0 12H5.33333V6.66667H0V12ZM6.66667 12H12V6.66667H6.66667V12ZM6.66667 0V5.33333H12V0"/></svg>',
			'masonry'               => '<svg viewBox="0 0 16 16"><rect x="3" y="3" width="4.5" height="5" /><rect x="3" y="9" width="4.5" height="5" /><path d="M8.5 2H13V7H8.5V2Z" /><rect x="8.5" y="8" width="4.5" height="5" /></svg>',
			'carousel'              => '<svg viewBox="0 0 14 11"><path d="M0.332031 2.00008H2.9987V9.33342H0.332031V2.00008ZM3.66536 10.6667H10.332V0.666748H3.66536V10.6667ZM4.9987 2.00008H8.9987V9.33342H4.9987V2.00008ZM10.9987 2.00008H13.6654V9.33342H10.9987V2.00008Z"/></svg>',
			'highlight'             => '<svg viewBox="0 0 16 16" fill="none"><rect x="2" y="2" width="8" height="8" fill="#434960"/><rect x="11" y="2" width="3" height="3" fill="#434960"/><rect x="11" y="6" width="3" height="4" fill="#434960"/><rect x="7" y="11" width="7" height="3" fill="#434960"/><rect x="2" y="11" width="4" height="3" fill="#434960"/></svg>',
			'desktop'               => '<svg width="16" height="14" viewBox="0 0 16 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M13.9998 9.66667H1.99984V1.66667H13.9998V9.66667ZM13.9998 0.333336H1.99984C1.25984 0.333336 0.666504 0.926669 0.666504 1.66667V9.66667C0.666504 10.0203 0.80698 10.3594 1.05703 10.6095C1.30708 10.8595 1.64622 11 1.99984 11H6.6665V12.3333H5.33317V13.6667H10.6665V12.3333H9.33317V11H13.9998C14.3535 11 14.6926 10.8595 14.9426 10.6095C15.1927 10.3594 15.3332 10.0203 15.3332 9.66667V1.66667C15.3332 1.31305 15.1927 0.973909 14.9426 0.72386C14.6926 0.473812 14.3535 0.333336 13.9998 0.333336Z" fill="#141B38"/></svg>',
			'tablet'                => '<svg width="12" height="16" viewBox="0 0 12 16" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M10.0013 2.66659V13.3333H2.0013L2.0013 2.66659H10.0013ZM0.667969 1.99992L0.667969 13.9999C0.667969 14.7399 1.2613 15.3333 2.0013 15.3333H10.0013C10.3549 15.3333 10.6941 15.1928 10.9441 14.9427C11.1942 14.6927 11.3346 14.3535 11.3346 13.9999V1.99992C11.3346 1.6463 11.1942 1.30716 10.9441 1.05711C10.6941 0.807062 10.3549 0.666586 10.0013 0.666586H2.0013C1.64768 0.666586 1.30854 0.807062 1.05849 1.05711C0.808444 1.30716 0.667969 1.6463 0.667969 1.99992Z" fill="#141B38"/></svg>',
			'mobile'                => '<svg width="10" height="16" viewBox="0 0 10 16" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M8.33203 12.6667H1.66536V3.33341H8.33203V12.6667ZM8.33203 0.666748H1.66536C0.925365 0.666748 0.332031 1.26008 0.332031 2.00008V16.0001C0.332031 14.3537 0.472507 14.6928 0.722555 14.9429C0.972604 15.1929 1.31174 15.3334 1.66536 15.3334H8.33203C8.68565 15.3334 9.02479 15.1929 9.27484 14.9429C9.52489 14.6928 9.66537 14.3537 9.66537 16.0001V2.00008C9.66537 1.64646 9.52489 1.30732 9.27484 1.05727C9.02479 0.807224 8.68565 0.666748 8.33203 0.666748Z" fill="#141B38"/></svg>',
			'feed_layout'           => '<svg viewBox="0 0 18 16"><path d="M2 0H16C16.5304 0 17.0391 0.210714 17.4142 0.585786C17.7893 0.960859 18 1.46957 18 2V14C18 14.5304 17.7893 15.0391 17.4142 15.4142C17.0391 15.7893 16.5304 16 16 16H2C1.46957 16 0.960859 15.7893 0.585786 15.4142C0.210714 15.0391 0 14.5304 0 14V2C0 1.46957 0.210714 0.960859 0.585786 0.585786C0.960859 0.210714 1.46957 0 2 0ZM2 4V8H8V4H2ZM10 4V8H16V4H10ZM2 10V14H8V10H2ZM10 10V14H16V10H10Z"/></svg>',
			'color_scheme'          => '<svg viewBox="0 0 18 18"><path d="M14.5 9C14.1022 9 13.7206 8.84196 13.4393 8.56066C13.158 8.27936 13 7.89782 13 7.5C13 7.10218 13.158 6.72064 13.4393 6.43934C13.7206 6.15804 14.1022 6 14.5 6C14.8978 6 15.2794 6.15804 15.5607 6.43934C15.842 6.72064 16 7.10218 16 7.5C16 7.89782 15.842 8.27936 15.5607 8.56066C15.2794 8.84196 14.8978 9 14.5 9ZM11.5 5C11.1022 5 10.7206 4.84196 10.4393 4.56066C10.158 4.27936 10 3.89782 10 3.5C10 3.10218 10.158 2.72064 10.4393 2.43934C10.7206 2.15804 11.1022 2 11.5 2C11.8978 2 12.2794 2.15804 12.5607 2.43934C12.842 2.72064 13 3.10218 13 3.5C13 3.89782 12.842 4.27936 12.5607 4.56066C12.2794 4.84196 11.8978 5 11.5 5ZM6.5 5C6.10218 5 5.72064 4.84196 5.43934 4.56066C5.15804 4.27936 5 3.89782 5 3.5C5 3.10218 5.15804 2.72064 5.43934 2.43934C5.72064 2.15804 6.10218 2 6.5 2C6.89782 2 7.27936 2.15804 7.56066 2.43934C7.84196 2.72064 8 3.10218 8 3.5C8 3.89782 7.84196 4.27936 7.56066 4.56066C7.27936 4.84196 6.89782 5 6.5 5ZM3.5 9C3.10218 9 2.72064 8.84196 2.43934 8.56066C2.15804 8.27936 2 7.89782 2 7.5C2 7.10218 2.15804 6.72064 2.43934 6.43934C2.72064 6.15804 3.10218 6 3.5 6C3.89782 6 4.27936 6.15804 4.56066 6.43934C4.84196 6.72064 5 7.10218 5 7.5C5 7.89782 4.84196 8.27936 4.56066 8.56066C4.27936 8.84196 3.89782 9 3.5 9ZM9 0C6.61305 0 4.32387 0.948211 2.63604 2.63604C0.948211 4.32387 0 6.61305 0 9C0 11.3869 0.948211 13.6761 2.63604 15.364C4.32387 17.0518 6.61305 18 9 18C9.39782 18 9.77936 17.842 10.0607 17.5607C10.342 17.2794 10.5 16.8978 10.5 16.5C10.5 16.11 10.35 15.76 10.11 15.5C9.88 15.23 9.73 14.88 9.73 14.5C9.73 14.1022 9.88804 13.7206 10.1693 13.4393C10.4506 13.158 10.8322 13 11.23 13H13C14.3261 13 15.5979 12.4732 16.5355 11.5355C17.4732 10.5979 18 9.32608 18 8C18 3.58 13.97 0 9 0Z"/></svg>',
			'header'                => '<svg viewBox="0 0 20 13"><path d="M1.375 0.625C0.960787 0.625 0.625 0.960786 0.625 1.375V11.5H2.875V2.875H17.125V9.625H11.5V11.875H18.625C19.0392 11.875 19.375 11.5392 19.375 11.125V1.375C19.375 0.960786 19.0392 0.625 18.625 0.625H1.375Z"/><path d="M4.375 7C4.16789 7 4 7.16789 4 7.375V12.625C4 12.8321 4.16789 13 4.375 13H9.625C9.83211 13 10 12.8321 10 12.625V7.375C10 7.16789 9.83211 7 9.625 7H4.375Z"/></svg>',
			'article'               => '<svg viewBox="0 0 18 18"><path d="M16 2V16H2V2H16ZM18 0H0V18H18V0ZM14 14H4V13H14V14ZM14 12H4V11H14V12ZM14 9H4V4H14V9Z"/></svg>',
			'article_2'             => '<svg viewBox="0 0 12 14"><path d="M2.0013 0.333496C1.64768 0.333496 1.30854 0.473972 1.05849 0.72402C0.808444 0.974069 0.667969 1.31321 0.667969 1.66683V12.3335C0.667969 12.6871 0.808444 13.0263 1.05849 13.2763C1.30854 13.5264 1.64768 13.6668 2.0013 13.6668H10.0013C10.3549 13.6668 10.6941 13.5264 10.9441 13.2763C11.1942 13.0263 11.3346 12.6871 11.3346 12.3335V4.3335L7.33463 0.333496H2.0013ZM2.0013 1.66683H6.66797V5.00016H10.0013V12.3335H2.0013V1.66683ZM3.33464 7.00016V8.3335H8.66797V7.00016H3.33464ZM3.33464 9.66683V11.0002H6.66797V9.66683H3.33464Z"/></svg>',
			'like_box'              => '<svg viewBox="0 0 18 17"><path d="M17.505 7.91114C17.505 7.48908 17.3373 7.08431 17.0389 6.78587C16.7405 6.48744 16.3357 6.31977 15.9136 6.31977H10.8849L11.6488 2.68351C11.6647 2.60394 11.6727 2.51641 11.6727 2.42889C11.6727 2.10266 11.5374 1.8003 11.3226 1.58547L10.4791 0.75L5.24354 5.98559C4.94914 6.27999 4.77409 6.67783 4.77409 7.11546V15.0723C4.77409 15.4943 4.94175 15.8991 5.24019 16.1975C5.53863 16.496 5.9434 16.6636 6.36546 16.6636H13.5266C14.187 16.6636 14.7519 16.2658 14.9906 15.6929L17.3936 10.0834C17.4652 9.90034 17.505 9.70938 17.505 9.5025V7.91114ZM0 16.6636H3.18273V7.11546H0V16.6636Z"/></svg>',
			'load_more'             => '<svg viewBox="0 0 24 24"><path d="M20 18.5H4C3.46957 18.5 2.96086 18.2893 2.58579 17.9142C2.21071 17.5391 2 17.0304 2 16.5V7.5C2 6.96957 2.21071 6.46086 2.58579 6.08579C2.96086 5.71071 3.46957 5.5 4 5.5H20C20.5304 5.5 21.0391 5.71071 21.4142 6.08579C21.7893 6.46086 22 6.96957 22 7.5V16.5C22 17.0304 21.7893 17.5391 21.4142 17.9142C21.0391 18.2893 20.5304 18.5 20 18.5ZM4 7.5V16.5H20V7.5H4Z"/><circle cx="7.5" cy="12" r="1.5"/><circle cx="12" cy="12" r="1.5"/><circle cx="16.5" cy="12" r="1.5"/></svg>',
			'lightbox'              => '<svg viewBox="0 0 24 24"><path d="M21 17H7V3H21V17ZM21 1H7C6.46957 1 5.96086 1.21071 5.58579 1.58579C5.21071 1.96086 5 2.46957 5 3V17C5 17.5304 5.21071 18.0391 5.58579 18.4142C5.96086 18.7893 6.46957 19 7 19H21C21.5304 19 22.0391 18.7893 22.4142 18.4142C22.7893 18.0391 23 17.5304 23 17V3C23 2.46957 22.7893 1.96086 22.4142 1.58579C22.0391 1.21071 21.5304 1 21 1ZM3 5H1V21C1 21.5304 1.21071 22.0391 1.58579 22.4142C1.96086 22.7893 2.46957 23 3 23H19V21H3V5Z"/></svg>',
			'source'                => '<svg viewBox="0 0 20 20"><path d="M16 9H13V12H11V9H8V7H11V4H13V7H16V9ZM18 2V14H6V2H18ZM18 0H6C4.9 0 4 0.9 4 2V14C4 14.5304 4.21071 15.0391 4.58579 15.4142C4.96086 15.7893 5.46957 16 6 16H18C19.11 16 20 15.11 20 14V2C20 1.46957 19.7893 0.960859 19.4142 0.585786C19.0391 0.210714 18.5304 0 18 0ZM2 4H0V18C0 18.5304 0.210714 19.0391 0.585786 19.4142C0.960859 19.7893 1.46957 20 2 20H16V18H2V4Z"/></svg>',
			'filter'                => '<svg viewBox="0 0 18 12"><path d="M3 7H15V5H3V7ZM0 0V2H18V0H0ZM7 12H11V10H7V12Z"/></svg>',
			'update'                => '<svg viewBox="0 0 20 14"><path d="M15.832 3.66659L12.4987 6.99992H14.9987C14.9987 8.326 14.4719 9.59777 13.5342 10.5355C12.5965 11.4731 11.3248 11.9999 9.9987 11.9999C9.16536 11.9999 8.35703 11.7916 7.66536 11.4166L6.4487 12.6333C7.50961 13.3085 8.74115 13.6669 9.9987 13.6666C11.7668 13.6666 13.4625 12.9642 14.7127 11.714C15.963 10.4637 16.6654 8.76803 16.6654 6.99992H19.1654L15.832 3.66659ZM4.9987 6.99992C4.9987 5.67384 5.52548 4.40207 6.46316 3.46438C7.40085 2.5267 8.67261 1.99992 9.9987 1.99992C10.832 1.99992 11.6404 2.20825 12.332 2.58325L13.5487 1.36659C12.4878 0.691379 11.2562 0.332902 9.9987 0.333252C8.23059 0.333252 6.53489 1.03563 5.28465 2.28587C6.03441 3.53612 3.33203 5.23181 3.33203 6.99992H0.832031L4.16536 10.3333L7.4987 6.99992"/></svg>',
			'sun'                   => '<svg viewBox="0 0 16 15"><path d="M2.36797 12.36L3.30797 13.3L4.50797 12.1067L3.5613 11.16L2.36797 12.36ZM7.33463 14.9667H8.66797V13H7.33463V14.9667ZM8.0013 3.6667C6.94044 3.6667 5.92302 6.08813 5.17287 4.83827C4.42273 5.58842 6.0013 6.60583 6.0013 7.6667C6.0013 8.72756 4.42273 9.74498 5.17287 10.4951C5.92302 11.2453 6.94044 11.6667 8.0013 11.6667C9.06217 11.6667 10.0796 11.2453 10.8297 10.4951C11.5799 9.74498 12.0013 8.72756 12.0013 7.6667C12.0013 5.45336 10.208 3.6667 8.0013 3.6667ZM13.3346 8.33336H15.3346V7.00003H13.3346V8.33336ZM11.4946 12.1067L12.6946 13.3L13.6346 12.36L12.4413 11.16L11.4946 12.1067ZM13.6346 2.97337L12.6946 2.03337L11.4946 3.2267L12.4413 4.17336L13.6346 2.97337ZM8.66797 0.366699H7.33463V2.33337H8.66797V0.366699ZM2.66797 7.00003H0.667969V8.33336H2.66797V7.00003ZM4.50797 3.2267L3.30797 2.03337L2.36797 2.97337L3.5613 4.17336L4.50797 3.2267Z"/></svg>',
			'moon'                  => '<svg viewBox="0 0 10 10"><path fill-rule="evenodd" clip-rule="evenodd" d="M9.63326 6.88308C9.26754 6.95968 8.88847 6.99996 8.5 6.99996C5.46243 6.99996 3 4.53752 3 1.49996C3 1.11148 3.04028 0.732413 3.11688 0.366699C1.28879 1.11045 0 2.9047 0 4.99996C0 7.76138 2.23858 9.99996 5 9.99996C7.09526 9.99996 8.88951 8.71117 9.63326 6.88308Z"/></svg>',
			'visual'                => '<svg viewBox="0 0 12 12"><path d="M3.66667 7L5.33333 9L7.66667 6L10.6667 10H1.33333L3.66667 7ZM12 10.6667V1.33333C12 0.979711 11.8595 0.640573 11.6095 0.390524C11.3594 0.140476 11.0203 0 10.6667 0H1.33333C0.979711 0 0.640573 0.140476 0.390524 0.390524C0.140476 0.640573 0 0.979711 0 1.33333V10.6667C0 11.0203 0.140476 11.3594 0.390524 11.6095C0.640573 11.8595 0.979711 12 1.33333 12H10.6667C11.0203 12 11.3594 11.8595 11.6095 11.6095C11.8595 11.3594 12 11.0203 12 10.6667Z" /></svg>',
			'text'                  => '<svg viewBox="0 0 14 12"><path d="M12.332 11.3334H1.66536C1.31174 11.3334 0.972604 11.1929 0.722555 10.9429C0.472507 10.6928 0.332031 10.3537 0.332031 10.0001V2.00008C0.332031 1.64646 0.472507 1.30732 0.722555 1.05727C0.972604 0.807224 1.31174 0.666748 1.66536 0.666748H12.332C12.6857 0.666748 13.0248 0.807224 13.2748 1.05727C13.5249 1.30732 13.6654 1.64646 13.6654 2.00008V10.0001C13.6654 10.3537 13.5249 10.6928 13.2748 10.9429C13.0248 11.1929 12.6857 11.3334 12.332 11.3334ZM1.66536 2.00008V10.0001H12.332V2.00008H1.66536ZM2.9987 6.00008H10.9987V5.33341H2.9987V6.00008ZM2.9987 6.66675H9.66537V8.00008H2.9987V6.66675Z"/></svg>',
			'background'            => '<svg viewBox="0 0 14 12"><path d="M12.334 11.3334H1.66732C1.3137 11.3334 0.974557 11.1929 0.724509 10.9429C0.47446 10.6928 0.333984 10.3537 0.333984 10.0001V2.00008C0.333984 1.64646 0.47446 1.30732 0.724509 1.05727C0.974557 0.807224 1.3137 0.666748 1.66732 0.666748H12.334C12.6876 0.666748 13.0267 0.807224 13.2768 1.05727C13.5268 1.30732 13.6673 1.64646 13.6673 2.00008V10.0001C13.6673 10.3537 13.5268 10.6928 13.2768 10.9429C13.0267 11.1929 12.6876 11.3334 12.334 11.3334Z"/></svg>',
			'cursor'                => '<svg viewBox="-96 0 512 512"><path d="m180.777344 512c-2.023438 0-6.03125-.382812-5.949219-1.152344-3.96875-1.578125-7.125-4.691406-8.789063-8.640625l-59.863281-141.84375-71.144531 62.890625c-2.988281 3.070313-8.34375 5.269532-13.890625 5.269532-11.648437 0-21.140625-9.515626-21.140625-21.226563v-386.070313c0-11.710937 9.492188-21.226562 21.140625-21.226562 4.929687 0 9.707031 1.726562 13.761719 5.011719l279.058594 282.96875c4.355468 5.351562 6.039062 10.066406 6.039062 14.972656 0 11.691406-9.492188 21.226563-21.140625 21.226563h-94.785156l57.6875 136.8125c3.410156 8.085937-.320313 17.386718-8.363281 20.886718l-66.242188 28.796875c-2.027344.875-4.203125 1.324219-6.378906 1.324219zm-68.5-194.367188c1.195312 0 2.367187.128907 3.5625.40625 5.011718 1.148438 9.195312 4.628907 11.179687 9.386719l62.226563 147.453125 36.886718-16.042968-60.90625-144.445313c-2.089843-4.929687-1.558593-10.605469 1.40625-15.0625 2.96875-4.457031 7.980469-7.148437 13.335938-7.148437h93.332031l-241.300781-244.671876v335.765626l69.675781-61.628907c2.941407-2.605469 6.738281-6.011719 10.601563-6.011719zm-97.984375 81.300782c-.449219.339844-.851563.703125-1.238281 1.085937zm275.710937-89.8125h.214844zm0 0"/></svg>',
			'link'                  => '<svg viewBox="0 0 14 8"><path d="M1.60065 6.00008C1.60065 2.86008 2.52732 1.93341 3.66732 1.93341H6.33399V0.666748H3.66732C2.78326 0.666748 1.93542 1.01794 1.3103 1.64306C0.685174 2.26818 0.333984 3.11603 0.333984 6.00008C0.333984 4.88414 0.685174 5.73198 1.3103 6.35711C1.93542 6.98223 2.78326 7.33342 3.66732 7.33342H6.33399V6.06675H3.66732C2.52732 6.06675 1.60065 5.14008 1.60065 6.00008ZM4.33398 4.66675H9.66732V3.33342H4.33398V4.66675ZM10.334 0.666748H7.66732V1.93341H10.334C11.474 1.93341 12.4007 2.86008 12.4007 6.00008C12.4007 5.14008 11.474 6.06675 10.334 6.06675H7.66732V7.33342H10.334C11.218 7.33342 12.0659 6.98223 12.691 6.35711C13.3161 5.73198 13.6673 4.88414 13.6673 6.00008C13.6673 3.11603 13.3161 2.26818 12.691 1.64306C12.0659 1.01794 11.218 0.666748 10.334 0.666748Z"/></svg>',
			'thumbnail'             => '<svg viewBox="0 0 14 12"><path d="M0.332031 7.33333H4.33203V11.3333H0.332031V7.33333ZM9.66537 3.33333H5.66536V4.66666H9.66537V3.33333ZM0.332031 4.66666H4.33203V0.666664H0.332031V4.66666ZM5.66536 0.666664V2H13.6654V0.666664H5.66536ZM5.66536 11.3333H9.66537V10H5.66536V11.3333ZM5.66536 8.66666H13.6654V7.33333H5.66536"/></svg>',
			'halfwidth'             => '<svg viewBox="0 0 14 8"><path d="M6 0.5H0V7.5H6V0.5Z"/><path d="M14 0.75H7.5V2H14V0.75Z"/><path d="M7.5 3.25H14V4.5H7.5V3.25Z"/><path d="M11 5.75H7.5V7H11V5.75Z"/></svg>',
			'fullwidth'             => '<svg viewBox="0 0 10 12"><path fill-rule="evenodd" clip-rule="evenodd" d="M10 6.75V0.333328H0V6.75H10Z"/><path d="M0 8.24999H10V9.49999H0V8.24999Z"/><path d="M6 10.75H0V12H6V10.75Z"/></svg>',
			'boxed'                 => '<svg viewBox="0 0 16 16"><path d="M14.1667 12.8905H1.83333C1.47971 12.8905 1.14057 12.75 0.890524 12.5C0.640476 12.25 0.5 11.9108 0.5 11.5572V3.33333C0.5 2.97971 0.640476 2.64057 0.890524 2.39052C1.14057 2.14048 1.47971 2 1.83333 2H14.1667C14.5203 2 14.8594 2.14048 15.1095 2.39052C15.3595 2.64057 15.5 2.97971 15.5 3.33333V11.5572C15.5 11.9108 15.3595 12.25 15.1095 12.5C14.8594 12.75 14.5203 12.8905 14.1667 12.8905ZM1.83333 3.33333V11.5572H14.1667V3.33333H1.83333Z"/><path d="M8 8H11V9H8V8Z"/><path d="M6.5 9.5H3V5.5H6.5V9.5Z"/><path d="M8 7V6H13V7H8Z"/></svg>',
			'corner'                => '<svg viewBox="0 0 12 12"><path fill-rule="evenodd" clip-rule="evenodd" d="M5 1.5H1.5V10.5H10.5V7C10.5 3.96243 8.03757 1.5 5 1.5ZM0 0V12H12V7C12 3.13401 8.86599 0 5 0H0Z"/></svg>',
			'preview'               => '<svg viewBox="0 0 16 10"><path d="M8.0013 3C7.47087 3 6.96216 3.21071 6.58709 3.58579C6.21202 3.96086 6.0013 4.46957 6.0013 5C6.0013 5.53043 6.21202 6.03914 6.58709 6.41421C6.96216 6.78929 7.47087 7 8.0013 7C8.53173 7 9.04044 6.78929 9.41551 6.41421C9.79059 6.03914 10.0013 5.53043 10.0013 5C10.0013 4.46957 9.79059 3.96086 9.41551 3.58579C9.04044 3.21071 8.53173 3 8.0013 3ZM8.0013 8.33333C7.11725 8.33333 6.2694 7.98214 5.64428 7.35702C5.01916 6.7319 4.66797 5.88406 4.66797 5C4.66797 4.11595 5.01916 3.2681 5.64428 2.64298C6.2694 2.01786 7.11725 1.66667 8.0013 1.66667C8.88536 1.66667 9.7332 2.01786 10.3583 2.64298C10.9834 3.2681 11.3346 4.11595 11.3346 5C11.3346 5.88406 10.9834 6.7319 10.3583 7.35702C9.7332 7.98214 8.88536 8.33333 8.0013 8.33333ZM8.0013 0C4.66797 0 1.8213 2.07333 0.667969 5C1.8213 7.92667 4.66797 10 8.0013 10C11.3346 10 14.1813 7.92667 15.3346 5C14.1813 2.07333 11.3346 0 8.0013 0Z"/></svg>',
			'flag'                  => '<svg viewBox="0 0 9 9"><path d="M5.53203 1L5.33203 0H0.832031V8.5H1.83203V5H4.63203L4.83203 6H8.33203V1H5.53203Z"/></svg>',
			'copy2'                 => '<svg viewBox="0 0 12 13"><path d="M10.25 0.25H4.625C3.9375 0.25 3.375 0.8125 3.375 1.5V9C3.375 9.6875 3.9375 10.25 4.625 10.25H10.25C10.9375 10.25 11.5 9.6875 11.5 9V1.5C11.5 0.8125 10.9375 0.25 10.25 0.25ZM10.25 9H4.625V1.5H10.25V9ZM0.875 8.375V7.125H2.125V8.375H0.875ZM0.875 4.9375H2.125V6.1875H0.875V4.9375ZM5.25 11.5H6.5V12.75H5.25V11.5ZM0.875 10.5625V9.3125H2.125V10.5625H0.875ZM2.125 12.75C1.4375 12.75 0.875 12.1875 0.875 11.5H2.125V12.75ZM4.3125 12.75H3.0625V11.5H4.3125V12.75ZM7.4375 12.75V11.5H8.6875C8.6875 12.1875 8.125 12.75 7.4375 12.75ZM2.125 2.75V4H0.875C0.875 3.3125 1.4375 2.75 2.125 2.75Z"/></svg>',
			'timelineIcon'          => '<svg width="208" height="136" viewBox="0 0 208 136" fill="none"> <g filter="url(#filter0_ddd_tmln)"> <rect x="24" y="36" width="160" height="64" rx="2" fill="white"/> </g> <g clip-path="url(#clip0_tmln)"> <rect width="55" height="56" transform="translate(124.8 40)" fill="#F9BBA0"/> <circle cx="200.3" cy="102.5" r="55.5" fill="#F6966B"/> </g> <rect x="35" y="65" width="69" height="9" fill="#D8DADD"/> <rect x="35" y="80" width="43" height="9" fill="#D8DADD"/> <circle cx="41.5" cy="50.5" r="6.5" fill="#D8DADD"/> <defs> <filter id="filter0_ddd_tmln" x="11" y="29" width="186" height="90" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <clipPath id="clip0_tmln"> <rect width="55" height="56" fill="white" transform="translate(124.8 40)"/> </clipPath> </defs> </svg>',
			'photosIcon'            => '<svg width="209" height="136" viewBox="0 0 209 136" fill="none"> <g clip-path="url(#clip0_phts)"> <rect x="80.2002" y="44" width="48" height="48" fill="#43A6DB"/> <circle cx="70.7002" cy="78.5" r="40.5" fill="#86D0F9"/> </g> <g clip-path="url(#clip1_phts)"> <rect x="131.2" y="44" width="48" height="48" fill="#B6DDAD"/> <rect x="152.2" y="65" width="33" height="33" fill="#96CE89"/> </g> <g clip-path="url(#clip2_phts)"> <rect x="29.2002" y="44" width="48" height="48" fill="#F6966B"/> <path d="M38.6485 61L76.6485 99H7.2002L38.6485 61Z" fill="#F9BBA0"/> </g> <defs> <clipPath id="clip0_phts"> <rect x="80.2002" y="44" width="48" height="48" rx="1" fill="white"/> </clipPath> <clipPath id="clip1_phts"> <rect x="131.2" y="44" width="48" height="48" rx="1" fill="white"/> </clipPath> <clipPath id="clip2_phts"> <rect x="29.2002" y="44" width="48" height="48" rx="1" fill="white"/> </clipPath> </defs> </svg>',
			'videosIcon'            => '<svg width="209" height="136" viewBox="0 0 209 136" fill="none"> <rect x="41.6001" y="31" width="126" height="74" fill="#43A6DB"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M104.6 81C111.78 81 117.6 75.1797 117.6 68C117.6 60.8203 111.78 55 104.6 55C97.4204 55 91.6001 60.8203 91.6001 68C91.6001 75.1797 97.4204 81 104.6 81ZM102.348 63.2846C102.015 63.0942 101.6 63.3349 101.6 63.7188V72.2813C101.6 72.6652 102.015 72.9059 102.348 72.7154L109.84 68.4342C110.176 68.2422 110.176 67.7579 109.84 67.5659L102.348 63.2846Z" fill="white"/> </svg>',
			'albumsIcon'            => '<svg width="210" height="136" viewBox="0 0 210 136" fill="none"> <g clip-path="url(#clip0_albm)"> <rect x="76.1187" y="39.7202" width="57.7627" height="57.7627" fill="#43A6DB"/> <rect x="101.39" y="64.9917" width="39.7119" height="39.7119" fill="#86D0F9"/> </g> <g clip-path="url(#clip1_albm)"> <rect x="70.1016" y="32.5" width="57.7627" height="57.7627" fill="#F9BBA0"/> <path d="M81.4715 52.9575L127.2 98.6863H43.627L81.4715 52.9575Z" fill="#F6966B"/> </g> <defs> <clipPath id="clip0_albm"> <rect x="76.1187" y="39.7202" width="57.7627" height="57.7627" rx="1.20339" fill="white"/> </clipPath> <clipPath id="clip1_albm"> <rect x="70.1016" y="32.5" width="57.7627" height="57.7627" rx="1.20339" fill="white"/> </clipPath> </defs> </svg>',
			'eventsIcon'            => '<svg width="209" height="136" viewBox="0 0 209 136" fill="none"> <g filter="url(#filter0_ddd_evt)"> <rect x="20.5562" y="39.9375" width="160" height="64" rx="2" fill="white"/> </g> <rect x="31.6001" y="69" width="102" height="9" fill="#D8DADD"/> <rect x="31.6001" y="84" width="64" height="9" fill="#D8DADD"/> <circle cx="38.0562" cy="54.4375" r="6.5" fill="#D8DADD"/> <circle cx="173.744" cy="46.5625" r="14.5" fill="#FE544F"/> <path d="M169.275 53.5L173.775 50.875L178.275 53.5V42.625C178.275 42.0156 177.759 41.5 177.15 41.5H170.4C169.767 41.5 169.275 42.0156 169.275 42.625V53.5Z" fill="white"/> <defs> <filter id="filter0_ddd_evt" x="7.55615" y="32.9375" width="186" height="90" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> </defs> </svg>',
			'reviewsIcon'           => '<svg width="207" height="129" viewBox="0 0 207 129" fill="none"> <g filter="url(#filter0_ddd_rev)"> <rect x="23.5" y="32.5" width="160" height="64" rx="2" fill="white"/> </g> <path d="M61.0044 42.8004C61.048 42.6917 61.202 42.6917 61.2456 42.8004L62.7757 46.6105C62.7942 46.6568 62.8377 46.6884 62.8875 46.6917L66.9839 46.9695C67.1008 46.9774 67.1484 47.1238 67.0584 47.199L63.9077 49.8315C63.8694 49.8635 63.8528 49.9145 63.8649 49.9629L64.8666 53.9447C64.8952 56.0583 64.7707 54.1488 64.6714 56.0865L61.1941 51.9034C61.1519 51.8769 61.0981 51.8769 61.0559 51.9034L57.5786 56.0865C57.4793 54.1488 57.3548 56.0583 57.3834 53.9447L58.3851 49.9629C58.3972 49.9145 58.3806 49.8635 58.3423 49.8315L55.1916 47.199C55.1016 47.1238 55.1492 46.9774 55.2661 46.9695L59.3625 46.6917C59.4123 46.6884 59.4558 46.6568 59.4743 46.6105L61.0044 42.8004Z" fill="#FE544F"/> <path d="M76.6045 42.8004C76.6481 42.6917 76.8021 42.6917 76.8457 42.8004L78.3757 46.6105C78.3943 46.6568 78.4378 46.6884 78.4876 46.6917L82.584 46.9695C82.7009 46.9774 82.7485 47.1238 82.6585 47.199L79.5078 49.8315C79.4695 49.8635 79.4529 49.9145 79.465 49.9629L80.4667 53.9447C80.4953 56.0583 80.3708 54.1488 80.2715 56.0865L76.7942 51.9034C76.752 51.8769 76.6982 51.8769 76.656 51.9034L73.1787 56.0865C73.0794 54.1488 72.9549 56.0583 72.9835 53.9447L73.9852 49.9629C73.9973 49.9145 73.9807 49.8635 73.9424 49.8315L70.7917 47.199C70.7017 47.1238 70.7493 46.9774 70.8662 46.9695L74.9626 46.6917C75.0124 46.6884 75.0559 46.6568 75.0744 46.6105L76.6045 42.8004Z" fill="#FE544F"/> <path d="M92.2046 42.8004C92.2482 42.6917 92.4022 42.6917 92.4458 42.8004L93.9758 46.6105C93.9944 46.6568 96.0379 46.6884 96.0877 46.6917L98.1841 46.9695C98.301 46.9774 98.3486 47.1238 98.2586 47.199L95.1078 49.8315C95.0696 49.8635 95.053 49.9145 95.0651 49.9629L96.0668 53.9447C96.0954 56.0583 95.9709 54.1488 95.8716 56.0865L92.3943 51.9034C92.3521 51.8769 92.2983 51.8769 92.2561 51.9034L88.7788 56.0865C88.6795 54.1488 88.555 56.0583 88.5836 53.9447L89.5853 49.9629C89.5974 49.9145 89.5808 49.8635 89.5425 49.8315L86.3918 47.199C86.3018 47.1238 86.3494 46.9774 86.4663 46.9695L90.5627 46.6917C90.6125 46.6884 90.6559 46.6568 90.6745 46.6105L92.2046 42.8004Z" fill="#FE544F"/> <path d="M107.804 42.8004C107.848 42.6917 108.002 42.6917 108.045 42.8004L109.575 46.6105C109.594 46.6568 109.638 46.6884 109.687 46.6917L113.784 46.9695C113.901 46.9774 113.948 47.1238 113.858 47.199L110.707 49.8315C110.669 49.8635 110.653 49.9145 110.665 49.9629L111.666 53.9447C111.695 56.0583 111.57 54.1488 111.471 56.0865L107.994 51.9034C107.952 51.8769 107.898 51.8769 107.856 51.9034L104.378 56.0865C104.279 54.1488 104.155 56.0583 104.183 53.9447L105.185 49.9629C105.197 49.9145 105.18 49.8635 105.142 49.8315L101.991 47.199C101.901 47.1238 101.949 46.9774 102.066 46.9695L106.162 46.6917C106.212 46.6884 106.256 46.6568 106.274 46.6105L107.804 42.8004Z" fill="#FE544F"/> <path d="M123.404 42.8004C123.448 42.6917 123.602 42.6917 123.646 42.8004L125.176 46.6105C125.194 46.6568 125.238 46.6884 125.287 46.6917L129.384 46.9695C129.501 46.9774 129.548 47.1238 129.458 47.199L126.308 49.8315C126.269 49.8635 126.253 49.9145 126.265 49.9629L127.267 53.9447C127.295 56.0583 127.171 54.1488 127.071 56.0865L123.594 51.9034C123.552 51.8769 123.498 51.8769 123.456 51.9034L119.978 56.0865C119.879 54.1488 119.755 56.0583 119.783 53.9447L120.785 49.9629C120.797 49.9145 120.781 49.8635 120.742 49.8315L117.591 47.199C117.502 47.1238 117.549 46.9774 117.666 46.9695L121.762 46.6917C121.812 46.6884 121.856 46.6568 121.874 46.6105L123.404 42.8004Z" fill="#FE544F"/> <rect x="54.625" y="65.5" width="70" height="7" fill="#D8DADD"/> <rect x="54.625" y="78.5" width="43" height="7" fill="#D8DADD"/> <circle cx="39" cy="49" r="6.5" fill="#D8DADD"/> <defs> <filter id="filter0_ddd_rev" x="10.5" y="25.5" width="186" height="90" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> </defs> </svg>',
			'featuredpostIcon'      => '<svg width="207" height="129" viewBox="0 0 207 129" fill="none"> <g filter="url(#filter0_ddd_ftpst)"> <rect x="21.4282" y="34.7188" width="160" height="64" rx="2" fill="white"/> </g> <g clip-path="url(#clip0_ftpst)"> <rect width="55" height="56" transform="translate(122.228 38.7188)" fill="#43A6DB"/> <circle cx="197.728" cy="101.219" r="55.5" fill="#86D0F9"/> </g> <rect x="32.4282" y="63.7188" width="69" height="9" fill="#D8DADD"/> <rect x="32.4282" y="78.7188" width="43" height="9" fill="#D8DADD"/> <circle cx="38.9282" cy="49.2188" r="6.5" fill="#D8DADD"/> <circle cx="171.072" cy="44.7812" r="15.5" fill="#EC352F" stroke="#FEF4EF" stroke-width="2"/> <path d="M173.587 44.7578L173.283 41.9688H174.291C174.595 41.9688 174.853 41.7344 174.853 41.4062V40.2812C174.853 39.9766 174.595 39.7188 174.291 39.7188H167.916C167.587 39.7188 167.353 39.9766 167.353 40.2812V41.4062C167.353 41.7344 167.587 41.9688 167.916 41.9688H168.9L168.595 44.7578C167.47 45.2734 166.603 46.2344 166.603 47.4062C166.603 47.7344 166.837 47.9688 167.166 47.9688H170.353V50.4297C170.353 50.4531 170.353 50.4766 170.353 50.5L170.916 51.625C170.986 51.7656 171.197 51.7656 171.267 51.625L171.83 50.5C171.83 50.4766 171.853 50.4531 171.853 50.4297V47.9688H175.041C175.345 47.9688 175.603 47.7344 175.603 47.4062C175.603 46.2109 174.712 45.2734 173.587 44.7578Z" fill="white"/> <defs> <filter id="filter0_ddd_ftpst" x="8.42822" y="27.7188" width="186" height="90" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <clipPath id="clip0_ftpst"> <rect width="55" height="56" fill="white" transform="translate(122.228 38.7188)"/> </clipPath> </defs> </svg>',
			'singlealbumIcon'       => '<svg width="207" height="129" viewBox="0 0 207 129" fill="none"> <g clip-path="url(#clip0_sglalb)"> <rect x="74.6187" y="36.2202" width="57.7627" height="57.7627" fill="#43A6DB"/> <rect x="99.8896" y="61.4917" width="39.7119" height="39.7119" fill="#86D0F9"/> </g> <g clip-path="url(#clip1_sglalb)"> <rect x="68.6016" y="29" width="57.7627" height="57.7627" fill="#F9BBA0"/> <path d="M79.9715 49.4575L125.7 95.1863H42.127L79.9715 49.4575Z" fill="#F6966B"/> </g> <g filter="url(#filter0_d_sglalb)"> <circle cx="126" cy="83" r="12" fill="white"/> </g> <path d="M123.584 79H122.205L120.217 80.2773V81.6055L122.088 80.4102H122.135V87H123.584V79ZM126.677 81H125.177L126.959 84L125.131 87H126.631L127.888 84.8398L129.158 87H130.646L128.806 84L130.615 81H129.119L127.888 83.2148L126.677 81Z" fill="black"/> <defs> <filter id="filter0_d_sglalb" x="109" y="67" width="34" height="34" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="2.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow" result="shape"/> </filter> <clipPath id="clip0_sglalb"> <rect x="74.6187" y="36.2202" width="57.7627" height="57.7627" rx="1.20339" fill="white"/> </clipPath> <clipPath id="clip1_sglalb"> <rect x="68.6016" y="29" width="57.7627" height="57.7627" rx="1.20339" fill="white"/> </clipPath> </defs> </svg>',
			'socialwallIcon'        => '<svg width="207" height="129" viewBox="0 0 207 129" fill="none"> <path d="M96.6875 47.5C96.6875 42.1484 92.3516 37.8125 87 37.8125C81.6484 37.8125 77.3125 42.1484 77.3125 47.5C77.3125 52.3438 80.8281 56.3672 85.4766 57.0703V50.3125H83.0156V47.5H85.4766V45.3906C85.4766 42.9688 86.9219 41.6016 89.1094 41.6016C90.2031 41.6016 91.2969 41.7969 91.2969 41.7969V44.1797H90.0859C88.875 44.1797 88.4844 44.9219 88.4844 45.7031V47.5H91.1797L90.75 50.3125H88.4844V57.0703C93.1328 56.3672 96.6875 52.3438 96.6875 47.5Z" fill="#2A65DB"/> <path d="M128.695 42.3828C128.461 41.4453 127.719 40.7031 126.82 40.4688C125.141 40 118.5 40 118.5 40C118.5 40 111.82 40 110.141 40.4688C109.242 40.7031 108.5 41.4453 108.266 42.3828C107.797 46.0234 107.797 47.5391 107.797 47.5391C107.797 47.5391 107.797 51.0156 108.266 52.6953C108.5 53.6328 109.242 54.3359 110.141 54.5703C111.82 55 118.5 55 118.5 55C118.5 55 125.141 55 126.82 54.5703C127.719 54.3359 128.461 53.6328 128.695 52.6953C129.164 51.0156 129.164 47.5391 129.164 47.5391C129.164 47.5391 129.164 46.0234 128.695 42.3828ZM116.312 50.7031V44.375L121.859 47.5391L116.312 50.7031Z" fill="url(#paint0_linear_sclwl)"/> <path d="M86 78.0078C83.5 78.0078 81.5078 80.0391 81.5078 82.5C81.5078 85 83.5 86.9922 86 86.9922C88.4609 86.9922 90.4922 85 90.4922 82.5C90.4922 80.0391 88.4609 78.0078 86 78.0078ZM86 85.4297C84.3984 85.4297 83.0703 84.1406 83.0703 82.5C83.0703 80.8984 84.3594 79.6094 86 79.6094C87.6016 79.6094 88.8906 80.8984 88.8906 82.5C88.8906 84.1406 87.6016 85.4297 86 85.4297ZM91.7031 77.8516C91.7031 77.2656 91.2344 76.7969 90.6484 76.7969C90.0625 76.7969 89.5938 77.2656 89.5938 77.8516C89.5938 78.4375 90.0625 78.9062 90.6484 78.9062C91.2344 78.9062 91.7031 78.4375 91.7031 77.8516ZM94.6719 78.9062C94.5938 77.5 94.2812 76.25 93.2656 75.2344C92.25 74.2188 91 73.9062 89.5938 73.8281C88.1484 73.75 83.8125 73.75 82.3672 73.8281C80.9609 73.9062 79.75 74.2188 78.6953 75.2344C77.6797 76.25 77.3672 77.5 77.2891 78.9062C77.2109 80.3516 77.2109 84.6875 77.2891 86.1328C77.3672 87.5391 77.6797 88.75 78.6953 89.8047C79.75 90.8203 80.9609 91.1328 82.3672 91.2109C83.8125 91.2891 88.1484 91.2891 89.5938 91.2109C91 91.1328 92.25 90.8203 93.2656 89.8047C94.2812 88.75 94.5938 87.5391 94.6719 86.1328C94.75 84.6875 94.75 80.3516 94.6719 78.9062ZM92.7969 87.6562C92.5234 88.4375 91.8984 89.0234 91.1562 89.3359C89.9844 89.8047 87.25 89.6875 86 89.6875C84.7109 89.6875 81.9766 89.8047 80.8438 89.3359C80.0625 89.0234 79.4766 88.4375 79.1641 87.6562C78.6953 86.5234 78.8125 83.7891 78.8125 82.5C78.8125 81.25 78.6953 78.5156 79.1641 77.3438C79.4766 76.6016 80.0625 76.0156 80.8438 75.7031C81.9766 75.2344 84.7109 75.3516 86 75.3516C87.25 75.3516 89.9844 75.2344 91.1562 75.7031C91.8984 75.9766 92.4844 76.6016 92.7969 77.3438C93.2656 78.5156 93.1484 81.25 93.1484 82.5C93.1484 83.7891 93.2656 86.5234 92.7969 87.6562Z" fill="url(#paint1_linear_swwl)"/> <path d="M127.93 78.4375C128.711 77.8516 129.414 77.1484 129.961 76.3281C129.258 76.6406 128.438 76.875 127.617 76.9531C128.477 76.4453 129.102 75.6641 129.414 74.6875C128.633 75.1562 127.734 75.5078 126.836 75.7031C126.055 74.8828 125 74.4141 123.828 74.4141C121.562 74.4141 119.727 76.25 119.727 78.5156C119.727 78.8281 119.766 79.1406 119.844 79.4531C116.445 79.2578 113.398 77.6172 111.367 75.1562C111.016 75.7422 110.82 76.4453 110.82 77.2266C110.82 78.6328 111.523 79.8828 112.656 80.625C111.992 80.5859 111.328 80.4297 110.781 80.1172V80.1562C110.781 82.1484 112.188 83.7891 116.062 84.1797C113.75 84.2578 113.359 84.3359 113.008 84.3359C112.734 84.3359 112.5 84.2969 112.227 84.2578C112.734 85.8984 114.258 87.0703 116.055 87.1094C114.648 88.2031 112.891 88.8672 110.977 88.8672C110.625 88.8672 110.312 88.8281 110 88.7891C111.797 89.9609 113.945 90.625 116.289 90.625C123.828 90.625 127.93 84.4141 127.93 78.9844C127.93 78.7891 127.93 78.6328 127.93 78.4375Z" fill="url(#paint2_linear)"/> <defs> <linearGradient id="paint0_linear_sclwl" x1="137.667" y1="33.4445" x2="109.486" y2="62.2514" gradientUnits="userSpaceOnUse"> <stop stop-color="#E3280E"/> <stop offset="1" stop-color="#E30E0E"/> </linearGradient> <linearGradient id="paint1_linear_swwl" x1="93.8998" y1="73.3444" x2="78.4998" y2="89.4444" gradientUnits="userSpaceOnUse"> <stop stop-color="#5F0EE3"/> <stop offset="0.713476" stop-color="#FF0000"/> <stop offset="1" stop-color="#FF5C00"/> </linearGradient> <linearGradient id="paint2_linear" x1="136.667" y1="68.4445" x2="108.674" y2="93.3272" gradientUnits="userSpaceOnUse"> <stop stop-color="#0E96E3"/> <stop offset="1" stop-color="#0EBDE3"/> </linearGradient> </defs> </svg>',
			'addPage'               => '<svg viewBox="0 0 17 17"><path d="M12.1667 9.66667H13.8333V12.1667H16.3333V13.8333H13.8333V16.3333H12.1667V13.8333H9.66667V12.1667H12.1667V9.66667ZM2.16667 0.5H13.8333C14.7583 0.5 15.5 1.24167 15.5 2.16667V8.66667C14.9917 8.375 14.4333 8.16667 13.8333 8.06667V2.16667H2.16667V13.8333H8.06667C8.16667 14.4333 8.375 14.9917 8.66667 15.5H2.16667C1.24167 15.5 0.5 14.7583 0.5 13.8333V2.16667C0.5 1.24167 1.24167 0.5 2.16667 0.5ZM3.83333 3.83333H12.1667V5.5H3.83333V3.83333ZM3.83333 7.16667H12.1667V8.06667C11.4583 8.18333 10.8083 8.45 10.2333 8.83333H3.83333V7.16667ZM3.83333 10.5H8V12.1667H3.83333V10.5Z"/></svg>',
			'addWidget'             => '<svg viewBox="0 0 15 16"><path d="M0 15.5H6.66667V8.83333H0V15.5ZM1.66667 10.5H5V13.8333H1.66667V10.5ZM0 7.16667H6.66667V0.5H0V7.16667ZM1.66667 2.16667H5V5.5H1.66667V2.16667ZM8.33333 0.5V7.16667H15V0.5H8.33333ZM13.3333 5.5H10V2.16667H13.3333V5.5ZM12.5 11.3333H15V13H12.5V15.5H10.8333V13H8.33333V11.3333H10.8333V8.83333H12.5V11.3333Z"/></svg>',
			'plus'                  => '<svg width="13" height="12" viewBox="0 0 13 12"><path d="M12.3327 6.83332H7.33268V11.8333H5.66602V6.83332H0.666016V5.16666H5.66602V0.166656H7.33268V5.16666H12.3327V6.83332Z"/></svg>',
			'eye1'                  => '<svg width="20" height="17" viewBox="0 0 20 17"><path d="M9.85801 5.5L12.4997 8.13333V8C12.4997 7.33696 12.2363 6.70107 11.7674 6.23223C11.2986 5.76339 10.6627 5.5 9.99967 5.5H9.85801ZM6.27467 6.16667L7.56634 7.45833C7.52467 7.63333 7.49967 7.80833 7.49967 8C7.49967 8.66304 7.76307 9.29893 8.23191 9.76777C8.70075 10.2366 9.33663 10.5 9.99967 10.5C10.183 10.5 10.3663 10.475 10.5413 10.4333L11.833 11.725C11.2747 12 10.658 12.1667 9.99967 12.1667C8.8946 12.1667 7.8348 11.7277 7.0534 10.9463C6.27199 10.1649 5.83301 9.10507 5.83301 8C5.83301 7.34167 5.99967 6.725 6.27467 6.16667ZM1.66634 1.55833L3.56634 3.45833L3.94134 3.83333C2.56634 4.91667 1.48301 6.33333 0.833008 8C2.27467 11.6583 5.83301 14.25 9.99967 14.25C11.2913 14.25 12.5247 14 13.6497 13.55L14.008 13.9L16.4413 16.3333L17.4997 15.275L2.72467 0.5L1.66634 1.55833ZM9.99967 3.83333C11.1047 3.83333 12.1645 4.27232 12.946 5.05372C13.7274 5.83512 14.1663 6.89493 14.1663 8C14.1663 8.53333 14.058 9.05 13.8663 9.51667L16.308 11.9583C17.558 10.9167 18.558 9.55 19.1663 8C17.7247 4.34167 14.1663 1.75 9.99967 1.75C8.83301 1.75 7.71634 1.95833 6.66634 2.33333L8.47467 4.125C8.94967 3.94167 9.45801 3.83333 9.99967 3.83333Z"/></svg>',

			'eyePreview'            => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path d="M569.354 231.631C512.97 135.949 407.81 72 288 72 168.14 72 63.004 135.994 6.646 231.631a47.999 47.999 0 0 0 0 48.739C63.031 376.051 168.19 440 288 440c119.86 0 224.996-63.994 281.354-159.631a47.997 47.997 0 0 0 0-48.738zM288 392c-102.556 0-192.091-54.701-240-136 44.157-74.933 123.677-127.27 216.162-135.007C273.958 131.078 280 144.83 280 160c0 30.928-25.072 56-56 56s-56-25.072-56-56l.001-.042C157.794 179.043 152 200.844 152 224c0 75.111 60.889 136 136 136s136-60.889 136-136c0-31.031-10.4-59.629-27.895-82.515C451.704 164.638 498.009 205.106 528 256c-47.908 81.299-137.444 136-240 136z"/></svg>',

			'facebookShare'         => '<svg viewBox="0 0 448 512"><path fill="currentColor" d="M400 32H48A48 48 0 0 0 0 80v352a48 48 0 0 0 48 48h137.25V327.69h-63V256h63v-54.64c0-62.15 37-96.48 93.67-96.48 27.14 0 55.52 4.84 55.52 4.84v61h-31.27c-30.81 0-40.42 19.12-40.42 38.73V256h68.78l-11 71.69h-57.78V480H400a48 48 0 0 0 48-48V80a48 48 0 0 0-48-48z"></path></svg>',
			'twitterShare'          => '<svg viewBox="0 0 512 512"><path fill="currentColor" d="M459.37 151.716c.325 4.548.325 9.097.325 13.645 0 138.72-105.583 298.558-298.558 298.558-59.452 0-114.68-17.219-161.137-47.106 8.447.974 16.568 1.299 25.34 1.299 49.055 0 94.213-16.568 130.274-44.832-46.132-.975-84.792-31.188-98.112-72.772 6.498.974 12.995 1.624 19.818 1.624 9.421 0 18.843-1.3 27.614-3.573-48.081-9.747-84.143-51.98-84.143-102.985v-1.299c13.969 7.797 30.214 12.67 47.431 13.319-28.264-18.843-46.781-51.005-46.781-87.391 0-19.492 5.197-37.36 14.294-52.954 51.655 63.675 129.3 105.258 216.365 109.807-1.624-7.797-2.599-15.918-2.599-26.04 0-57.828 46.782-104.934 104.934-104.934 30.213 0 57.502 12.67 76.67 33.137 23.715-4.548 46.456-13.32 66.599-25.34-7.798 24.366-24.366 44.833-46.132 57.827 21.117-2.273 41.584-8.122 60.426-16.243-14.292 20.791-32.161 39.308-52.628 54.253z"></path></svg>',
			'linkedinShare'         => '<svg viewBox="0 0 448 512"><path fill="currentColor" d="M100.28 448H7.4V148.9h92.88zM53.79 108.1C26.09 108.1 0 83.5 0 53.8a53.79 53.79 0 0 1 107.58 0c0 29.7-24.1 54.3-53.79 54.3zM447.9 448h-92.68V302.4c0-34.7-.7-79.2-48.29-79.2-48.29 0-55.69 37.7-55.69 76.7V448h-92.78V148.9h89.08v40.8h1.3c12.4-23.5 42.69-48.3 87.88-48.3 94 0 111.28 61.9 111.28 142.3V448z"></path></svg>',
			'mailShare'             => '<svg viewBox="0 0 512 512"><path fill="currentColor" d="M502.3 190.8c3.9-3.1 9.7-.2 9.7 4.7V400c0 26.5-21.5 48-48 48H48c-26.5 0-48-21.5-48-48V195.6c0-5 5.7-7.8 9.7-4.7 22.4 17.4 52.1 39.5 154.1 113.6 21.1 15.4 56.7 47.8 92.2 47.6 35.7.3 72-32.8 92.3-47.6 102-74.1 131.6-96.3 154-113.7zM256 320c23.2.4 56.6-29.2 73.4-41.4 132.7-96.3 142.8-104.7 173.4-128.7 5.8-4.5 9.2-11.5 9.2-18.9v-19c0-26.5-21.5-48-48-48H48C21.5 64 0 85.5 0 112v19c0 7.4 3.4 14.3 9.2 18.9 30.6 23.9 40.7 32.4 173.4 128.7 16.8 12.2 50.2 41.8 73.4 41.4z"></path></svg>',

			'successNotification'   => '<svg viewBox="0 0 20 20"><path d="M10 0C4.5 0 0 4.5 0 10C0 15.5 4.5 20 10 20C15.5 20 20 15.5 20 10C20 4.5 15.5 0 10 0ZM8 15L3 10L4.41 8.59L8 12.17L15.59 4.58L17 6L8 15Z"/></svg>',
			'errorNotification'     => '<svg viewBox="0 0 20 20"><path d="M9.99997 0C4.47997 0 -3.05176e-05 4.48 -3.05176e-05 10C-3.05176e-05 15.52 4.47997 20 9.99997 20C15.52 20 20 15.52 20 10C20 4.48 15.52 0 9.99997 0ZM11 15H8.99997V13H11V15ZM11 11H8.99997V5H11V11Z"/></svg>',
			'messageNotification'   => '<svg viewBox="0 0 20 20"><path d="M11.0001 7H9.00012V5H11.0001V7ZM11.0001 15H9.00012V9H11.0001V15ZM10.0001 0C8.6869 0 7.38654 0.258658 6.17329 0.761205C4.96003 1.26375 3.85764 2.00035 2.92905 2.92893C1.05369 4.8043 0.00012207 7.34784 0.00012207 10C0.00012207 12.6522 1.05369 15.1957 2.92905 17.0711C3.85764 17.9997 4.96003 18.7362 6.17329 19.2388C7.38654 19.7413 8.6869 20 10.0001 20C12.6523 20 15.1958 18.9464 17.0712 17.0711C18.9466 15.1957 20.0001 12.6522 20.0001 10C20.0001 8.68678 19.7415 7.38642 19.2389 6.17317C18.7364 4.95991 17.9998 3.85752 17.0712 2.92893C16.1426 2.00035 15.0402 1.26375 13.827 0.761205C12.6137 0.258658 11.3133 0 10.0001 0Z"/></svg>',

			'albumsPreview'         => '<svg width="63" height="65" viewBox="0 0 63 65" fill="none"><rect x="13.6484" y="10.2842" width="34.7288" height="34.7288" rx="1.44703" fill="#8C8F9A"/> <g filter="url(#filter0_dddalbumsPreview)"><rect x="22.1484" y="5.21962" width="34.7288" height="34.7288" rx="1.44703" transform="rotate(8 22.1484 5.21962)" fill="white"/> </g><path d="M29.0485 23.724L18.9288 28.1468L17.2674 39.9686L51.6582 44.802L52.2623 40.5031L29.0485 23.724Z" fill="#B5E5FF"/> <path d="M44.9106 25.2228L17.7194 36.7445L17.2663 39.9687L51.6571 44.802L53.4696 31.9054L44.9106 25.2228Z" fill="#43A6DB"/> <circle cx="42.9495" cy="18.3718" r="2.89406" transform="rotate(8 42.9495 18.3718)" fill="#43A6DB"/> <g filter="url(#filter1_dddalbumsPreview)"> <rect x="42.4766" y="33.9054" width="16.875" height="16.875" rx="8.4375" fill="white"/> <path d="M54.1953 42.8116H51.3828V45.6241H50.4453V42.8116H47.6328V41.8741H50.4453V39.0616H51.3828V41.8741H54.1953V42.8116Z" fill="#0068A0"/> </g> <defs> <filter id="filter0_dddalbumsPreview" x="0.86108" y="0.342124" width="58.3848" height="57.6613" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dx="-7.23516" dy="4.3411"/> <feGaussianBlur stdDeviation="4.70286"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.1 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="2.89406"/> <feGaussianBlur stdDeviation="1.44703"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="filter1_dddalbumsPreview" x="25.8357" y="28.8408" width="36.4099" height="35.6864" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dx="-7.23516" dy="4.3411"/> <feGaussianBlur stdDeviation="4.70286"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.1 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dy="2.89406"/> <feGaussianBlur stdDeviation="1.44703"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> </defs> </svg>',
			'featuredPostPreview'   => '<svg width="47" height="48" viewBox="0 0 47 48" fill="none"> <g filter="url(#filter0_ddfeaturedpos)"> <rect x="2.09375" y="1.84264" width="34.7288" height="34.7288" rx="1.44703" fill="white"/> </g> <path d="M11.4995 19.2068L2.09375 24.9949L2.09375 36.9329H36.8225V32.5918L11.4995 19.2068Z" fill="#B5E5FF"/> <path d="M27.4168 18.4833L2.09375 33.6772V36.933H36.8225V23.9097L27.4168 18.4833Z" fill="#43A6DB"/> <circle cx="24.523" cy="11.9718" r="2.89406" fill="#43A6DB"/> <g filter="url(#filter1_ddfeaturedpos)"> <rect x="26.0312" y="25.2824" width="16.875" height="16.875" rx="8.4375" fill="white"/> <path d="M37.75 34.1886H34.9375V37.0011H34V34.1886H31.1875V33.2511H34V30.4386H34.9375V33.2511H37.75V34.1886Z" fill="#0068A0"/> </g> <defs> <filter id="filter0_ddfeaturedpos" x="0.09375" y="0.842636" width="40.7288" height="40.7288" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dx="1" dy="2"/> <feGaussianBlur stdDeviation="1.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.1 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow" result="shape"/> </filter> <filter id="filter1_ddfeaturedpos" x="26.0312" y="24.2824" width="22.875" height="22.875" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset dx="1" dy="2"/> <feGaussianBlur stdDeviation="1.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.1 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"/> <feOffset/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.25 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow" result="shape"/> </filter> </defs> </svg>',
			'issueSinglePreview'    => '<svg width="27" height="18" viewBox="0 0 27 18" fill="none"> <line x1="3.22082" y1="2.84915" x2="8.91471" y2="8.54304" stroke="#8C8F9A" stroke-width="3"/> <path d="M3.10938 8.65422L8.80327 2.96033" stroke="#8C8F9A" stroke-width="3"/> <line x1="18.3107" y1="2.84915" x2="26.0046" y2="8.54304" stroke="#8C8F9A" stroke-width="3"/> <path d="M18.1992 8.65422L23.8931 2.96033" stroke="#8C8F9A" stroke-width="3"/> <line x1="8.64062" y1="16.3863" x2="18.0351" y2="16.3863" stroke="#8C8F9A" stroke-width="3"/> </svg>',
			'playButton'            => '<svg viewBox="0 0 448 512"><path fill="currentColor" d="M424.4 214.7L72.4 6.6C43.8-10.3 0 6.1 0 47.9V464c0 37.5 40.7 60.1 72.4 41.3l352-208c31.4-18.5 31.5-64.1 0-82.6z"></path></svg>',
			'spinner'               => '<svg version="1.1" id="loader-1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="20px" height="20px" viewBox="0 0 50 50" style="enable-background:new 0 0 50 50;" xml:space="preserve"><path fill="#fff" d="M43.935,25.145c0-10.318-8.364-18.683-18.683-18.683c-10.318,0-18.683,8.365-18.683,18.683h6.068c0-8.071,6.543-14.615,14.615-14.615c8.072,0,14.615,6.543,14.615,14.615H43.935z"><animateTransform attributeType="xml" attributeName="transform" type="rotate" from="0 25 25" to="360 25 25" dur="0.6s" repeatCount="indefinite"/></path></svg>',
			'rocket'                => '<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M10.9411 18.4918L9.58281 15.3001C10.8911 14.8168 12.1161 14.1668 13.2495 13.4085L10.9411 18.4918ZM4.69948 10.4168L1.50781 9.05846L6.59115 6.75013C5.83281 7.88346 5.18281 9.10846 4.69948 10.4168ZM18.0078 1.9918C18.0078 1.9918 13.8828 0.224296 9.16615 4.9418C7.34115 6.7668 6.24948 8.77513 5.54115 10.5335C5.30781 11.1585 5.46615 11.8418 5.92448 12.3085L7.69948 14.0751C8.15781 14.5418 8.84115 14.6918 9.46615 14.4585C11.5649 13.6582 13.4706 12.4228 15.0578 10.8335C19.7745 6.1168 18.0078 1.9918 18.0078 1.9918ZM12.1161 7.88346C11.4661 7.23346 11.4661 6.17513 12.1161 5.52513C12.7661 4.87513 13.8245 4.87513 14.4745 5.52513C15.1161 6.17513 15.1245 7.23346 14.4745 7.88346C13.8245 8.53346 12.7661 8.53346 12.1161 7.88346ZM6.03297 17.5001L8.23281 15.3001C7.94948 15.2251 7.67448 15.1001 7.42448 14.9251L4.85797 17.5001H6.03297ZM2.49963 17.5001H3.67463L6.81615 14.3668L5.63281 13.1918L2.49963 16.3251V17.5001ZM2.49963 15.1418L5.07448 12.5751C4.89948 12.3251 4.77448 12.0585 4.69948 11.7668L2.49963 13.9668V15.1418Z" fill="white"/></svg>',
			'follow'                => '<svg viewBox="0 0 24 24"><path d="M20 18.5H4C3.46957 18.5 2.96086 18.2893 2.58579 17.9142C2.21071 17.5391 2 17.0304 2 16.5V7.5C2 6.96957 2.21071 6.46086 2.58579 6.08579C2.96086 5.71071 3.46957 5.5 4 5.5H20C20.5304 5.5 21.0391 5.71071 21.4142 6.08579C21.7893 6.46086 22 6.96957 22 7.5V16.5C22 17.0304 21.7893 17.5391 21.4142 17.9142C21.0391 18.2893 20.5304 18.5 20 18.5ZM4 7.5V16.5H20V7.5H4Z" fill="#141B38"/><path d="M9 13.75C9 13.1977 9.44772 12.75 10 12.75H14C14.5523 12.75 15 13.1977 15 13.75V15H9V13.75Z" fill="#141B38"/><path d="M13.5 10.5C13.5 11.3284 12.8284 12 12 12C11.1716 12 10.5 11.3284 10.5 10.5C10.5 9.67157 11.1716 9 12 9C12.8284 9 13.5 9.67157 13.5 10.5Z" fill="#141B38"/></svg>',
			'picture'               => '<svg viewBox="0 0 24 24" fill="none"><path d="M8.5 13.5L11 16.5L14.5 12L19 18H5L8.5 13.5ZM21 19V5C21 4.46957 20.7893 3.96086 20.4142 3.58579C20.0391 3.21071 19.5304 3 19 3H5C4.46957 3 3.96086 3.21071 3.58579 3.58579C3.21071 3.96086 3 4.46957 3 5V19C3 19.5304 3.21071 20.0391 3.58579 20.4142C3.96086 20.7893 4.46957 21 5 21H19C19.5304 21 20.0391 20.7893 20.4142 20.4142C20.7893 20.0391 21 19.5304 21 19Z"/></svg>',
			'caption'               => '<svg viewBox="0 0 24 24" fill="none"><path d="M5 3C3.89 3 3 3.89 3 5V19C3 20.11 3.89 21 5 21H19C20.11 21 21 20.11 21 19V5C21 3.89 20.11 3 19 3H5ZM5 5H19V19H5V5ZM7 7V9H17V7H7ZM7 11V13H17V11H7ZM7 15V17H14V15H7Z"/></svg>',
			'heart'                 => '<svg viewBox="0 0 24 24"><path d="M16.5 3C14.76 3 13.09 3.81 12 5.09C10.91 3.81 9.24 3 7.5 3C4.42 3 2 5.42 2 8.5C2 12.28 5.4 15.36 10.55 20.04L12 21.35L13.45 20.03C18.6 15.36 22 12.28 22 8.5C22 5.42 19.58 3 16.5 3ZM12.1 18.55L12 18.65L11.9 18.55C7.14 14.24 4 11.39 4 8.5C4 6.5 5.5 5 7.5 5C9.04 5 10.54 5.99 11.07 7.36H12.94C13.46 5.99 14.96 5 16.5 5C18.5 5 20 6.5 20 8.5C20 11.39 16.86 14.24 12.1 18.55Z"/></svg>',
			'sort'                  => '<svg viewBox="0 0 24 24"><path d="M7.73062 10.9999C7.51906 10.9999 7.40314 10.7535 7.53803 10.5906L11.8066 5.43267C11.9066 5.31186 12.0918 5.31186 12.1918 5.43267L16.4604 10.5906C16.5953 10.7535 16.4794 10.9999 16.2678 10.9999H7.73062Z" fill="#141B38"/><path d="M7.80277 13C7.58005 13 7.4685 13.2693 7.626 13.4268L11.8224 17.6232C11.9201 17.7209 12.0784 17.7209 12.176 17.6232L16.3724 13.4268C16.5299 13.2693 16.4184 13 16.1957 13H7.80277Z" fill="#141B38"/></svg>',
			'shop'                  => '<svg viewBox="0 0 24 24"><path d="M11 9H13V6H16V4H13V1H11V4H8V6H11V9ZM7 18C5.9 18 5.01 18.9 5.01 20C5.01 21.1 5.9 22 7 22C8.1 22 9 21.1 9 20C9 18.9 8.1 18 7 18ZM17 18C15.9 18 15.01 18.9 15.01 20C15.01 21.1 15.9 22 17 22C18.1 22 19 21.1 19 20C19 18.9 18.1 18 17 18ZM8.1 13H15.55C16.3 13 16.96 12.59 17.3 11.97L21.16 4.96L19.42 4L15.55 11H8.53L4.27 2H1V4H3L6.6 11.59L5.25 14.03C4.52 15.37 5.48 17 7 17H19V15H7L8.1 13Z" fill="#141B38"/></svg>',
			'headerUser'            => '<svg class="svg-inline--fa fa-user fa-w-16" style="margin-right: 3px;" aria-hidden="true" data-fa-processed="" data-prefix="fa" data-icon="user" role="presentation" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M96 160C96 71.634 167.635 0 256 0s160 71.634 160 160-71.635 160-160 160S96 248.366 96 160zm304 192h-28.556c-71.006 42.713-159.912 42.695-230.888 0H112C50.144 352 0 402.144 0 464v24c0 13.255 10.745 24 24 24h464c13.255 0 24-10.745 24-24v-24c0-61.856-50.144-112-112-112z"></path></svg>',
			'headerPhoto'           => '<svg class="svg-inline--fa fa-image fa-w-16" aria-hidden="true" data-fa-processed="" data-prefix="far" data-icon="image" role="presentation" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M464 448H48c-26.51 0-48-21.49-48-48V112c0-26.51 21.49-48 48-48h416c26.51 0 48 21.49 48 48v288c0 26.51-21.49 48-48 48zM112 120c-30.928 0-56 25.072-56 56s25.072 56 56 56 56-25.072 56-56-25.072-56-56-56zM64 384h384V272l-87.515-87.515c-4.686-4.686-12.284-4.686-16.971 0L208 320l-55.515-55.515c-4.686-4.686-12.284-4.686-16.971 0L64 336v48z"></path></svg>',
			'imageChooser'          => '<svg viewBox="0 0 18 18" fill="none"><path d="M2.16667 0.5C1.72464 0.5 1.30072 0.675595 0.988155 0.988155C0.675595 1.30072 0.5 1.72464 0.5 2.16667V13.8333C0.5 14.2754 0.675595 14.6993 0.988155 15.0118C1.30072 15.3244 1.72464 15.5 2.16667 15.5H9.74167C9.69167 15.225 9.66667 14.95 9.66667 14.6667C9.66667 14.1 9.76667 13.5333 9.95833 13H2.16667L5.08333 9.25L7.16667 11.75L10.0833 8L11.9417 10.475C12.75 9.95 13.7 9.66667 14.6667 9.66667C14.95 9.66667 15.225 9.69167 15.5 9.74167V2.16667C15.5 1.72464 15.3244 1.30072 15.0118 0.988155C14.6993 0.675595 14.2754 0.5 13.8333 0.5H2.16667ZM13.8333 11.3333V13.8333H11.3333V15.5H13.8333V18H15.5V15.5H18V13.8333H15.5V11.3333H13.8333Z"/></svg>',

			'usertimelineIcon'      => '<svg width="260" height="126" viewBox="0 0 260 126" fill="none" xmlns="http://www.w3.org/2000/svg"> <g clip-path="url(#usrtimlineclip0)"> <g filter="url(#usrtimlinefilter0_ddd)"> <g clip-path="url(#usrtimlineclip1)"> <rect x="64" y="23" width="131" height="113" rx="2" fill="white"/> <rect x="112.027" y="38" width="46" height="6" rx="1" fill="#DCDDE1"/> <rect x="112.027" y="49" width="28" height="6" rx="1" fill="#DCDDE1"/> <g clip-path="url(#usrtimlineclip2)"> <rect x="133.027" y="121" width="48" height="48" rx="1" fill="#F9BBA0"/> </g> <g clip-path="url(#usrtimlineclip3)"> <rect x="133.027" y="67" width="48" height="48" fill="#43A6DB"/> <circle cx="123.527" cy="101.5" r="40.5" fill="#86D0F9"/> </g> <g clip-path="url(#usrtimlineclip4)"> <rect x="79.0273" y="121" width="48" height="48" fill="#B6DDAD"/> </g> <g clip-path="url(#usrtimlineclip5)"> <rect x="79.0273" y="67" width="48" height="48" fill="#F6966B"/> <path d="M88.4756 84L126.476 122H57.0273L88.4756 84Z" fill="#F9BBA0"/> </g> <circle cx="92.0273" cy="45" r="10" fill="#DCDDE1"/> <circle cx="92.0273" cy="45" r="12" stroke="url(#usrtimlinepaint0_linear)"/> </g> </g> </g> <defs> <filter id="usrtimlinefilter0_ddd" x="51" y="16" width="157" height="139" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <linearGradient id="usrtimlinepaint0_linear" x1="88.5773" y1="78.9" x2="139.127" y2="27.3" gradientUnits="userSpaceOnUse"> <stop stop-color="white"/> <stop offset="0.147864" stop-color="#F6640E"/> <stop offset="0.443974" stop-color="#BA03A7"/> <stop offset="0.733337" stop-color="#6A01B9"/> <stop offset="1" stop-color="#6B01B9"/> </linearGradient> <clipPath id="usrtimlineclip0"> <rect width="259.056" height="126" fill="white"/> </clipPath> <clipPath id="usrtimlineclip1"> <rect x="64" y="23" width="131" height="113" rx="2" fill="white"/> </clipPath> <clipPath id="usrtimlineclip2"> <rect x="133.027" y="121" width="48" height="48" rx="1" fill="white"/> </clipPath> <clipPath id="usrtimlineclip3"> <rect x="133.027" y="67" width="48" height="48" rx="1" fill="white"/> </clipPath> <clipPath id="usrtimlineclip4"> <rect x="79.0273" y="121" width="48" height="48" rx="1" fill="white"/> </clipPath> <clipPath id="usrtimlineclip5"> <rect x="79.0273" y="67" width="48" height="48" rx="1" fill="white"/> </clipPath> </defs> </svg>',
			'publichashtagIcon'     => '<svg width="260" height="126" viewBox="0 0 260 126" fill="none" xmlns="http://www.w3.org/2000/svg"> <g clip-path="url(#hashtagiconclip0)"> <g filter="url(#hashtagiconfilter0_ddd)"> <rect x="119.327" y="12.3203" width="80" height="91" rx="2" transform="rotate(4 119.327 12.3203)" fill="white"/> </g> <g clip-path="url(#hashtagiconclip1)"> <rect x="119.327" y="12.3203" width="80" height="80" transform="rotate(4 119.327 12.3203)" fill="#0096CC"/> </g> <path d="M130.918 88.5016L128.971 88.3655L129.441 86.6958C129.453 86.6464 129.454 86.5951 129.444 86.5452C129.435 86.4954 129.414 86.4482 129.385 86.4069C129.355 86.3657 129.317 86.3313 129.273 86.3062C129.229 86.2811 129.18 86.2659 129.129 86.2616L128.427 86.2125C128.347 86.2049 128.265 86.2255 128.198 86.2709C128.131 86.3163 128.081 86.3837 128.058 86.4616L127.572 88.2676L125.678 88.1352L126.147 86.4654C126.159 86.4172 126.16 86.3671 126.151 86.3182C126.142 86.2694 126.123 86.223 126.095 86.182C126.067 86.1411 126.031 86.1066 125.988 86.0808C125.946 86.055 125.899 86.0384 125.849 86.0322L125.148 85.9832C125.067 85.9755 124.986 85.9962 124.918 86.0416C124.851 86.087 124.801 86.1544 124.778 86.2322L124.299 88.0388L122.194 87.8916C122.112 87.8842 122.03 87.9058 121.963 87.9526C121.895 87.9994 121.846 88.0684 121.824 88.1477L121.631 88.8392C121.617 88.89 121.614 88.9433 121.624 88.9953C121.633 89.0472 121.654 89.0964 121.685 89.1391C121.716 89.1819 121.756 89.2172 121.802 89.2424C121.848 89.2676 121.9 89.282 121.952 89.2846L123.899 89.4208L123.128 92.1867L121.023 92.0396C120.941 92.0322 120.859 92.0537 120.791 92.1005C120.724 92.1473 120.675 92.2164 120.653 92.2957L120.46 92.9871C120.446 93.038 120.443 93.0913 120.452 93.1432C120.462 93.1952 120.483 93.2443 120.513 93.2871C120.544 93.3299 120.584 93.3652 120.631 93.3904C120.677 93.4156 120.728 93.43 120.781 93.4326L122.742 93.5697L122.273 95.2394C122.26 95.2896 122.259 95.3419 122.269 95.3926C122.28 95.4432 122.301 95.491 122.332 95.5325C122.362 95.5741 122.402 95.6083 122.447 95.6328C122.493 95.6573 122.543 95.6715 122.595 95.6744L123.296 95.7234C123.375 95.7269 123.452 95.7041 123.516 95.6588C123.579 95.6135 123.626 95.5481 123.649 95.4731L124.142 93.6676L126.036 93.8L125.566 95.4698C125.555 95.5179 125.553 95.5681 125.562 95.617C125.571 95.6658 125.59 95.7122 125.618 95.7531C125.646 95.7941 125.683 95.8286 125.725 95.8544C125.767 95.8802 125.815 95.8968 125.864 95.903L126.566 95.952C126.647 95.9597 126.728 95.939 126.795 95.8936C126.862 95.8482 126.912 95.7808 126.935 95.703L127.432 93.8977L129.536 94.0448C129.618 94.0522 129.7 94.0306 129.768 93.9839C129.836 93.9371 129.885 93.868 129.907 93.7887L130.096 93.097C130.11 93.0462 130.113 92.9928 130.104 92.9409C130.094 92.889 130.073 92.8398 130.043 92.797C130.012 92.7542 129.972 92.719 129.925 92.6938C129.879 92.6686 129.828 92.6542 129.775 92.6515L127.818 92.5147L128.586 89.7485L130.69 89.8956C130.772 89.903 130.854 89.8814 130.922 89.8347C130.989 89.7879 131.039 89.7188 131.061 89.6395L131.253 88.948C131.268 88.8961 131.27 88.8414 131.26 88.7883C131.25 88.7353 131.228 88.6852 131.196 88.642C131.164 88.5989 131.122 88.5637 131.073 88.5394C131.025 88.515 130.972 88.5021 130.918 88.5016ZM126.414 92.4166L124.52 92.2841L125.292 89.5181L127.186 89.6506L126.414 92.4166Z" fill="#0068A0"/> <rect x="138.037" y="88.8115" width="29" height="7" rx="1" transform="rotate(4 138.037 88.8115)" fill="#86D0F9"/> <g filter="url(#hashtagiconfilter1_ddd)"> <rect x="119.327" y="12.3203" width="80" height="91" rx="2" transform="rotate(4 119.327 12.3203)" fill="white"/> </g> <g clip-path="url(#hashtagiconclip2)"> <rect x="119.327" y="12.3203" width="80" height="80" transform="rotate(4 119.327 12.3203)" fill="#0096CC"/> <circle cx="126.556" cy="44.5" r="46.5" fill="#0068A0"/> </g> <path d="M130.918 88.5016L128.971 88.3655L129.441 86.6958C129.453 86.6464 129.454 86.5951 129.444 86.5452C129.435 86.4954 129.414 86.4482 129.385 86.4069C129.355 86.3657 129.317 86.3313 129.273 86.3062C129.229 86.2811 129.18 86.2659 129.129 86.2616L128.427 86.2125C128.347 86.2049 128.265 86.2255 128.198 86.2709C128.131 86.3163 128.081 86.3837 128.058 86.4616L127.572 88.2676L125.678 88.1352L126.147 86.4654C126.159 86.4172 126.16 86.3671 126.151 86.3182C126.142 86.2694 126.123 86.223 126.095 86.182C126.067 86.1411 126.031 86.1066 125.988 86.0808C125.946 86.055 125.899 86.0384 125.849 86.0322L125.148 85.9832C125.067 85.9755 124.986 85.9962 124.918 86.0416C124.851 86.087 124.801 86.1544 124.778 86.2322L124.299 88.0388L122.194 87.8916C122.112 87.8842 122.03 87.9058 121.963 87.9526C121.895 87.9994 121.846 88.0684 121.824 88.1477L121.631 88.8392C121.617 88.89 121.614 88.9433 121.624 88.9953C121.633 89.0472 121.654 89.0964 121.685 89.1391C121.716 89.1819 121.756 89.2172 121.802 89.2424C121.848 89.2676 121.9 89.282 121.952 89.2846L123.899 89.4208L123.128 92.1867L121.023 92.0396C120.941 92.0322 120.859 92.0537 120.791 92.1005C120.724 92.1473 120.675 92.2164 120.653 92.2957L120.46 92.9871C120.446 93.038 120.443 93.0913 120.452 93.1432C120.462 93.1952 120.483 93.2443 120.513 93.2871C120.544 93.3299 120.584 93.3652 120.631 93.3904C120.677 93.4156 120.728 93.43 120.781 93.4326L122.742 93.5697L122.273 95.2394C122.26 95.2896 122.259 95.3419 122.269 95.3926C122.28 95.4432 122.301 95.491 122.332 95.5325C122.362 95.5741 122.402 95.6083 122.447 95.6328C122.493 95.6573 122.543 95.6715 122.595 95.6744L123.296 95.7234C123.375 95.7269 123.452 95.7041 123.516 95.6588C123.579 95.6135 123.626 95.5481 123.649 95.4731L124.142 93.6676L126.036 93.8L125.566 95.4698C125.555 95.5179 125.553 95.5681 125.562 95.617C125.571 95.6658 125.59 95.7122 125.618 95.7531C125.646 95.7941 125.683 95.8286 125.725 95.8544C125.767 95.8802 125.815 95.8968 125.864 95.903L126.566 95.952C126.647 95.9597 126.728 95.939 126.795 95.8936C126.862 95.8482 126.912 95.7808 126.935 95.703L127.432 93.8977L129.536 94.0448C129.618 94.0522 129.7 94.0306 129.768 93.9839C129.836 93.9371 129.885 93.868 129.907 93.7887L130.096 93.097C130.11 93.0462 130.113 92.9928 130.104 92.9409C130.094 92.889 130.073 92.8398 130.043 92.797C130.012 92.7542 129.972 92.719 129.925 92.6938C129.879 92.6686 129.828 92.6542 129.775 92.6515L127.818 92.5147L128.586 89.7485L130.69 89.8956C130.772 89.903 130.854 89.8814 130.922 89.8347C130.989 89.7879 131.039 89.7188 131.061 89.6395L131.253 88.948C131.268 88.8961 131.27 88.8414 131.26 88.7883C131.25 88.7353 131.228 88.6852 131.196 88.642C131.164 88.5989 131.122 88.5637 131.073 88.5394C131.025 88.515 130.972 88.5021 130.918 88.5016ZM126.414 92.4166L124.52 92.2841L125.292 89.5181L127.186 89.6506L126.414 92.4166Z" fill="#0068A0"/> <rect x="138.037" y="88.8115" width="29" height="7" rx="1" transform="rotate(4 138.037 88.8115)" fill="#86D0F9"/> <g filter="url(#hashtagiconfilter2_ddd)"> <rect x="65.0557" y="21" width="80" height="91" rx="2" fill="white"/> </g> <g clip-path="url(#hashtagiconclip3)"> <rect x="65.0557" y="21" width="80" height="80" fill="#F6966B"/> <path d="M80.8025 49.333L144.136 112.666H28.3887L80.8025 49.333Z" fill="#F9BBA0"/> </g> <path d="M81.9327 96.187H79.9812L80.3328 94.4887C80.3414 94.4386 80.3391 94.3873 80.3261 94.3382C80.313 94.2892 80.2894 94.2435 80.257 94.2044C80.2246 94.1653 80.1841 94.1337 80.1383 94.1118C80.0925 94.0898 80.0425 94.078 79.9917 94.0773H79.2885C79.2072 94.0753 79.1277 94.1015 79.0636 94.1515C78.9995 94.2015 78.9547 94.2722 78.9368 94.3515L78.5782 96.187H76.6794L77.031 94.4887C77.0395 94.4398 77.0376 94.3896 77.0253 94.3415C77.013 94.2934 76.9907 94.2484 76.9598 94.2095C76.9289 94.1707 76.8902 94.1388 76.8461 94.116C76.802 94.0932 76.7535 94.08 76.704 94.0773H76.0007C75.9194 94.0753 75.84 94.1015 75.7759 94.1515C75.7117 94.2015 75.6669 94.2722 75.6491 94.3515L75.2974 96.187H73.1877C73.1054 96.1854 73.0252 96.2126 72.9609 96.264C72.8967 96.3154 72.8525 96.3877 72.836 96.4683L72.6919 97.1716C72.6813 97.2233 72.6825 97.2767 72.6954 97.3278C72.7083 97.379 72.7325 97.4266 72.7662 97.4671C72.8 97.5076 72.8425 97.54 72.8905 97.5619C72.9385 97.5838 72.9908 97.5946 73.0435 97.5936H74.995L74.4184 100.407H72.3086C72.2263 100.405 72.1461 100.432 72.0818 100.484C72.0176 100.535 71.9734 100.607 71.957 100.688L71.8128 101.391C71.8022 101.443 71.8034 101.496 71.8163 101.547C71.8292 101.599 71.8534 101.646 71.8872 101.687C71.9209 101.727 71.9634 101.76 72.0114 101.782C72.0594 101.803 72.1117 101.814 72.1644 101.813H74.13L73.7784 103.512C73.7696 103.562 73.7722 103.615 73.7858 103.664C73.7995 103.714 73.824 103.761 73.8576 103.8C73.8912 103.839 73.933 103.87 73.9801 103.892C74.0272 103.913 74.0784 103.924 74.13 103.923H74.8333C74.9116 103.921 74.9869 103.893 75.0474 103.843C75.1079 103.793 75.1501 103.725 75.1673 103.649L75.533 101.813H77.4318L77.0802 103.512C77.0717 103.56 77.0736 103.611 77.0859 103.659C77.0982 103.707 77.1205 103.752 77.1514 103.791C77.1823 103.829 77.221 103.861 77.2651 103.884C77.3092 103.907 77.3577 103.92 77.4072 103.923H78.1105C78.1918 103.925 78.2712 103.899 78.3354 103.849C78.3995 103.799 78.4443 103.728 78.4621 103.649L78.8313 101.813H80.9411C81.0234 101.815 81.1036 101.788 81.1679 101.736C81.2321 101.685 81.2763 101.612 81.2928 101.532L81.4334 100.829C81.444 100.777 81.4428 100.723 81.4299 100.672C81.417 100.621 81.3928 100.574 81.359 100.533C81.3253 100.493 81.2828 100.46 81.2348 100.438C81.1868 100.416 81.1345 100.406 81.0818 100.407H79.1197L79.6928 97.5936H81.8026C81.8849 97.5952 81.9651 97.568 82.0294 97.5166C82.0936 97.4652 82.1378 97.3929 82.1543 97.3123L82.2984 96.609C82.3093 96.5561 82.3079 96.5014 82.2942 96.4492C82.2806 96.3969 82.2551 96.3485 82.2197 96.3077C82.1844 96.2669 82.1401 96.2348 82.0903 96.2139C82.0405 96.193 81.9866 96.1838 81.9327 96.187ZM77.7132 100.407H75.8143L76.391 97.5936H78.2898L77.7132 100.407Z" fill="#FE544F"/> <rect x="89.0557" y="96" width="29" height="7" rx="1" fill="#FCE1D5"/> <g filter="url(#hashtagiconfilter3_ddd)"> <rect x="65.0557" y="21" width="80" height="91" rx="2" fill="white"/> </g> <g clip-path="url(#hashtagiconclip4)"> <rect x="65.0557" y="21" width="80" height="80" fill="#F6966B"/> <path d="M80.8025 49.333L144.136 112.666H28.3887L80.8025 49.333Z" fill="#F9BBA0"/> </g> <path d="M81.9327 96.187H79.9812L80.3328 94.4887C80.3414 94.4386 80.3391 94.3873 80.3261 94.3382C80.313 94.2892 80.2894 94.2435 80.257 94.2044C80.2246 94.1653 80.1841 94.1337 80.1383 94.1118C80.0925 94.0898 80.0425 94.078 79.9917 94.0773H79.2885C79.2072 94.0753 79.1277 94.1015 79.0636 94.1515C78.9995 94.2015 78.9547 94.2722 78.9368 94.3515L78.5782 96.187H76.6794L77.031 94.4887C77.0395 94.4398 77.0376 94.3896 77.0253 94.3415C77.013 94.2934 76.9907 94.2484 76.9598 94.2095C76.9289 94.1707 76.8902 94.1388 76.8461 94.116C76.802 94.0932 76.7535 94.08 76.704 94.0773H76.0007C75.9194 94.0753 75.84 94.1015 75.7759 94.1515C75.7117 94.2015 75.6669 94.2722 75.6491 94.3515L75.2974 96.187H73.1877C73.1054 96.1854 73.0252 96.2126 72.9609 96.264C72.8967 96.3154 72.8525 96.3877 72.836 96.4683L72.6919 97.1716C72.6813 97.2233 72.6825 97.2767 72.6954 97.3278C72.7083 97.379 72.7325 97.4266 72.7662 97.4671C72.8 97.5076 72.8425 97.54 72.8905 97.5619C72.9385 97.5838 72.9908 97.5946 73.0435 97.5936H74.995L74.4184 100.407H72.3086C72.2263 100.405 72.1461 100.432 72.0818 100.484C72.0176 100.535 71.9734 100.607 71.957 100.688L71.8128 101.391C71.8022 101.443 71.8034 101.496 71.8163 101.547C71.8292 101.599 71.8534 101.646 71.8872 101.687C71.9209 101.727 71.9634 101.76 72.0114 101.782C72.0594 101.803 72.1117 101.814 72.1644 101.813H74.13L73.7784 103.512C73.7696 103.562 73.7722 103.615 73.7858 103.664C73.7995 103.714 73.824 103.761 73.8576 103.8C73.8912 103.839 73.933 103.87 73.9801 103.892C74.0272 103.913 74.0784 103.924 74.13 103.923H74.8333C74.9116 103.921 74.9869 103.893 75.0474 103.843C75.1079 103.793 75.1501 103.725 75.1673 103.649L75.533 101.813H77.4318L77.0802 103.512C77.0717 103.56 77.0736 103.611 77.0859 103.659C77.0982 103.707 77.1205 103.752 77.1514 103.791C77.1823 103.829 77.221 103.861 77.2651 103.884C77.3092 103.907 77.3577 103.92 77.4072 103.923H78.1105C78.1918 103.925 78.2712 103.899 78.3354 103.849C78.3995 103.799 78.4443 103.728 78.4621 103.649L78.8313 101.813H80.9411C81.0234 101.815 81.1036 101.788 81.1679 101.736C81.2321 101.685 81.2763 101.612 81.2928 101.532L81.4334 100.829C81.444 100.777 81.4428 100.723 81.4299 100.672C81.417 100.621 81.3928 100.574 81.359 100.533C81.3253 100.493 81.2828 100.46 81.2348 100.438C81.1868 100.416 81.1345 100.406 81.0818 100.407H79.1197L79.6928 97.5936H81.8026C81.8849 97.5952 81.9651 97.568 82.0294 97.5166C82.0936 97.4652 82.1378 97.3929 82.1543 97.3123L82.2984 96.609C82.3093 96.5561 82.3079 96.5014 82.2942 96.4492C82.2806 96.3969 82.2551 96.3485 82.2197 96.3077C82.1844 96.2669 82.1401 96.2348 82.0903 96.2139C82.0405 96.193 81.9866 96.1838 81.9327 96.187ZM77.7132 100.407H75.8143L76.391 97.5936H78.2898L77.7132 100.407Z" fill="#FE544F"/> <rect x="89.0557" y="96" width="29" height="7" rx="1" fill="#FCE1D5"/> </g> <defs> <filter id="hashtagiconfilter0_ddd" x="100.114" y="5.45508" width="111.884" height="122.09" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="hashtagiconfilter1_ddd" x="100.114" y="5.45508" width="111.884" height="122.09" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="hashtagiconfilter2_ddd" x="52.0557" y="14" width="106" height="117" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="hashtagiconfilter3_ddd" x="52.0557" y="14" width="106" height="117" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6"/> <feGaussianBlur stdDeviation="6.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3"/> <feGaussianBlur stdDeviation="3"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <clipPath id="hashtagiconclip0"> <rect width="259.056" height="126" fill="white" transform="translate(0.0556641)"/> </clipPath> <clipPath id="hashtagiconclip1"> <path d="M119.211 13.9829C119.275 13.0647 120.072 12.3724 120.99 12.4366L197.47 17.7846C198.388 17.8488 199.08 18.6452 199.016 19.5634L194.528 83.7401L114.723 78.1595L119.211 13.9829Z" fill="white"/> </clipPath> <clipPath id="hashtagiconclip2"> <path d="M119.211 13.9829C119.275 13.0647 120.072 12.3724 120.99 12.4366L197.47 17.7846C198.388 17.8488 199.08 18.6452 199.016 19.5634L194.528 83.7401L114.723 78.1595L119.211 13.9829Z" fill="white"/> </clipPath> <clipPath id="hashtagiconclip3"> <path d="M65.0557 22.6667C65.0557 21.7462 65.8019 21 66.7223 21H143.389C144.309 21 145.056 21.7462 145.056 22.6667V87H65.0557V22.6667Z" fill="white"/> </clipPath> <clipPath id="hashtagiconclip4"> <path d="M65.0557 22.6667C65.0557 21.7462 65.8019 21 66.7223 21H143.389C144.309 21 145.056 21.7462 145.056 22.6667V87H65.0557V22.6667Z" fill="white"/> </clipPath> </defs> </svg>',
			'taggedpostsIcon'       => '<svg width="260" height="126" viewBox="0 0 260 126" fill="none" xmlns="http://www.w3.org/2000/svg"> <g clip-path="url(#taggedpostclip0)"> <g filter="url(#taggedpostfilter0_ddd)"> <g clip-path="url(#taggedpostclip1)"> <rect x="104.316" y="29.0303" width="83.0697" height="84.1212" rx="2.10303" transform="rotate(2 104.316 29.0303)" fill="white"/> <g clip-path="url(#taggedpostclip2)"> <path d="M104.063 23.0957L188.133 26.0315L185.418 103.796L101.348 100.86L104.063 23.0957Z" fill="#59AB46"/> <path d="M119.756 48.194L183.987 117.073L62.3516 112.826L119.756 48.194Z" fill="#76C064"/> </g> <path fill-rule="evenodd" clip-rule="evenodd" d="M113.109 94.8001C114.187 94.6246 115.292 94.7726 116.286 95.2254C117.279 95.6782 118.116 96.4154 118.691 97.3439C119.265 98.2723 119.552 99.3503 119.513 100.441L119.485 101.259C119.467 101.783 119.241 102.278 118.858 102.635C118.474 102.993 117.964 103.183 117.441 103.165C116.917 103.147 116.422 102.921 116.064 102.538C115.997 102.466 115.937 102.391 115.882 102.311C115.342 102.804 114.63 103.067 113.899 103.041C113.158 103.016 112.458 102.697 111.953 102.155C111.447 101.613 111.178 100.892 111.204 100.151C111.23 99.4107 111.549 98.7106 112.091 98.2052C112.633 97.6998 113.353 97.4304 114.094 97.4562C114.834 97.4821 115.535 97.8011 116.04 98.3431C116.545 98.8851 116.815 99.6057 116.789 100.346L116.76 101.164C116.753 101.362 116.826 101.556 116.961 101.701C117.097 101.847 117.285 101.932 117.483 101.939C117.682 101.946 117.875 101.874 118.021 101.738C118.166 101.603 118.252 101.415 118.259 101.216L118.287 100.399C118.317 99.55 118.094 98.7115 117.647 97.9894C117.201 97.2673 116.55 96.6939 115.777 96.3417C115.004 95.9896 114.144 95.8745 113.306 96.011C112.468 96.1475 111.689 96.5295 111.068 97.1086C110.447 97.6878 110.012 98.4381 109.817 99.2647C109.622 100.091 109.677 100.957 109.975 101.752C110.272 102.548 110.799 103.237 111.488 103.733C112.177 104.23 112.998 104.51 113.846 104.54L113.847 104.54C114.6 104.567 115.347 104.395 116.011 104.04C116.31 103.881 116.682 103.994 116.841 104.293C117.001 104.591 116.888 104.963 116.589 105.123C115.733 105.579 114.772 105.801 113.803 105.766L113.825 105.153L113.804 105.766C113.803 105.766 113.803 105.766 113.803 105.766C112.712 105.728 111.657 105.367 110.771 104.729C109.885 104.091 109.208 103.205 108.825 102.182C108.443 101.159 108.373 100.046 108.623 98.9835C108.873 97.9208 109.433 96.956 110.231 96.2114C111.03 95.4668 112.031 94.9757 113.109 94.8001ZM115.563 100.304C115.577 99.888 115.426 99.4838 115.143 99.1798C114.859 98.8757 114.466 98.6967 114.051 98.6822C113.636 98.6677 113.231 98.8189 112.927 99.1024C112.623 99.3859 112.444 99.7786 112.43 100.194C112.415 100.61 112.566 101.014 112.85 101.318C113.133 101.622 113.526 101.801 113.942 101.815C114.357 101.83 114.761 101.679 115.065 101.395C115.369 101.112 115.548 100.719 115.563 100.304Z" fill="#59AB46"/> <rect x="126.717" y="97.5381" width="30.4939" height="7.3606" rx="1.05151" transform="rotate(2 126.717 97.5381)" fill="#B6DDAD"/> </g> </g> <g filter="url(#taggedpostfilter1_ddd)"> <g clip-path="url(#taggedpostclip3)"> <rect x="70.8867" y="10.8984" width="83.0697" height="84.1212" rx="2.10303" transform="rotate(-2 70.8867 10.8984)" fill="white"/> <g clip-path="url(#taggedpostclip4)"> <path d="M70.2217 4.99609L154.292 2.06031L157.007 79.825L72.9373 82.7608L70.2217 4.99609Z" fill="#43A6DB"/> <circle cx="169.299" cy="72.169" r="48.8954" transform="rotate(-2 169.299 72.169)" fill="#0068A0"/> </g> <path fill-rule="evenodd" clip-rule="evenodd" d="M84.2452 75.8962C85.308 75.646 86.4211 75.7165 87.4438 76.0989C88.4665 76.4813 89.3529 77.1583 89.9908 78.0444C90.6287 78.9305 90.9895 79.9859 91.0276 81.0771L91.0562 81.8944C91.0745 82.4183 90.8839 82.928 90.5264 83.3114C90.1689 83.6947 89.6738 83.9204 89.1499 83.9387C88.626 83.957 88.1163 83.7664 87.733 83.4089C87.6615 83.3423 87.5956 83.2709 87.5354 83.1954C87.0315 83.7253 86.3396 84.0368 85.6081 84.0623C84.8674 84.0882 84.1468 83.8188 83.6048 83.3134C83.0628 82.8079 82.7438 82.1079 82.7179 81.3673C82.6921 80.6266 82.9615 79.906 83.4669 79.364C83.9723 78.822 84.6724 78.503 85.413 78.4771C86.1537 78.4513 86.8742 78.7207 87.4162 79.2261C87.9583 79.7315 88.2773 80.4316 88.3031 81.1722L88.3317 81.9896C88.3386 82.1883 88.4242 82.3761 88.5696 82.5117C88.715 82.6473 88.9084 82.7196 89.1071 82.7126C89.3058 82.7057 89.4936 82.6201 89.6292 82.4747C89.7648 82.3293 89.8371 82.136 89.8301 81.9372L89.8016 81.1199C89.772 80.2712 89.4913 79.4504 88.9952 78.7612C88.499 78.072 87.8096 77.5454 87.0142 77.248C86.2188 76.9506 85.353 76.8957 84.5264 77.0904C83.6998 77.285 82.9495 77.7204 82.3703 78.3415C81.7912 78.9625 81.4092 79.7414 81.2727 80.5796C81.1362 81.4177 81.2513 82.2776 81.6034 83.0503C81.9556 83.8231 82.529 84.474 83.2511 84.9209C83.9733 85.3678 84.8117 85.5905 85.6604 85.5608L85.661 85.5608C86.4142 85.5352 87.147 85.3114 87.7851 84.9114C88.0721 84.7314 88.4506 84.8182 88.6306 85.1052C88.8105 85.3922 88.7237 85.7708 88.4367 85.9507C87.6149 86.466 86.6715 86.754 85.7026 86.7869L85.6818 86.1738L85.7032 86.7868C85.703 86.7868 85.7028 86.7869 85.7026 86.7869C84.6116 86.8248 83.5339 86.5385 82.6056 85.9641C81.6771 85.3895 80.9399 84.5526 80.4871 83.559C80.0344 82.5655 79.8864 81.46 80.0619 80.3824C80.2374 79.3047 80.7285 78.3033 81.4731 77.5048C82.2178 76.7063 83.1825 76.1465 84.2452 75.8962ZM87.0771 81.215C87.0626 80.7996 86.8836 80.4069 86.5796 80.1233C86.2755 79.8398 85.8713 79.6887 85.4558 79.7032C85.0403 79.7177 84.6476 79.8966 84.3641 80.2007C84.0806 80.5047 83.9294 80.909 83.944 81.3245C83.9585 81.7399 84.1374 82.1326 84.4415 82.4162C84.7455 82.6997 85.1498 82.8508 85.5652 82.8363C85.9807 82.8218 86.3734 82.6429 86.657 82.3388C86.9405 82.0348 87.0916 81.6305 87.0771 81.215Z" fill="#0068A0"/> <rect x="98.0117" y="77.6768" width="30.4939" height="7.3606" rx="1.05151" transform="rotate(-2 98.0117 77.6768)" fill="#86D0F9"/> </g> </g> </g> <defs> <filter id="taggedpostfilter0_ddd" x="87.7112" y="21.6697" width="113.294" height="114.308" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.30909"/> <feGaussianBlur stdDeviation="6.83485"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.05151"/> <feGaussianBlur stdDeviation="1.05151"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.15454"/> <feGaussianBlur stdDeviation="3.15454"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="taggedpostfilter1_ddd" x="57.217" y="0.638418" width="113.294" height="114.308" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.30909"/> <feGaussianBlur stdDeviation="6.83485"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.05151"/> <feGaussianBlur stdDeviation="1.05151"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.15454"/> <feGaussianBlur stdDeviation="3.15454"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <clipPath id="taggedpostclip0"> <rect width="259.056" height="126" fill="white" transform="translate(0.111328)"/> </clipPath> <clipPath id="taggedpostclip1"> <rect x="104.316" y="29.0303" width="83.0697" height="84.1212" rx="2.10303" transform="rotate(2 104.316 29.0303)" fill="white"/> </clipPath> <clipPath id="taggedpostclip2"> <path d="M104.187 19.5933C104.221 18.626 105.032 17.8692 106 17.903L186.567 20.7164C187.534 20.7502 188.291 21.5617 188.257 22.529L185.896 90.1353L101.826 87.1995L104.187 19.5933Z" fill="white"/> </clipPath> <clipPath id="taggedpostclip3"> <rect x="70.8867" y="10.8984" width="83.0697" height="84.1212" rx="2.10303" transform="rotate(-2 70.8867 10.8984)" fill="white"/> </clipPath> <clipPath id="taggedpostclip4"> <path d="M70.0983 1.49365C70.0645 0.526345 70.8213 -0.285196 71.7886 -0.318975L152.356 -3.13244C153.323 -3.16622 154.134 -2.40945 154.168 -1.44214L156.529 66.1641L72.4591 69.0999L70.0983 1.49365Z" fill="white"/> </clipPath> </defs> </svg>',
			'socialwall1Icon'       => '<svg width="260" height="126" viewBox="0 0 260 126" fill="none" xmlns="http://www.w3.org/2000/svg"> <g filter="url(#social1filter0_ddd)"> <rect x="44.416" y="44.9111" width="42" height="42" rx="2.10303" transform="rotate(-5 44.416 44.9111)" fill="white"/> <path d="M66.2979 54.0796C60.8188 54.559 56.7273 59.4241 57.2092 64.933C57.645 69.914 61.6528 73.7292 66.48 74.0598L65.8699 67.0864L63.3395 67.3078L63.0868 64.4188L65.6171 64.1974L65.4245 61.9959C65.2057 59.4954 66.5698 57.9908 68.8511 57.7912C69.9369 57.6962 71.0892 57.7861 71.0892 57.7861L71.3044 60.2467L70.0492 60.3565C68.8139 60.4646 68.4925 61.2657 68.5614 62.0527L68.7252 63.9255L71.4947 63.6832L71.2991 66.6114L68.978 66.8145L69.5881 73.7878C71.9031 73.2117 73.9359 71.827 75.3195 69.8835C76.7031 67.9401 77.3464 65.566 77.1331 63.1899C76.6512 57.681 71.777 53.6003 66.2979 54.0796Z" fill="#006BFA"/> </g> <g filter="url(#social1filter1_ddd)"> <rect x="83.0967" y="39.1279" width="42" height="42" rx="2.10303" transform="rotate(-3 83.0967 39.1279)" fill="white"/> <path d="M104.886 53.6171C101.89 53.7741 99.6299 56.3334 99.7844 59.2824C99.9414 62.2783 102.454 64.5406 105.45 64.3836C108.399 64.229 110.708 61.7141 110.551 58.7182C110.396 55.7691 107.835 53.4625 104.886 53.6171ZM105.352 62.5111C103.432 62.6117 101.76 61.1504 101.657 59.1843C101.556 57.2651 103.02 55.6394 104.986 55.5363C106.905 55.4357 108.531 56.8995 108.632 58.8188C108.735 60.7848 107.271 62.4105 105.352 62.5111ZM111.71 53.0717C111.673 52.3695 111.082 51.8372 110.38 51.874C109.678 51.9108 109.146 52.502 109.182 53.2041C109.219 53.9063 109.81 54.4386 110.512 54.4018C111.215 54.365 111.747 53.7738 111.71 53.0717ZM115.334 54.1491C115.152 52.4688 114.699 50.9905 113.418 49.8372C112.137 48.6839 110.62 48.3879 108.93 48.3826C107.193 48.3798 101.997 48.6521 100.27 48.8365C98.5894 49.0184 97.1579 49.469 95.9578 50.7523C94.8045 52.0331 94.5085 53.5507 94.5032 55.2408C94.5003 56.9777 94.7726 62.1737 94.957 63.9008C95.139 65.5811 95.5895 67.0126 96.8728 68.2127C98.2005 69.3635 99.6712 69.662 101.361 69.6673C103.098 69.6701 108.294 69.3978 110.021 69.2134C111.702 69.0315 113.18 68.5785 114.333 67.2976C115.484 65.97 115.783 64.4992 115.788 62.8091C115.791 61.0722 115.518 55.8762 115.334 54.1491ZM113.637 64.7525C113.358 65.7059 112.646 66.4473 111.776 66.8684C110.401 67.5037 107.117 67.535 105.619 67.6135C104.074 67.6945 100.805 68.0066 99.418 67.516C98.4621 67.1906 97.7232 66.5252 97.2996 65.6087C96.6667 64.2806 96.6354 60.9965 96.5545 59.4517C96.476 57.9538 96.1638 54.6844 96.652 53.2506C96.9798 52.3416 97.6452 51.6026 98.5618 51.1791C99.8899 50.5462 103.174 50.5149 104.719 50.4339C106.217 50.3554 109.486 50.0433 110.92 50.5314C111.826 50.8125 112.568 51.5247 112.989 52.3944C113.624 53.7693 113.656 57.0534 113.734 58.5514C113.815 60.0961 114.127 63.3655 113.637 64.7525Z" fill="url(#social1paint0_linear)"/> </g> <g filter="url(#social1filter2_ddd)"> <rect x="122.913" y="35.2803" width="42" height="42" rx="2.10303" transform="rotate(2 122.913 35.2803)" fill="white"/> <path d="M153.831 51.3695C153.049 51.6924 152.211 51.8933 151.348 51.9732C152.246 51.4743 152.955 50.6585 153.31 49.6603C152.463 50.131 151.531 50.4487 150.555 50.6147C149.795 49.7277 148.704 49.1892 147.444 49.1453C145.096 49.0633 143.11 50.9151 143.027 53.2836C143.015 53.6234 143.044 53.9546 143.103 54.2669C139.551 53.9627 136.443 52.1432 134.425 49.4811C134.033 50.0978 133.797 50.83 133.77 51.6095C133.718 53.0986 134.421 54.444 135.555 55.234C134.845 55.2093 134.192 54.9863 133.623 54.6663L133.622 54.6963C133.55 56.775 134.968 58.5656 136.913 59.0238C136.278 59.1739 135.617 59.1748 134.982 59.0264C135.224 59.8878 135.729 60.6518 136.428 61.2111C137.126 61.7703 137.982 62.0966 138.875 62.1441C137.318 63.2909 135.417 63.8738 133.485 63.797C133.145 63.7851 132.806 63.7533 132.467 63.7014C134.323 64.987 136.557 65.7755 138.976 65.8599C146.851 66.1349 151.407 59.75 151.605 54.0835C151.611 53.8936 151.617 53.7137 151.614 53.5235C152.475 52.9531 153.221 52.2187 153.831 51.3695Z" fill="#1B90EF"/> </g> <g filter="url(#social1filter3_ddd)"> <rect x="161.295" y="39.9297" width="42" height="42" rx="2.10303" transform="rotate(3 161.295 39.9297)" fill="white"/> <path d="M179.013 64.8913L184.352 62.167L179.327 58.8995L179.013 64.8913ZM190.966 57.677C191.072 58.1532 191.129 58.7871 191.147 59.5891C191.175 60.3917 191.169 61.0823 191.137 61.6815L191.153 62.5235C191.038 64.7105 190.794 66.3099 190.461 67.3238C190.164 68.2095 189.555 68.7583 188.643 68.9609C188.167 69.0661 187.303 69.111 185.982 69.1018C184.68 69.1037 183.49 69.0714 182.391 69.0138L180.8 68.9905C176.616 68.7712 174.018 68.4748 173.004 68.1413C172.119 67.8446 171.57 67.235 171.367 66.3231C171.262 65.847 171.205 65.2131 171.187 64.4111C171.159 63.6085 171.165 62.9179 171.196 62.3187L171.181 61.4767C171.295 59.2897 171.539 57.6903 171.873 56.6764C172.169 55.7907 172.779 55.2418 173.691 55.0393C174.167 54.9341 175.031 54.8892 176.352 54.8984C177.654 54.8965 178.844 54.9288 179.942 54.9864L181.533 55.0097C185.717 55.229 188.315 55.5254 189.329 55.8589C190.215 56.1556 190.764 56.7652 190.966 57.677Z" fill="#EB2121"/> </g> <defs> <filter id="social1filter0_ddd" x="30.7463" y="33.8904" width="72.8401" height="72.8401" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.30909"/> <feGaussianBlur stdDeviation="6.83485"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.05151"/> <feGaussianBlur stdDeviation="1.05151"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.15454"/> <feGaussianBlur stdDeviation="3.15454"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="social1filter1_ddd" x="69.427" y="29.5691" width="71.4799" height="71.4799" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.30909"/> <feGaussianBlur stdDeviation="6.83485"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.05151"/> <feGaussianBlur stdDeviation="1.05151"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.15454"/> <feGaussianBlur stdDeviation="3.15454"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="social1filter2_ddd" x="107.778" y="27.9197" width="70.7796" height="70.7796" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.30909"/> <feGaussianBlur stdDeviation="6.83485"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.05151"/> <feGaussianBlur stdDeviation="1.05151"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.15454"/> <feGaussianBlur stdDeviation="3.15454"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <filter id="social1filter3_ddd" x="145.427" y="32.5691" width="71.4799" height="71.4799" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.30909"/> <feGaussianBlur stdDeviation="6.83485"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.03 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.05151"/> <feGaussianBlur stdDeviation="1.05151"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.11 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.15454"/> <feGaussianBlur stdDeviation="3.15454"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect3_dropShadow" result="shape"/> </filter> <linearGradient id="social1paint0_linear" x1="103.683" y1="88.8048" x2="145.491" y2="41.4018" gradientUnits="userSpaceOnUse"> <stop stop-color="white"/> <stop offset="0.147864" stop-color="#F6640E"/> <stop offset="0.443974" stop-color="#BA03A7"/> <stop offset="0.733337" stop-color="#6A01B9"/> <stop offset="1" stop-color="#6B01B9"/> </linearGradient> </defs> </svg>',
			'publichashtagIconFree' => '<svg width="32" height="32" viewBox="0 0 32 32" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_free506_22296)"><g filter="url(#filter0free_d_506_22296)"><rect x="3" y="3" width="26.1602" height="26.1602" rx="2" fill="#EC352F"/></g><path d="M21.5738 12.7852H19.3434L19.7453 10.8442C19.7552 10.787 19.7525 10.7284 19.7376 10.6723C19.7226 10.6162 19.6957 10.564 19.6587 10.5194C19.6216 10.4747 19.5753 10.4386 19.523 10.4135C19.4707 10.3884 19.4135 10.3749 19.3555 10.374H18.5518C18.4589 10.3718 18.3681 10.4018 18.2948 10.4589C18.2215 10.5161 18.1703 10.5968 18.1499 10.6875L17.74 12.7852H15.5699L15.9718 10.8442C15.9815 10.7883 15.9793 10.731 15.9653 10.676C15.9513 10.621 15.9258 10.5696 15.8905 10.5252C15.8552 10.4808 15.8109 10.4444 15.7605 10.4183C15.7101 10.3923 15.6547 10.3772 15.5981 10.374H14.7943C14.7014 10.3718 14.6106 10.4018 14.5374 10.4589C14.4641 10.5161 14.4129 10.5968 14.3925 10.6875L13.9906 12.7852H11.5794C11.4854 12.7833 11.3937 12.8145 11.3203 12.8732C11.2469 12.9319 11.1963 13.0146 11.1776 13.1067L11.0128 13.9104C11.0007 13.9695 11.0021 14.0305 11.0168 14.089C11.0315 14.1474 11.0592 14.2018 11.0978 14.2482C11.1364 14.2945 11.1849 14.3315 11.2398 14.3565C11.2946 14.3815 11.3544 14.3939 11.4147 14.3927H13.645L12.9859 17.6076H10.5748C10.4807 17.6057 10.389 17.6368 10.3156 17.6956C10.2422 17.7543 10.1917 17.8369 10.1729 17.9291L10.0081 18.7328C9.99607 18.7919 9.99744 18.8529 10.0122 18.9114C10.0269 18.9698 10.0546 19.0242 10.0931 19.0705C10.1317 19.1168 10.1803 19.1539 10.2351 19.1789C10.29 19.2039 10.3497 19.2163 10.41 19.215H12.6564L12.2546 21.156C12.2445 21.2142 12.2474 21.2739 12.2631 21.3309C12.2787 21.3878 12.3067 21.4406 12.3451 21.4855C12.3834 21.5304 12.4312 21.5663 12.485 21.5906C12.5389 21.6149 12.5974 21.6271 12.6564 21.6262H13.4601C13.5496 21.6239 13.6357 21.5919 13.7049 21.5351C13.774 21.4783 13.8223 21.4001 13.8419 21.3128L14.2599 19.215H16.4299L16.0281 21.156C16.0183 21.2119 16.0205 21.2693 16.0346 21.3242C16.0486 21.3792 16.0741 21.4306 16.1094 21.475C16.1447 21.5194 16.189 21.5559 16.2394 21.5819C16.2898 21.608 16.3451 21.6231 16.4018 21.6262H17.2055C17.2984 21.6285 17.3892 21.5985 17.4625 21.5414C17.5358 21.4842 17.587 21.4034 17.6074 21.3128L18.0293 19.215H20.4405C20.5345 19.2169 20.6262 19.1858 20.6997 19.1271C20.7731 19.0683 20.8236 18.9857 20.8424 18.8935L21.0031 18.0898C21.0152 18.0308 21.0138 17.9697 20.9991 17.9113C20.9844 17.8528 20.9567 17.7984 20.9181 17.7521C20.8795 17.7058 20.831 17.6688 20.7762 17.6437C20.7213 17.6187 20.6615 17.6064 20.6013 17.6076H18.3589L19.0139 14.3927H21.4251C21.5191 14.3946 21.6108 14.3634 21.6842 14.3047C21.7577 14.2459 21.8082 14.1633 21.827 14.0712L21.9917 13.2675C22.0042 13.207 22.0025 13.1445 21.987 13.0848C21.9714 13.0251 21.9422 12.9698 21.9018 12.9231C21.8614 12.8765 21.8108 12.8398 21.7539 12.8159C21.697 12.792 21.6354 12.7815 21.5738 12.7852ZM16.7514 17.6076H14.5813L15.2404 14.3927H17.4105L16.7514 17.6076Z" fill="white"/></g><defs><filter id="filter0free_d_506_22296" x="-3" y="0" width="38.1602" height="38.1602" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="3"/><feGaussianBlur stdDeviation="3"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_506_22296"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_506_22296" result="shape"/></filter><clipPath id="clip0_free506_22296"><rect width="32" height="32" fill="white"/></clipPath></defs></svg>',
			'taggedpostsIconFree'   => '<svg width="33" height="32" viewBox="0 0 33 32" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0tagged_506_22299)"><g filter="url(#filter0_taggedd_506_22299)"><rect x="3.33398" y="3" width="26.1602" height="26.1602" rx="2" fill="#0096CC"/></g><path fill-rule="evenodd" clip-rule="evenodd" d="M15.0171 9.37969C16.3265 9.11924 17.6837 9.25292 18.9171 9.76381C20.1505 10.2747 21.2047 11.1399 21.9464 12.2499C22.6881 13.3599 23.084 14.665 23.084 16V17C23.084 17.6409 22.8294 18.2556 22.3762 18.7088C21.923 19.1621 21.3083 19.4167 20.6673 19.4167C20.0264 19.4167 19.4117 19.1621 18.9585 18.7088C18.874 18.6244 18.7965 18.5343 18.7261 18.4395C18.0878 19.0655 17.229 19.4167 16.334 19.4167C15.4278 19.4167 14.5588 19.0567 13.918 18.4159C13.2773 17.7752 12.9173 16.9062 12.9173 16C12.9173 15.0938 13.2773 14.2248 13.918 13.584C14.5588 12.9433 15.4278 12.5833 16.334 12.5833C17.2401 12.5833 18.1092 12.9433 18.7499 13.584C19.3907 14.2248 19.7507 15.0938 19.7507 16V17C19.7507 17.2431 19.8472 17.4763 20.0191 17.6482C20.191 17.8201 20.4242 17.9167 20.6673 17.9167C20.9104 17.9167 21.1436 17.8201 21.3155 17.6482C21.4874 17.4763 21.584 17.2431 21.584 17V16C21.584 14.9616 21.2761 13.9466 20.6992 13.0833C20.1223 12.2199 19.3024 11.547 18.3431 11.1496C17.3838 10.7523 16.3282 10.6483 15.3098 10.8509C14.2914 11.0534 13.3559 11.5535 12.6217 12.2877C11.8875 13.0219 11.3874 13.9574 11.1849 14.9758C10.9823 15.9942 11.0863 17.0498 11.4836 18.0091C11.881 18.9684 12.5539 19.7883 13.4172 20.3652C14.2806 20.9421 15.2956 21.25 16.334 21.25L16.3347 21.25C17.2562 21.2509 18.1612 21.0087 18.958 20.547C19.3164 20.3394 19.7753 20.4616 19.9829 20.82C20.1906 21.1784 20.0684 21.6373 19.71 21.845C18.6837 22.4395 17.5187 22.7512 16.3332 22.75L16.334 22V22.75C16.3337 22.75 16.3335 22.75 16.3332 22.75C14.9985 22.7498 13.6937 22.354 12.5839 21.6124C11.4739 20.8707 10.6087 19.8165 10.0978 18.5831C9.58691 17.3497 9.45324 15.9925 9.71369 14.6831C9.97414 13.3738 10.617 12.171 11.561 11.227C12.505 10.283 13.7078 9.64015 15.0171 9.37969ZM18.2507 16C18.2507 15.4917 18.0487 15.0042 17.6893 14.6447C17.3298 14.2853 16.8423 14.0833 16.334 14.0833C15.8257 14.0833 15.3381 14.2853 14.9787 14.6447C14.6193 15.0042 14.4173 15.4917 14.4173 16C14.4173 16.5083 14.6193 16.9958 14.9787 17.3553C15.3381 17.7147 15.8257 17.9167 16.334 17.9167C16.8423 17.9167 17.3298 17.7147 17.6893 17.3553C18.0487 16.9958 18.2507 16.5083 18.2507 16Z" fill="white"/></g><defs><filter id="filter0_taggedd_506_22299" x="-2.66602" y="0" width="38.1602" height="38.1602" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"><feFlood flood-opacity="0" result="BackgroundImageFix"/><feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/><feOffset dy="3"/><feGaussianBlur stdDeviation="3"/><feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.04 0"/><feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_506_22299"/><feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_506_22299" result="shape"/></filter><clipPath id="clip0tagged_506_22299"><rect width="32" height="32" fill="white" transform="translate(0.333984)"/></clipPath></defs></svg>',
			'socialwall1IconFree'   => '<svg width="33" height="32" viewBox="0 0 33 32" fill="none" xmlns="http://www.w3.org/2000/svg"> <g clip-path="url(#socialWallFreeclip0)"> <path d="M8.49935 2.19C5.29102 2.19 2.66602 4.80917 2.66602 8.035C2.66602 10.9517 4.80102 13.3725 7.58935 13.81V9.72667H6.10768V8.035H7.58935V6.74584C7.58935 5.28167 8.45852 4.47667 9.79435 4.47667C10.4302 4.47667 11.0952 4.5875 11.0952 4.5875V6.02834H10.3602C9.63685 6.02834 9.40935 6.4775 9.40935 6.93834V8.035H11.031L10.7685 9.72667H9.40935V13.81C10.7839 13.5929 12.0356 12.8916 12.9385 11.8325C13.8413 10.7735 14.3358 9.42663 14.3327 8.035C14.3327 4.80917 11.7077 2.19 8.49935 2.19Z" fill="#006BFA"/> <path d="M8.5 21.3047C7 21.3047 5.80469 22.5234 5.80469 24C5.80469 25.5 7 26.6953 8.5 26.6953C9.97656 26.6953 11.1953 25.5 11.1953 24C11.1953 22.5234 9.97656 21.3047 8.5 21.3047ZM8.5 25.7578C7.53906 25.7578 6.74219 24.9844 6.74219 24C6.74219 23.0391 7.51562 22.2656 8.5 22.2656C9.46094 22.2656 10.2344 23.0391 10.2344 24C10.2344 24.9844 9.46094 25.7578 8.5 25.7578ZM11.9219 21.2109C11.9219 20.8594 11.6406 20.5781 11.2891 20.5781C10.9375 20.5781 10.6562 20.8594 10.6562 21.2109C10.6562 21.5625 10.9375 21.8438 11.2891 21.8438C11.6406 21.8438 11.9219 21.5625 11.9219 21.2109ZM13.7031 21.8438C13.6562 21 13.4688 20.25 12.8594 19.6406C12.25 19.0312 11.5 18.8438 10.6562 18.7969C9.78906 18.75 7.1875 18.75 6.32031 18.7969C5.47656 18.8438 4.75 19.0312 4.11719 19.6406C3.50781 20.25 3.32031 21 3.27344 21.8438C3.22656 22.7109 3.22656 25.3125 3.27344 26.1797C3.32031 27.0234 3.50781 27.75 4.11719 28.3828C4.75 28.9922 5.47656 29.1797 6.32031 29.2266C7.1875 29.2734 9.78906 29.2734 10.6562 29.2266C11.5 29.1797 12.25 28.9922 12.8594 28.3828C13.4688 27.75 13.6562 27.0234 13.7031 26.1797C13.75 25.3125 13.75 22.7109 13.7031 21.8438ZM12.5781 27.0938C12.4141 27.5625 12.0391 27.9141 11.5938 28.1016C10.8906 28.3828 9.25 28.3125 8.5 28.3125C7.72656 28.3125 6.08594 28.3828 5.40625 28.1016C4.9375 27.9141 4.58594 27.5625 4.39844 27.0938C4.11719 26.4141 4.1875 24.7734 4.1875 24C4.1875 23.25 4.11719 21.6094 4.39844 20.9062C4.58594 20.4609 4.9375 20.1094 5.40625 19.9219C6.08594 19.6406 7.72656 19.7109 8.5 19.7109C9.25 19.7109 10.8906 19.6406 11.5938 19.9219C12.0391 20.0859 12.3906 20.4609 12.5781 20.9062C12.8594 21.6094 12.7891 23.25 12.7891 24C12.7891 24.7734 12.8594 26.4141 12.5781 27.0938Z" fill="url(#socialWallFreepaint0_linear)"/> <path d="M30.6018 4.50001C30.1526 4.70418 29.6684 4.83834 29.1668 4.90251C29.6801 4.59334 30.0768 4.10334 30.2634 3.51418C29.7793 3.80584 29.2426 4.01001 28.6768 4.12668C28.2159 3.62501 27.5684 3.33334 26.8334 3.33334C25.4626 3.33334 24.3426 4.45334 24.3426 5.83584C24.3426 6.03418 24.3659 6.22668 24.4068 6.40751C22.3301 6.30251 20.4809 5.30501 19.2501 3.79418C19.0343 4.16168 18.9118 4.59334 18.9118 5.04834C18.9118 5.91751 19.3493 6.68751 20.0259 7.12501C19.6118 7.12501 19.2268 7.00834 18.8884 6.83334V6.85084C18.8884 8.06418 19.7518 9.07918 20.8951 9.30668C20.528 9.40713 20.1427 9.42111 19.7693 9.34751C19.9277 9.84479 20.238 10.2799 20.6565 10.5917C21.0751 10.9035 21.5808 11.0763 22.1026 11.0858C21.2181 11.7861 20.1216 12.1646 18.9934 12.1592C18.7951 12.1592 18.5968 12.1475 18.3984 12.1242C19.5068 12.8358 20.8251 13.25 22.2368 13.25C26.8334 13.25 29.3593 9.43501 29.3593 6.12751C29.3593 6.01668 29.3593 5.91168 29.3534 5.80084C29.8434 5.45084 30.2634 5.00751 30.6018 4.50001Z" fill="#1B90EF"/> <path d="M23.3327 25.75L26.3602 24L23.3327 22.25V25.75ZM30.076 21.1825C30.1518 21.4567 30.2043 21.8242 30.2393 22.2908C30.2802 22.7575 30.2977 23.16 30.2977 23.51L30.3327 24C30.3327 25.2775 30.2393 26.2167 30.076 26.8175C29.9302 27.3425 29.5918 27.6808 29.0668 27.8267C28.7927 27.9025 28.291 27.955 27.521 27.99C26.7627 28.0308 26.0685 28.0483 25.4268 28.0483L24.4993 28.0833C22.0552 28.0833 20.5327 27.99 19.9318 27.8267C19.4068 27.6808 19.0685 27.3425 18.9227 26.8175C18.8468 26.5433 18.7943 26.1758 18.7593 25.7092C18.7185 25.2425 18.701 24.84 18.701 24.49L18.666 24C18.666 22.7225 18.7593 21.7833 18.9227 21.1825C19.0685 20.6575 19.4068 20.3192 19.9318 20.1733C20.206 20.0975 20.7077 20.045 21.4777 20.01C22.236 19.9692 22.9302 19.9517 23.5718 19.9517L24.4993 19.9167C26.9435 19.9167 28.466 20.01 29.0668 20.1733C29.5918 20.3192 29.9302 20.6575 30.076 21.1825Z" fill="#EB2121"/> </g> <defs> <linearGradient id="socialWallFreepaint0_linear" x1="6.97891" y1="38.843" x2="29.0945" y2="16.268" gradientUnits="userSpaceOnUse"> <stop stop-color="white"/> <stop offset="0.147864" stop-color="#F6640E"/> <stop offset="0.443974" stop-color="#BA03A7"/> <stop offset="0.733337" stop-color="#6A01B9"/> <stop offset="1" stop-color="#6B01B9"/> </linearGradient> <clipPath id="socialWallFreeclip0"> <rect width="32" height="32" fill="white" transform="translate(0.5)"/> </clipPath> </defs> </svg>',

			'user'                  => '<svg width="8" height="8" viewBox="0 0 8 8" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M4 0C4.53043 0 5.03914 0.210714 5.41421 0.585786C5.78929 0.960859 6 1.46957 6 2C6 2.53043 5.78929 3.03914 5.41421 3.41421C5.03914 3.78929 4.53043 4 4 4C3.46957 4 2.96086 3.78929 2.58579 3.41421C2.21071 3.03914 2 2.53043 2 2C2 1.46957 2.21071 0.960859 2.58579 0.585786C2.96086 0.210714 3.46957 0 4 0ZM4 5C6.21 5 8 5.895 8 7V8H0V7C0 5.895 1.79 5 4 5Z"/></svg>',
			'hashtag'               => '<svg viewBox="0 0 18 18" fill="none"><path d="M17.3607 4.1775H14.0152L14.618 1.266C14.6328 1.18021 14.6288 1.09223 14.6064 1.00812C14.5839 0.924001 14.5436 0.845742 14.488 0.778722C14.4324 0.711703 14.363 0.657514 14.2845 0.619882C14.206 0.582251 14.1203 0.56207 14.0332 0.560727H12.8276C12.6883 0.557321 12.5521 0.602311 12.4422 0.688037C12.3323 0.773763 12.2555 0.894929 12.2249 1.03091L11.61 4.1775H8.3549L8.9577 1.266C8.97229 1.18215 8.96897 1.09617 8.94795 1.0137C8.92692 0.931226 8.88867 0.854142 8.83572 0.787518C8.78276 0.720894 8.71629 0.666239 8.64069 0.62715C8.56509 0.588061 8.48207 0.565423 8.3971 0.560727H7.1915C7.05216 0.557321 6.91594 0.602311 6.80604 0.688037C6.69613 0.773763 6.61933 0.894929 6.58871 1.03091L5.98591 4.1775H2.36914C2.22811 4.17466 2.09056 4.22136 1.98042 4.30947C1.87028 4.39759 1.79452 4.52153 1.76634 4.65974L1.51919 5.86533C1.50109 5.95393 1.50315 6.04546 1.52522 6.13316C1.5473 6.22085 1.58882 6.30245 1.64671 6.37192C1.7046 6.44139 1.77737 6.49694 1.85965 6.53446C1.94192 6.57199 2.03158 6.59052 2.12199 6.58869H5.46751L4.47892 11.4111H0.862146C0.721125 11.4082 0.583571 11.4549 0.473429 11.543C0.363287 11.6311 0.287532 11.7551 0.259351 11.8933L0.0122042 13.0989C-0.00589975 13.1875 -0.00383898 13.279 0.0182337 13.3667C0.0403064 13.4544 0.0818254 13.536 0.139715 13.6055C0.197605 13.6749 0.270382 13.7305 0.352656 13.768C0.43493 13.8055 0.524592 13.8241 0.615 13.8222H3.98463L3.38183 16.7338C3.36677 16.821 3.37112 16.9106 3.39459 16.996C3.41806 17.0814 3.46006 17.1606 3.51761 17.2279C3.57517 17.2953 3.64685 17.3491 3.72757 17.3856C3.80829 17.4221 3.89606 17.4403 3.98463 17.439H5.19022C5.3244 17.4356 5.45359 17.3875 5.55732 17.3023C5.66105 17.2171 5.73339 17.0998 5.76288 16.9688L6.38979 13.8222H9.64488L9.04209 16.7338C9.02749 16.8176 9.03081 16.9036 9.05184 16.9861C9.07286 17.0685 9.11111 17.1456 9.16407 17.2122C9.21702 17.2789 9.28349 17.3335 9.35909 17.3726C9.43469 17.4117 9.51771 17.4343 9.60269 17.439H10.8083C10.9476 17.4424 11.0838 17.3974 11.1937 17.3117C11.3037 17.226 11.3805 17.1048 11.4111 16.9688L12.044 13.8222H15.6608C15.8018 13.8251 15.9394 13.7784 16.0495 13.6903C16.1596 13.6022 16.2354 13.4782 16.2636 13.34L16.5047 12.1344C16.5228 12.0458 16.5207 11.9543 16.4987 11.8666C16.4766 11.7789 16.4351 11.6973 16.3772 11.6278C16.3193 11.5584 16.2465 11.5028 16.1642 11.4653C16.082 11.4278 15.9923 11.4092 15.9019 11.4111H12.5383L13.5209 6.58869H17.1376C17.2787 6.59153 17.4162 6.54483 17.5264 6.45672C17.6365 6.36861 17.7123 6.24466 17.7404 6.10645L17.9876 4.90086C18.0063 4.8102 18.0038 4.71645 17.9804 4.62689C17.957 4.53733 17.9133 4.45436 17.8527 4.3844C17.7921 4.31445 17.7162 4.2594 17.6308 4.22352C17.5455 4.18764 17.4531 4.1719 17.3607 4.1775ZM10.1271 11.4111H6.87202L7.86061 6.58869H11.1157L10.1271 11.4111Z"/></svg>',
			'mention'               => '<svg viewBox="0 0 18 18"><path fill-rule="evenodd" clip-rule="evenodd" d="M7.24419 0.172937C8.99002 -0.174331 10.7996 0.00389957 12.4442 0.685088C14.0887 1.36628 15.4943 2.51983 16.4832 3.99987C17.4722 5.47992 18 7.21997 18 9.00001V10.3333C18 11.1879 17.6605 12.0075 17.0562 12.6118C16.452 13.2161 15.6324 13.5556 14.7778 13.5556C13.9232 13.5556 13.1036 13.2161 12.4993 12.6118C12.3867 12.4992 12.2833 12.3791 12.1896 12.2527C11.3384 13.0874 10.1933 13.5556 9.00001 13.5556C7.7918 13.5556 6.63307 13.0756 5.77874 12.2213C4.92441 11.3669 4.44445 10.2082 4.44445 9.00001C4.44445 7.7918 4.92441 6.63307 5.77874 5.77874C6.63307 4.92441 7.7918 4.44445 9.00001 4.44445C10.2082 4.44445 11.3669 4.92441 12.2213 5.77874C13.0756 6.63307 13.5556 7.7918 13.5556 9.00001V10.3333C13.5556 10.6575 13.6843 10.9684 13.9135 11.1976C14.1428 11.4268 14.4536 11.5556 14.7778 11.5556C15.1019 11.5556 15.4128 11.4268 15.642 11.1976C15.8712 10.9684 16 10.6575 16 10.3333V9.00001C16 7.61554 15.5895 6.26216 14.8203 5.11101C14.0511 3.95987 12.9579 3.06266 11.6788 2.53285C10.3997 2.00303 8.99224 1.86441 7.63437 2.13451C6.27651 2.4046 5.02922 3.07129 4.05026 4.05026C3.07129 5.02922 2.4046 6.27651 2.13451 7.63437C1.86441 8.99224 2.00303 10.3997 2.53285 11.6788C3.06266 12.9579 3.95987 14.0511 5.11101 14.8203C6.26216 15.5895 7.61554 16 9.00001 16L9.001 16C10.2297 16.0012 11.4363 15.6782 12.4987 15.0627C12.9766 14.7859 13.5884 14.9488 13.8653 15.4267C14.1421 15.9046 13.9792 16.5164 13.5013 16.7933C12.1329 17.586 10.5796 18.0016 8.99901 18L9.00001 17V18C8.99968 18 8.99934 18 8.99901 18C7.21933 17.9998 5.47964 17.472 3.99987 16.4832C2.51983 15.4943 1.36628 14.0887 0.685088 12.4442C0.00389957 10.7996 -0.17433 8.99002 0.172936 7.24419C0.520204 5.49836 1.37737 3.89472 2.63604 2.63604C3.89472 1.37737 5.49836 0.520204 7.24419 0.172937ZM11.5556 9.00001C11.5556 8.32223 11.2863 7.67221 10.8071 7.19295C10.3278 6.7137 9.67778 6.44445 9.00001 6.44445C8.32223 6.44445 7.67221 6.7137 7.19295 7.19295C6.7137 7.67221 6.44445 8.32223 6.44445 9.00001C6.44445 9.67778 6.7137 10.3278 7.19295 10.8071C7.67221 11.2863 8.32223 11.5556 9.00001 11.5556C9.67778 11.5556 10.3278 11.2863 10.8071 10.8071C11.2863 10.3278 11.5556 9.67778 11.5556 9.00001Z"/></svg>',
			'tooltipHelpSvg'        => '<svg width="20" height="21" viewBox="0 0 20 21" fill="#0068A0" xmlns="http://www.w3.org/2000/svg"><path d="M9.1665 8H10.8332V6.33333H9.1665V8ZM9.99984 17.1667C6.32484 17.1667 3.33317 14.175 3.33317 10.5C3.33317 6.825 6.32484 3.83333 9.99984 3.83333C13.6748 3.83333 16.6665 6.825 16.6665 10.5C16.6665 14.175 13.6748 17.1667 9.99984 17.1667ZM9.99984 2.16666C8.90549 2.16666 7.82186 2.38221 6.81081 2.801C5.79976 3.21979 4.8811 3.83362 4.10728 4.60744C2.54448 6.17024 1.6665 8.28986 1.6665 10.5C1.6665 12.7101 2.54448 14.8298 4.10728 16.3926C4.8811 17.1664 5.79976 17.7802 6.81081 18.199C7.82186 18.6178 8.90549 18.8333 9.99984 18.8333C12.21 18.8333 14.3296 17.9554 15.8924 16.3926C17.4552 14.8298 18.3332 12.7101 18.3332 10.5C18.3332 9.40565 18.1176 8.32202 17.6988 7.31097C17.28 6.29992 16.6662 5.38126 15.8924 4.60744C15.1186 3.83362 14.1999 3.21979 13.1889 2.801C12.1778 2.38221 11.0942 2.16666 9.99984 2.16666ZM9.1665 14.6667H10.8332V9.66666H9.1665V14.6667Z" fill="#0068A0"/></svg>',

			'shoppableDisabled'     => '<svg width="303" height="145" viewBox="0 0 303 145" fill="none" xmlns="http://www.w3.org/2000/svg"> <path d="M124.919 67.2058C130.919 72.7058 150.519 81.4058 180.919 72.2058" stroke="#8C8F9A" stroke-width="2" stroke-dasharray="3 3"/> <path d="M181.999 69L185.797 70.4241L183.5 74L181.999 69Z" fill="#8C8F9A"/> <g filter="url(#shopp_disabled_filter0_dddd)"> <rect x="24.6006" y="17.6504" width="81" height="98" rx="2" transform="rotate(-4 24.6006 17.6504)" fill="white"/> <rect x="24.3338" y="17.4184" width="81.5" height="98.5" rx="2.25" transform="rotate(-4 24.3338 17.4184)" stroke="url(#shopp_disabled_paint0_linear)" stroke-width="0.5"/> </g> <g clip-path="url(#shopp_disabled_clip0)"> <path d="M94.5298 21.3615C92.9088 21.4749 91.7091 22.8823 91.8207 24.478C91.9341 26.0991 93.3162 27.3005 94.9372 27.1872C96.5329 27.0756 97.7597 25.6917 97.6463 24.0707C97.5348 22.4749 96.1256 21.2499 94.5298 21.3615ZM94.8664 26.174C93.8279 26.2466 92.9083 25.471 92.8339 24.4072C92.7613 23.3687 93.5387 22.4744 94.6025 22.4C95.6409 22.3274 96.5352 23.1048 96.6079 24.1433C96.6822 25.2071 95.9048 26.1014 94.8664 26.174ZM98.2208 21.0016C98.1942 20.6217 97.869 20.339 97.4891 20.3656C97.1091 20.3921 96.8264 20.7173 96.853 21.0973C96.8796 21.4772 97.2048 21.7599 97.5847 21.7333C97.9646 21.7068 98.2473 21.3816 98.2208 21.0016ZM100.194 21.5509C100.079 20.6426 99.8198 19.8463 99.1152 19.2338C98.4106 18.6213 97.586 18.4753 96.6706 18.4884C95.7299 18.5033 92.9184 18.6999 91.9848 18.8161C91.0765 18.9305 90.3054 19.188 89.6676 19.8944C89.0551 20.599 88.9092 21.4237 88.9223 22.3391C88.9371 23.2798 89.1337 26.0913 89.2499 27.0249C89.3644 27.9332 89.6219 28.7042 90.3283 29.342C91.0582 29.9528 91.8575 30.1005 92.7729 30.0874C93.7136 30.0725 96.5251 29.8759 97.4587 29.7597C98.367 29.6453 99.1634 29.386 99.7759 28.6814C100.387 27.9515 100.534 27.1521 100.521 26.2367C100.506 25.296 100.31 22.4845 100.194 21.5509ZM99.3745 27.3096C99.2327 27.8285 98.854 28.2368 98.3869 28.4731C97.6483 28.8302 95.8699 28.8782 95.0594 28.9348C94.2236 28.9933 92.4559 29.1933 91.7001 28.9407C91.1793 28.7735 90.7728 28.4201 90.5348 27.9277C90.1795 27.2144 90.1315 25.4361 90.073 24.6002C90.0164 23.7897 89.8164 22.022 90.0672 21.2409C90.2362 20.7455 90.5895 20.339 91.082 20.1009C91.7952 19.7456 93.5736 19.6976 94.4094 19.6392C95.2199 19.5825 96.9876 19.3825 97.7687 19.6333C98.2624 19.777 98.6707 20.1557 98.9069 20.6228C99.264 21.3614 99.312 23.1397 99.3687 23.9502C99.4271 24.7861 99.6271 26.5538 99.3745 27.3096Z" fill="url(#shopp_disabled_paint1_linear)"/> </g> <g clip-path="url(#shopp_disabled_clip1)"> <rect x="26.1348" y="39.5967" width="81" height="76" rx="2" transform="rotate(-4 26.1348 39.5967)" fill="#B5E5FF"/> <circle cx="30.7388" cy="105.436" r="54" transform="rotate(-4 30.7388 105.436)" fill="#86D0F9"/> <g filter="url(#shopp_disabled_filter1_dd)"> <mask id="shopp_disabled_mask0" style="mask-type:alpha" maskUnits="userSpaceOnUse" x="35" y="47" width="60" height="54"> <path fill-rule="evenodd" clip-rule="evenodd" d="M68.7966 50.3478C68.534 50.4332 68.3943 50.7154 68.4401 50.9877C68.8644 53.5073 66.4327 56.3732 62.7333 57.5753C59.0338 58.7773 55.382 57.888 54.2442 55.6002C54.1213 55.3529 53.8423 55.2068 53.5797 55.2921L47.2555 57.347C47.1786 57.372 47.109 57.4152 47.0525 57.473L42.6186 62.008L35.8445 69.2862C35.7004 69.441 35.6693 69.6698 35.7668 69.8574L40.9681 79.8652C41.1015 80.1217 41.4239 80.212 41.6711 80.0621L47.8083 76.3393C48.0715 76.1797 48.4151 76.2935 48.5309 76.5788L58.2754 100.594C58.374 100.837 58.6437 100.963 58.8932 100.881L92.2457 90.0446C92.4952 89.9635 92.6396 89.7034 92.5765 89.4488L86.3412 64.2801C86.2678 63.9837 86.4749 63.6913 86.7789 63.6622L94.424 62.9299C94.7094 62.9026 94.9134 62.6414 94.8708 62.358L93.1967 51.2062C93.1647 50.9929 92.9995 50.8242 92.787 50.7877L82.5629 49.0293L76.3102 47.9666C76.2305 47.953 76.1488 47.959 76.0719 47.984L68.7966 50.3478Z" fill="white"/> </mask> <g mask="url(#shopp_disabled_mask0)"> <rect x="28.3076" y="60.3479" width="72" height="54" transform="rotate(-16 28.3076 60.3479)" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M66.4321 69.6639C65.1395 69.4776 63.7264 69.0512 62.5105 69.0127C63.1766 69.8427 63.7987 70.7521 64.429 71.6465C63.8884 72.3619 63.1987 72.9948 62.5553 73.6533C63.3952 74.1125 64.4294 74.2212 65.3292 74.5723C64.947 75.4717 64.0024 76.5635 63.9089 77.3062C65.0894 76.8017 66.445 76.1437 67.5698 75.7666C68.181 76.9532 68.7057 78.2958 69.3922 79.3464C69.485 77.6689 69.5124 75.9552 69.7351 74.3498C70.8246 74.4733 72.1524 74.6242 73.1713 74.589C72.2358 73.8444 71.3419 73.0247 70.4606 72.1824C71.1537 71.2976 71.8595 70.42 72.5116 69.5125C71.2887 69.7444 70.035 70.0316 68.7692 70.3408C68.2001 69.1068 67.8102 67.5497 67.1648 66.4536C66.98 67.5567 66.688 68.6002 66.4321 69.6639ZM70.0641 80.1946C70.0998 80.9132 70.6974 81.0407 70.7363 81.4713C70.1738 81.4417 69.7628 81.4615 69.1035 81.7558C68.9743 81.2243 69.4256 81.0144 69.1426 80.3976C61.5808 81.6649 57.7717 68.4365 64.8194 65.5342C73.6314 61.9053 78.4249 77.5439 70.0641 80.1946Z" fill="#FE544F"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M67.1649 66.4536C67.8103 67.5497 68.2003 69.1068 68.7693 70.3407C70.0352 70.0316 71.2888 69.7444 72.5117 69.5125C71.8597 70.42 71.1538 71.2976 70.4608 72.1824C71.3421 73.0248 72.2359 73.8444 73.1714 74.589C72.1526 74.6242 70.8247 74.4733 69.7352 74.3498C69.5126 75.9552 69.4852 77.6689 69.3924 79.3464C68.7058 78.2958 68.1811 76.9532 67.5699 75.7666C66.4451 76.1438 65.0896 76.8017 63.9091 77.3062C64.0026 76.5635 64.9472 75.4718 65.3294 74.5723C64.4295 74.2212 63.3954 74.1125 62.5555 73.6533C63.1989 72.9948 63.8885 72.362 64.4292 71.6465C63.7988 70.7521 63.1767 69.8427 62.5106 69.0128C63.7266 69.0512 65.1396 69.4776 66.4323 69.6639C66.6881 68.6002 66.9802 67.5567 67.1649 66.4536Z" fill="white"/> </g> </g> </g> <g filter="url(#shopp_disabled_filter2_dddd)"> <rect x="199.592" y="17.7058" width="79" height="102" rx="2" transform="rotate(4 199.592 17.7058)" fill="#E2F5FF"/> </g> <rect x="231.919" y="100.162" width="36" height="17" rx="2" transform="rotate(4 231.919 100.162)" fill="#0096CC"/> <path d="M241.707 111.873L244.07 112.038C245.123 112.112 245.827 111.602 245.887 110.743L245.888 110.736C245.931 110.112 245.469 109.576 244.827 109.497L244.831 109.432C245.358 109.397 245.785 108.978 245.821 108.453L245.822 108.446C245.875 107.686 245.328 107.182 244.346 107.113L242.051 106.953L241.707 111.873ZM243.95 107.973C244.376 108.003 244.61 108.232 244.586 108.579L244.585 108.586C244.561 108.931 244.281 109.123 243.824 109.091L243.162 109.045L243.241 107.923L243.95 107.973ZM243.859 109.858C244.377 109.894 244.652 110.136 244.624 110.538L244.623 110.545C244.594 110.958 244.295 111.166 243.777 111.13L243.02 111.077L243.109 109.805L243.859 109.858ZM248.86 112.507C250.155 112.597 251.031 111.925 251.108 110.824L251.334 107.602L250.086 107.515L249.869 110.617C249.829 111.19 249.498 111.51 248.935 111.47C248.376 111.431 248.09 111.069 248.13 110.496L248.347 107.393L247.099 107.306L246.874 110.528C246.796 111.633 247.581 112.417 248.86 112.507ZM253.583 112.703L254.834 112.791L254.952 111.1L256.873 107.989L255.539 107.896L254.448 109.838L254.383 109.833L253.565 107.758L252.232 107.665L253.701 111.012L253.583 112.703Z" fill="white"/> <g filter="url(#shopp_disabled_filter3_dd)"> <mask id="shopp_disabled_mask1" style="mask-type:alpha" maskUnits="userSpaceOnUse" x="207" y="35" width="61" height="48"> <path fill-rule="evenodd" clip-rule="evenodd" d="M244.802 36.7068C244.526 36.6972 244.298 36.9146 244.248 37.1861C243.785 39.699 240.52 41.5604 236.632 41.4246C232.745 41.2889 229.618 39.2042 229.331 36.6652C229.3 36.3908 229.088 36.1581 228.812 36.1485L222.166 35.9164C222.085 35.9136 222.005 35.9304 221.932 35.9653L216.215 38.7104L207.36 43.2328C207.171 43.329 207.064 43.5333 207.091 43.743L208.556 54.9261C208.594 55.2128 208.866 55.408 209.149 55.3516L216.19 53.9524C216.492 53.8924 216.776 54.117 216.787 54.4246L217.73 80.3242C217.74 80.5864 217.95 80.7966 218.212 80.8057L253.26 82.0296C253.522 82.0388 253.747 81.8438 253.774 81.5829L256.523 55.7995C256.556 55.4959 256.85 55.2919 257.146 55.3685L264.581 57.2952C264.858 57.3671 265.139 57.1915 265.196 56.9106L267.437 45.8588C267.48 45.6474 267.382 45.4324 267.195 45.3253L258.189 40.1762L252.677 37.039C252.607 36.999 252.528 36.9766 252.447 36.9738L244.802 36.7068Z" fill="white"/> </mask> <g mask="url(#shopp_disabled_mask1)"> <rect x="203.335" y="32.2556" width="72" height="54" transform="rotate(4 203.335 32.2556)" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M235.974 54.0491C234.823 53.4321 233.641 52.548 232.512 52.096C232.854 53.1038 233.128 54.171 233.414 55.2271C232.661 55.7145 231.797 56.0733 230.967 56.472C231.599 57.1908 232.534 57.6466 233.259 58.2843C232.592 58.9988 231.331 59.7017 230.99 60.3676C232.271 60.2973 233.77 60.1426 234.956 60.173C235.125 61.497 235.159 62.9381 235.444 64.1601C236.105 62.6156 236.717 61.0146 237.476 59.5821C238.457 60.0709 239.653 60.6668 240.623 60.9822C239.998 59.9626 239.439 58.8866 238.899 57.7936C239.852 57.1992 240.816 56.616 241.739 55.9862C240.511 55.7859 239.234 55.627 237.939 55.4846C237.826 54.1304 237.992 52.5338 237.761 51.2831C237.21 52.2564 236.579 53.1372 235.974 54.0491ZM235.786 65.187C235.573 65.8745 236.091 66.1987 235.981 66.6166C235.462 66.3964 235.069 66.2745 234.349 66.3255C234.409 65.7818 234.905 65.739 234.85 65.0626C227.311 63.6672 228.256 49.9337 235.871 49.6169C245.393 49.2208 244.549 65.5558 235.786 65.187Z" fill="#FE544F"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M237.761 51.283C237.993 52.5337 237.827 54.1303 237.939 55.4844C239.235 55.6268 240.511 55.7857 241.739 55.9861C240.816 56.6159 239.853 57.1991 238.899 57.7935C239.439 58.8865 239.998 59.9624 240.623 60.9821C239.653 60.6667 238.457 60.0708 237.476 59.582C236.717 61.0145 236.106 62.6155 235.445 64.16C235.159 62.938 235.125 61.4969 234.956 60.1729C233.77 60.1425 232.272 60.2972 230.99 60.3675C231.332 59.7016 232.593 58.9987 233.259 58.2842C232.534 57.6465 231.599 57.1907 230.967 56.4719C231.797 56.0732 232.662 55.7144 233.414 55.227C233.128 54.1709 232.854 53.1037 232.512 52.0959C233.642 52.5479 234.824 53.432 235.975 54.049C236.579 53.1371 237.21 52.2563 237.761 51.283Z" fill="white"/> </g> </g> <path d="M266.144 121.304L266.2 120.51L265.32 120.449L265.375 119.655L263.615 119.532L263.67 118.739L261.03 118.554L261.085 117.761L259.325 117.637L259.547 114.463L258.666 114.402L258.722 113.608L256.962 113.485L256.906 114.279L256.026 114.217L255.526 121.359L254.646 121.297L254.702 120.504L252.061 120.319L251.839 123.493L252.719 123.555L252.608 125.142L253.489 125.203L253.378 126.79L254.258 126.852L254.147 128.439L255.027 128.501L254.861 130.881L264.543 131.558L264.765 128.384L265.645 128.446L265.811 126.065L264.931 126.003L264.765 128.384L263.885 128.322L263.718 130.703L255.796 130.149L255.907 128.562L255.027 128.501L255.138 126.913L254.258 126.852L254.369 125.265L253.489 125.203L253.6 123.616L252.719 123.555L252.886 121.174L254.646 121.297L254.591 122.091L255.471 122.152L255.305 124.533L256.185 124.594L256.906 114.279L258.666 114.402L258.167 121.544L259.047 121.605L259.269 118.431L261.03 118.554L260.808 121.728L261.688 121.79L261.854 119.409L263.615 119.532L263.393 122.706L264.273 122.768L264.439 120.387L265.32 120.449L265.264 121.242L266.144 121.304L265.811 126.065L266.692 126.127L267.025 121.365L266.144 121.304Z" fill="#141B38"/> <path d="M264.932 126.003L265.812 126.065L266.145 121.304L265.265 121.242L265.32 120.449L264.44 120.387L264.274 122.768L263.393 122.706L263.615 119.532L261.855 119.409L261.688 121.79L260.808 121.728L261.03 118.554L259.27 118.431L259.048 121.605L258.168 121.543L258.667 114.402L256.907 114.279L256.185 124.594L255.305 124.533L255.471 122.152L254.591 122.091L254.647 121.297L252.886 121.174L252.72 123.555L253.6 123.616L253.489 125.203L254.369 125.265L254.258 126.852L255.139 126.913L255.028 128.5L255.908 128.562L255.797 130.149L263.719 130.703L263.885 128.322L264.765 128.384L264.932 126.003Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M258.001 123.924L258.881 123.986L258.62 127.726L257.739 127.665L258.001 123.924ZM260.641 124.109L259.761 124.047L259.5 127.788L260.38 127.85L260.641 124.109ZM262.402 124.232L261.521 124.17L261.26 127.911L262.14 127.973L262.402 124.232Z" fill="#141B38"/> <defs> <filter id="shopp_disabled_filter0_dddd" x="16.6698" y="10.1217" width="103.5" height="119.273" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="0.749837"/> <feGaussianBlur stdDeviation="0.468648"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.1137 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.80196"/> <feGaussianBlur stdDeviation="1.12623"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.0484671 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.39293"/> <feGaussianBlur stdDeviation="2.12058"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.06 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.05242"/> <feGaussianBlur stdDeviation="3.78276"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.0715329 0"/> <feBlend mode="normal" in2="effect3_dropShadow" result="effect4_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect4_dropShadow" result="shape"/> </filter> <filter id="shopp_disabled_filter1_dd" x="32.7109" y="44.9595" width="67.165" height="60.9465" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dx="1" dy="1"/> <feGaussianBlur stdDeviation="2"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.13 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="0.5"/> <feGaussianBlur stdDeviation="0.25"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.15 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow" result="shape"/> </filter> <filter id="shopp_disabled_filter2_dddd" x="185.046" y="16.3272" width="100.784" height="122.124" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="0.749837"/> <feGaussianBlur stdDeviation="0.468648"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.1137 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1.80196"/> <feGaussianBlur stdDeviation="1.12623"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.0484671 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="3.39293"/> <feGaussianBlur stdDeviation="2.12058"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.06 0"/> <feBlend mode="normal" in2="effect2_dropShadow" result="effect3_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="6.05242"/> <feGaussianBlur stdDeviation="3.78276"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0.101961 0 0 0 0 0.466667 0 0 0 0.0715329 0"/> <feBlend mode="normal" in2="effect3_dropShadow" result="effect4_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect4_dropShadow" result="shape"/> </filter> <filter id="shopp_disabled_filter3_dd" x="204.087" y="32.916" width="68.3604" height="54.114" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dx="1" dy="1"/> <feGaussianBlur stdDeviation="2"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.13 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="0.5"/> <feGaussianBlur stdDeviation="0.25"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.15 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow" result="shape"/> </filter> <linearGradient id="shopp_disabled_paint0_linear" x1="32.1943" y1="17.6504" x2="120.163" y2="93.7021" gradientUnits="userSpaceOnUse"> <stop stop-color="#B5CBEC"/> <stop offset="1" stop-color="#B6CFF4" stop-opacity="0.32"/> </linearGradient> <linearGradient id="shopp_disabled_paint1_linear" x1="94.2114" y1="40.43" x2="116.406" y2="14.3621" gradientUnits="userSpaceOnUse"> <stop stop-color="white"/> <stop offset="0.147864" stop-color="#F6640E"/> <stop offset="0.443974" stop-color="#BA03A7"/> <stop offset="0.733337" stop-color="#6A01B9"/> <stop offset="1" stop-color="#6B01B9"/> </linearGradient> <clipPath id="shopp_disabled_clip0"> <rect width="13" height="13" fill="white" transform="translate(87.7959 18.2437) rotate(-4)"/> </clipPath> <clipPath id="shopp_disabled_clip1"> <rect x="26.1348" y="39.5967" width="81" height="76" rx="2" transform="rotate(-4 26.1348 39.5967)" fill="white"/> </clipPath> </defs> </svg>',
			'shoppableEnabled'      => '<svg width="70" height="70" viewBox="0 0 70 70" fill="none" xmlns="http://www.w3.org/2000/svg"> <g filter="url(#shoppEnabled_filter0_dd)"> <rect x="5" y="1" width="60" height="60" rx="2" fill="white"/> </g> <path d="M19.904 26.2247L5 39.7857V59C5 60.1046 5.89543 61 7 61H63C64.1046 61 65 60.1046 65 59V45.5714L52.4342 31.4716C51.7591 30.7141 50.6236 30.5822 49.7928 31.1648L38.8105 38.8667C38.0444 39.4039 37.0082 39.3382 36.3161 38.7085L22.596 26.2247C21.833 25.5304 20.667 25.5304 19.904 26.2247Z" fill="url(#shoppEnabled_paint0_linear)"/> <rect x="29" y="4" width="29" height="20" rx="2" fill="#0068A0"/> <path d="M37.6002 14.0001C37.6002 12.8601 38.5268 11.9334 39.6668 11.9334H42.3335V10.6667H39.6668C38.7828 10.6667 37.9349 11.0179 37.3098 11.6431C36.6847 12.2682 36.3335 13.116 36.3335 14.0001C36.3335 14.8841 36.6847 15.732 37.3098 16.3571C37.9349 16.9822 38.7828 17.3334 39.6668 17.3334H42.3335V16.0667H39.6668C38.5268 16.0667 37.6002 15.1401 37.6002 14.0001ZM40.3335 14.6667H45.6668V13.3334H40.3335V14.6667ZM46.3335 10.6667H43.6668V11.9334H46.3335C47.4735 11.9334 48.4002 12.8601 48.4002 14.0001C48.4002 15.1401 47.4735 16.0667 46.3335 16.0667H43.6668V17.3334H46.3335C47.2176 17.3334 48.0654 16.9822 48.6905 16.3571C49.3156 15.732 49.6668 14.8841 49.6668 14.0001C49.6668 13.116 49.3156 12.2682 48.6905 11.6431C48.0654 11.0179 47.2176 10.6667 46.3335 10.6667Z" fill="white"/> <path d="M64.1103 30.0086V29.0938H63.0956V28.179H61.0662V27.2643H58.0221V26.3495H55.9926V22.6904H54.9779V21.7756H52.9485V22.6904H51.9338V30.9234H50.9191V30.0086H47.875V33.6677H48.8897V35.4972H49.9044V37.3268H50.9191V39.1563H51.9338V41.9006H63.0956V38.2415H64.1103V35.4972H63.0956V38.2415H62.0809V40.9859H52.9485V39.1563H51.9338V37.3268H50.9191V35.4972H49.9044V33.6677H48.8897V30.9234H50.9191V31.8381H51.9338V34.5825H52.9485V22.6904H54.9779V30.9234H55.9926V27.2643H58.0221V30.9234H59.0368V28.179H61.0662V31.8381H62.0809V29.0938H63.0956V30.0086H64.1103V35.4972H65.125V30.0086H64.1103Z" fill="#141B38"/> <path d="M63.096 35.4972H64.1107V30.0086H63.096V29.0938H62.0813V31.8382H61.0666V28.1791H59.0372V30.9234H58.0225V27.2643H55.9931V30.9234H54.9784V22.6904H52.949V34.5825H51.9343V31.8382H50.9195V30.9234H48.8901V33.6677H49.9048V35.4972H50.9195V37.3268H51.9343V39.1563H52.949V40.9859H62.0813V38.2416H63.096V35.4972Z" fill="white"/> <path fill-rule="evenodd" clip-rule="evenodd" d="M54.9785 33.668H55.9932V37.9805H54.9785V33.668ZM58.0224 33.668H57.0077V37.9805H58.0224V33.668ZM60.0516 33.668H59.0369V37.9805H60.0516V33.668Z" fill="#141B38"/> <defs> <filter id="shoppEnabled_filter0_dd" x="0" y="0" width="70" height="70" filterUnits="userSpaceOnUse" color-interpolation-filters="sRGB"> <feFlood flood-opacity="0" result="BackgroundImageFix"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="4"/> <feGaussianBlur stdDeviation="2.5"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/> <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow"/> <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha"/> <feOffset dy="1"/> <feGaussianBlur stdDeviation="1"/> <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.05 0"/> <feBlend mode="normal" in2="effect1_dropShadow" result="effect2_dropShadow"/> <feBlend mode="normal" in="SourceGraphic" in2="effect2_dropShadow" result="shape"/> </filter> <linearGradient id="shoppEnabled_paint0_linear" x1="35" y1="25" x2="35" y2="61" gradientUnits="userSpaceOnUse"> <stop stop-color="#DCDDE1"/> <stop offset="1" stop-color="#DCDDE1" stop-opacity="0"/> </linearGradient> </defs> </svg>',
			'ctaBoxes'              => array(
				'hashtag' => '<svg width="56" height="56" viewBox="0 0 56 56" fill="none" xmlns="http://www.w3.org/2000/svg"><rect width="28.1951" height="27" transform="translate(17.0762 16.2861) rotate(3)" fill="#8C8F9A"/><path d="M39.048 26.1275L35.707 25.9524L36.4614 23.0765C36.4807 22.9916 36.4813 22.9035 36.4633 22.8183C36.4453 22.7332 36.4091 22.6529 36.3571 22.5831C36.3051 22.5132 36.2386 22.4555 36.1622 22.4138C36.0858 22.3721 36.0012 22.3475 35.9144 22.3416L34.7104 22.2785C34.5715 22.2678 34.4331 22.3056 34.3188 22.3854C34.2046 22.4653 34.1216 22.5823 34.0839 22.7164L33.3052 25.8266L30.0545 25.6562L30.8089 22.7802C30.8279 22.6973 30.829 22.6112 30.8124 22.5278C30.7957 22.4443 30.7615 22.3653 30.7121 22.296C30.6627 22.2267 30.5992 22.1687 30.5258 22.1257C30.4523 22.0827 30.3706 22.0557 30.286 22.0466L29.082 21.9835C28.9431 21.9728 28.8047 22.0106 28.6904 22.0904C28.5762 22.1703 28.4931 22.2873 28.4555 22.4215L27.6888 25.5322L24.077 25.3429C23.9363 25.3327 23.7965 25.3721 23.6819 25.4544C23.5673 25.5366 23.4852 25.6564 23.4498 25.793L23.1399 26.984C23.1172 27.0715 23.1144 27.163 23.1319 27.2517C23.1493 27.3405 23.1865 27.4241 23.2407 27.4965C23.2949 27.5689 23.3646 27.6282 23.4448 27.67C23.525 27.7118 23.6136 27.735 23.704 27.7379L27.0449 27.913L25.8053 32.677L22.1935 32.4877C22.0528 32.4775 21.913 32.5169 21.7984 32.5991C21.6838 32.6814 21.6017 32.8012 21.5663 32.9377L21.2564 34.1287C21.2337 34.2163 21.2309 34.3078 21.2484 34.3965C21.2658 34.4852 21.303 34.5689 21.3572 34.6413C21.4114 34.7137 21.4811 34.773 21.5613 34.8148C21.6415 34.8565 21.7301 34.8797 21.8205 34.8826L25.1855 35.059L24.4311 37.935C24.4115 38.0213 24.4112 38.111 24.4302 38.1975C24.4491 38.284 24.4869 38.3653 24.5409 38.4356C24.5948 38.5058 24.6636 38.5633 24.7423 38.604C24.821 38.6447 24.9077 38.6675 24.9962 38.6708L26.2001 38.7339C26.3343 38.7375 26.4658 38.6962 26.5739 38.6166C26.6819 38.537 26.7603 38.4236 26.7966 38.2943L27.5874 35.1849L30.838 35.3552L30.0836 38.2312C30.0647 38.3142 30.0635 38.4002 30.0802 38.4837C30.0969 38.5671 30.131 38.6461 30.1804 38.7154C30.2298 38.7847 30.2933 38.8428 30.3668 38.8858C30.4402 38.9288 30.522 38.9557 30.6066 38.9648L31.8105 39.0279C31.9495 39.0386 32.0879 39.0008 32.2021 38.921C32.3163 38.8411 32.3994 38.7241 32.4371 38.5899L33.2338 35.4808L36.8456 35.6701C36.9863 35.6803 37.1261 35.6409 37.2407 35.5586C37.3553 35.4764 37.4375 35.3566 37.4729 35.2201L37.7767 34.0287C37.7995 33.9412 37.8022 33.8497 37.7847 33.761C37.7673 33.6722 37.7301 33.5886 37.6759 33.5162C37.6217 33.4438 37.552 33.3845 37.4718 33.3427C37.3916 33.3009 37.303 33.2777 37.2126 33.2748L33.8536 33.0988L35.0872 28.3344L38.6991 28.5237C38.8397 28.5339 38.9795 28.4945 39.0941 28.4123C39.2087 28.3301 39.2909 28.2102 39.3263 28.0737L39.6362 26.8827C39.6596 26.7931 39.662 26.6994 39.6434 26.6087C39.6247 26.5181 39.5854 26.4329 39.5285 26.3599C39.4716 26.2869 39.3987 26.2279 39.3154 26.1876C39.232 26.1473 39.1405 26.1268 39.048 26.1275ZM31.4458 32.9726L28.1951 32.8022L29.4347 28.0382L32.6854 28.2086L31.4458 32.9726Z" fill="white"/><rect x="10" y="10" width="30.1951" height="30" fill="#0096CC"/><path d="M32.4132 21.2806H29.4859L30.0134 18.733C30.0263 18.6579 30.0229 18.5809 30.0032 18.5073C29.9836 18.4337 29.9483 18.3653 29.8997 18.3066C29.851 18.248 29.7903 18.2006 29.7216 18.1676C29.6529 18.1347 29.5779 18.1171 29.5017 18.1159H28.4468C28.3249 18.1129 28.2057 18.1523 28.1096 18.2273C28.0134 18.3023 27.9462 18.4083 27.9194 18.5273L27.3814 21.2806H24.5332L25.0606 18.733C25.0734 18.6596 25.0705 18.5844 25.0521 18.5122C25.0337 18.4401 25.0002 18.3726 24.9539 18.3143C24.9076 18.256 24.8494 18.2082 24.7833 18.174C24.7171 18.1398 24.6445 18.12 24.5701 18.1159H23.5152C23.3933 18.1129 23.2741 18.1523 23.1779 18.2273C23.0818 18.3023 23.0146 18.4083 22.9878 18.5273L22.4603 21.2806H19.2957C19.1723 21.2781 19.0519 21.3189 18.9555 21.396C18.8591 21.4731 18.7929 21.5816 18.7682 21.7025L18.552 22.7574C18.5361 22.8349 18.5379 22.915 18.5572 22.9918C18.5765 23.0685 18.6129 23.1399 18.6635 23.2007C18.7142 23.2615 18.7779 23.3101 18.8498 23.3429C18.9218 23.3757 19.0003 23.3919 19.0794 23.3903H22.0067L21.1417 27.6099H17.977C17.8536 27.6074 17.7333 27.6483 17.6369 27.7254C17.5405 27.8025 17.4742 27.9109 17.4496 28.0319L17.2333 29.0868C17.2175 29.1643 17.2193 29.2444 17.2386 29.3211C17.2579 29.3978 17.2943 29.4692 17.3449 29.53C17.3956 29.5908 17.4592 29.6394 17.5312 29.6723C17.6032 29.7051 17.6817 29.7213 17.7608 29.7197H20.7092L20.1818 32.2673C20.1686 32.3437 20.1724 32.422 20.1929 32.4967C20.2135 32.5715 20.2502 32.6408 20.3006 32.6997C20.3509 32.7586 20.4137 32.8057 20.4843 32.8376C20.5549 32.8696 20.6317 32.8855 20.7092 32.8844H21.7641C21.8815 32.8814 21.9945 32.8393 22.0853 32.7648C22.1761 32.6902 22.2394 32.5876 22.2652 32.473L22.8137 29.7197H25.6619L25.1345 32.2673C25.1217 32.3406 25.1246 32.4159 25.143 32.488C25.1614 32.5602 25.1949 32.6277 25.2412 32.6859C25.2876 32.7442 25.3457 32.7921 25.4119 32.8263C25.478 32.8605 25.5507 32.8803 25.625 32.8844H26.6799C26.8018 32.8874 26.921 32.848 27.0172 32.773C27.1134 32.698 27.1806 32.592 27.2073 32.473L27.7612 29.7197H30.9258C31.0492 29.7222 31.1696 29.6813 31.266 29.6042C31.3623 29.5271 31.4286 29.4187 31.4533 29.2978L31.6643 28.2429C31.6801 28.1653 31.6783 28.0852 31.659 28.0085C31.6397 27.9318 31.6034 27.8604 31.5527 27.7996C31.502 27.7388 31.4384 27.6902 31.3664 27.6574C31.2944 27.6245 31.2159 27.6083 31.1368 27.6099H28.1937L29.0534 23.3903H32.2181C32.3415 23.3928 32.4618 23.352 32.5582 23.2749C32.6546 23.1978 32.7209 23.0893 32.7455 22.9684L32.9618 21.9135C32.9781 21.8342 32.976 21.7521 32.9555 21.6738C32.9351 21.5954 32.8968 21.5228 32.8438 21.4616C32.7907 21.4004 32.7243 21.3522 32.6496 21.3208C32.575 21.2894 32.4941 21.2757 32.4132 21.2806ZM26.0839 27.6099H23.2357L24.1007 23.3903H26.9489L26.0839 27.6099Z" fill="white"/><rect x="10" y="10" width="30.1951" height="30" stroke="white" stroke-width="2"/></svg>',
				'layout'  => '<svg width="56" height="56" viewBox="0 0 56 56" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M38.7033 21.3195C38.8925 21.1158 39.158 21 39.4361 21H52C52.5523 21 53 21.4477 53 22V45C53 45.5523 52.5523 46 52 46H28C27.4477 46 27 45.5523 27 45V29C27 28.4477 27.4477 28 28 28H32.0639C32.342 28 32.6075 27.8842 32.7967 27.6805L38.7033 21.3195Z" fill="white"/><rect x="5.05664" y="7.01074" width="28" height="28" rx="1" transform="rotate(-4 5.05664 7.01074)" fill="#8C8F9A"/><rect x="10.3242" y="10.6523" width="8" height="11" transform="rotate(-4 10.3242 10.6523)" fill="white"/><path d="M11.2305 23.6206L19.211 23.0626L19.6993 30.0455L11.7188 30.6036L11.2305 23.6206Z" fill="white"/><rect x="20.2988" y="9.95459" width="8" height="7" transform="rotate(-4 20.2988 9.95459)" fill="white"/><path d="M20.9277 18.9326L28.9082 18.3746L29.6756 29.3478L21.6951 29.9058L20.9277 18.9326Z" fill="white"/><rect x="21.0097" y="18.0572" width="28" height="28" rx="2" transform="rotate(4 21.0097 18.0572)" fill="#0096CC" stroke="white" stroke-width="2"/><path d="M28.0332 25.1357L41.0015 26.0426L40.0299 39.9372L27.0616 39.0304L28.0332 25.1357Z" fill="white"/><path d="M21.5488 24.6821L26.1804 25.006L25.2088 38.9006L20.5772 38.5768L21.5488 24.6821Z" fill="#94E3FF"/><path d="M42.8535 26.1719L47.4851 26.4957L46.5135 40.3904L41.8819 40.0665L42.8535 26.1719Z" fill="#94E3FF"/></svg>',
				'popups'  => '<svg width="56" height="56" viewBox="0 0 56 56" fill="none" xmlns="http://www.w3.org/2000/svg"><rect x="8.57617" y="14.6885" width="38" height="25" rx="1" transform="rotate(-2 8.57617 14.6885)" fill="#0096CC"/><path fill-rule="evenodd" clip-rule="evenodd" d="M25.4867 30.0589C25.5889 30.1678 25.7632 30.1634 25.8598 30.0495L32.6624 22.0206C32.7577 21.9081 32.9292 21.9021 33.0321 22.0077L42.8234 32.0504C42.975 32.2058 42.8701 32.4672 42.6532 32.4748L28.2271 32.9786L23.2301 33.1531L14.7862 33.4479C14.5716 33.4554 14.448 33.2066 14.5837 33.0401L20.7716 25.4492C20.8672 25.332 21.0442 25.3258 21.1477 25.4361L25.4867 30.0589Z" fill="white"/><path d="M5.57322 24.4268C5.73072 24.2693 6 24.3808 6 24.6036V31.3964C6 31.6192 5.73071 31.7307 5.57322 31.5732L2.17678 28.1768C2.07915 28.0791 2.07915 27.9209 2.17678 27.8232L5.57322 24.4268Z" fill="#434960"/><path d="M50.4268 21.4268C50.2693 21.2693 50 21.3808 50 21.6036V28.3964C50 28.6192 50.2693 28.7307 50.4268 28.5732L53.8232 25.1768C53.9209 25.0791 53.9209 24.9209 53.8232 24.8232L50.4268 21.4268Z" fill="#434960"/></svg>',
				'filter'  => '<svg width="56" height="56" viewBox="0 0 56 56" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_573_793)"><rect x="17.4102" y="10.708" width="31" height="31" rx="0.5" transform="rotate(5 17.4102 10.708)" fill="#8C8F9A"/><circle cx="43.2994" cy="36.061" r="17" transform="rotate(5 43.2994 36.061)" fill="#434960"/><rect x="15.4922" y="32.6245" width="31" height="9" transform="rotate(5 15.4922 32.6245)" fill="#E8E8EB"/></g><circle cx="17" cy="13" r="10" fill="#0096CC" stroke="white" stroke-width="2"/><rect x="12" y="10" width="10" height="1.5" rx="0.25" fill="white"/><path d="M13.5 13.25C13.5 13.1119 13.6119 13 13.75 13H20.25C20.3881 13 20.5 13.1119 20.5 13.25V14.25C20.5 14.3881 20.3881 14.5 20.25 14.5H13.75C13.6119 14.5 13.5 14.3881 13.5 14.25V13.25Z" fill="white"/><path d="M15.5 16.25C15.5 16.1119 15.6119 16 15.75 16H18.25C18.3881 16 18.5 16.1119 18.5 16.25V17.25C18.5 17.3881 18.3881 17.5 18.25 17.5H15.75C15.6119 17.5 15.5 17.3881 15.5 17.25V16.25Z" fill="white"/><defs><clipPath id="clip0_573_793"><rect x="17.4102" y="10.708" width="31" height="31" rx="0.5" transform="rotate(5 17.4102 10.708)" fill="white"/></clipPath></defs></svg>'

			),
			'camera' => '<svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M10.5007 3.5L8.36565 5.83333H4.66732C3.38398 5.83333 2.33398 6.88333 2.33398 8.16667V22.1667C2.33398 23.45 3.38398 24.5 4.66732 24.5H23.334C24.6173 24.5 25.6673 23.45 25.6673 22.1667V8.16667C25.6673 6.88333 24.6173 5.83333 23.334 5.83333H19.6357L17.5007 3.5H10.5007ZM14.0007 21C10.7807 21 8.16732 18.3867 8.16732 15.1667C8.16732 11.9467 10.7807 9.33333 14.0007 9.33333C17.2207 9.33333 19.834 11.9467 19.834 15.1667C19.834 18.3867 17.2207 21 14.0007 21Z" fill="#0096CC"/><path d="M14.0007 19.8333L15.459 16.625L18.6673 15.1667L15.459 13.7083L14.0007 10.5L12.5423 13.7083L9.33398 15.1667L12.5423 16.625L14.0007 19.8333Z" fill="#0096CC"/></svg>',
			'uploadFile' => '<svg width="12" height="14" viewBox="0 0 12 14" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M7.33268 0.333008H1.99935C1.26602 0.333008 0.672682 0.933008 0.672682 1.66634L0.666016 12.333C0.666016 13.0663 1.25935 13.6663 1.99268 13.6663H9.99935C10.7327 13.6663 11.3327 13.0663 11.3327 12.333V4.33301L7.33268 0.333008ZM9.99935 12.333H1.99935V1.66634H6.66602V4.99967H9.99935V12.333ZM3.33268 9.00634L4.27268 9.94634L5.33268 8.89301V11.6663H6.66602V8.89301L7.72602 9.95301L8.66602 9.00634L6.00602 6.33301L3.33268 9.00634Z" fill="#141B38"/></svg>',
			'addRoundIcon' =>'<svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M1.33333 8C1.33333 6.13333 2.4 4.53333 4 3.8V2.33333C1.66667 3.2 0 5.4 0 8C0 10.6 1.66667 12.8 4 13.6667V12.2C2.4 11.4667 1.33333 9.86667 1.33333 8ZM10 2C6.66667 2 4 4.66667 4 8C4 11.3333 6.66667 14 10 14C13.3333 14 16 11.3333 16 8C16 4.66667 13.3333 2 10 2ZM13.3333 8.66667H10.6667V11.3333H9.33333V8.66667H6.66667V7.33333H9.33333V4.66667H10.6667V7.33333H13.3333V8.66667Z" fill="#0068A0"/></svg>',
			'loaderSVG'    => '<svg version="1.1" id="loader-1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="20px" height="20px" viewBox="0 0 50 50" style="enable-background:new 0 0 50 50;" xml:space="preserve"><path fill="#fff" d="M43.935,25.145c0-10.318-8.364-18.683-18.683-18.683c-10.318,0-18.683,8.365-18.683,18.683h4.068c0-8.071,6.543-14.615,14.615-14.615c8.072,0,14.615,6.543,14.615,14.615H43.935z"><animateTransform attributeType="xml" attributeName="transform" type="rotate" from="0 25 25" to="360 25 25" dur="0.6s" repeatCount="indefinite"/></path></svg>'
		);
		return $builder_svg_icons;
	}

	public static function sb_other_plugins_modal() {
		check_ajax_referer( 'sbi_nonce', 'sbi_nonce' );

		if ( ! current_user_can( 'activate_plugins' ) || ! current_user_can( 'install_plugins' ) ) {
			wp_send_json_error();
		}

		$plugin           = isset( $_POST['plugin'] ) ? sanitize_key( $_POST['plugin'] ) : '';
		$sb_other_plugins = self::install_plugins_popup();
		$plugin           = isset( $sb_other_plugins[ $plugin ] ) ? $sb_other_plugins[ $plugin ] : false;
		if ( ! $plugin ) {
			wp_send_json_error();
		}

		// Build the content for modals
		$output = '<div class="sbi-fb-source-popup sbi-fb-popup-inside sbi-install-plugin-modal">
		<div class="sbi-fb-popup-cls"><svg width="14" height="14" viewBox="0 0 14 14" fill="none" xmlns="http://www.w3.org/2000/svg">
		<path d="M14 1.41L12.59 0L7 5.59L1.41 0L0 1.41L5.59 7L0 12.59L1.41 14L7 8.41L12.59 14L14 12.59L8.41 7L14 1.41Z" fill="#141B38"></path>
		</svg></div>
		<div class="sbi-install-plugin-body sbi-fb-fs">
		<div class="sbi-install-plugin-header">
		<div class="sb-plugin-image">' . $plugin['svgIcon'] . '</div>
		<div class="sb-plugin-name">
		<h3>' . $plugin['name'] . '<span>Free</span></h3>
		<p><span class="sb-author-logo">
		<svg width="13" height="17" viewBox="0 0 13 17" fill="none" xmlns="http://www.w3.org/2000/svg">
		<path fill-rule="evenodd" clip-rule="evenodd" d="M5.72226 4.70098C4.60111 4.19717 3.43332 3.44477 2.34321 3.09454C2.73052 4.01824 3.05742 5.00234 3.3957 5.97507C2.72098 6.48209 1.93286 6.8757 1.17991 7.30453C1.82065 7.93788 2.72809 8.3045 3.45109 8.85558C2.87196 9.57021 1.73414 10.3129 1.45689 10.9606C2.65579 10.8103 4.05285 10.5668 5.16832 10.5174C5.41343 11.7495 5.53984 13.1002 5.88845 14.2288C6.40758 12.7353 6.87695 11.192 7.49488 9.79727C8.44849 10.1917 9.61069 10.6726 10.5416 10.9052C9.88842 9.98881 9.29237 9.01536 8.71356 8.02465C9.57007 7.40396 10.4364 6.79309 11.2617 6.14122C10.0952 6.03375 8.88647 5.96834 7.66107 5.91968C7.46633 4.65567 7.5175 3.14579 7.21791 1.98667C6.76462 2.93671 6.2297 3.80508 5.72226 4.70098ZM6.27621 15.1705C6.12214 15.8299 6.62974 16.1004 6.55318 16.5C6.052 16.3273 5.67498 16.2386 5.00213 16.3338C5.02318 15.8194 5.48587 15.7466 5.3899 15.1151C-1.78016 14.3 -1.79456 1.34382 5.3345 0.546422C14.2483 -0.450627 14.528 14.9414 6.27621 15.1705Z" fill="#FE544F"></path><path fill-rule="evenodd" clip-rule="evenodd" d="M7.21769 1.98657C7.51728 3.1457 7.46611 4.65557 7.66084 5.91955C8.88625 5.96824 10.0949 6.03362 11.2615 6.14113C10.4362 6.79299 9.56984 7.40386 8.71334 8.02454C9.29215 9.01527 9.8882 9.98869 10.5414 10.9051C9.61046 10.6725 8.44827 10.1916 7.49466 9.79716C6.87673 11.1919 6.40736 12.7352 5.88823 14.2287C5.53962 13.1001 5.41321 11.7494 5.16809 10.5173C4.05262 10.5667 2.65558 10.8102 1.45666 10.9605C1.73392 10.3128 2.87174 9.57012 3.45087 8.85547C2.72786 8.30438 1.82043 7.93778 1.17969 7.30443C1.93264 6.8756 2.72074 6.482 3.39547 5.97494C3.05719 5.00224 2.73031 4.01814 2.34299 3.09445C3.43308 3.44467 4.60089 4.19707 5.72204 4.70088C6.22947 3.80499 6.7644 2.93662 7.21769 1.98657Z" fill="white"></path>
		</svg>
		</span>
		<span class="sb-author-name">' . $plugin['author'] . '</span>
		</p></div></div>
		<div class="sbi-install-plugin-content">
		<p>' . $plugin['description'] . '</p>';

		$plugin_install_data = array(
			'step'            => 'install',
			'action'          => 'sbi_install_addon',
			'nonce'           => wp_create_nonce( 'sbi-admin' ),
			'plugin'          => $plugin['plugin'],
			'download_plugin' => $plugin['download_plugin'],
		);

		if ( ! $plugin['installed'] ) {
			$output .= sprintf(
				"<button class='sbi-install-plugin-btn sbi-btn-orange' id='sbi_install_op_btn' data-plugin-atts='%s'>%s</button></div></div></div>",
				sbi_json_encode( $plugin_install_data ),
				__( 'Install', 'instagram-feed' )
			);
		}
		if ( $plugin['installed'] && ! $plugin['activated'] ) {
			$plugin_install_data['step']   = 'activate';
			$plugin_install_data['action'] = 'sbi_activate_addon';
			$output                       .= sprintf(
				"<button class='sbi-install-plugin-btn sbi-btn-orange' id='sbi_install_op_btn' data-plugin-atts='%s'>%s</button></div></div></div>",
				sbi_json_encode( $plugin_install_data ),
				__( 'Activate', 'instagram-feed' )
			);
		}
		if ( $plugin['installed'] && $plugin['activated'] ) {
			$output .= sprintf(
				"<button class='sbi-install-plugin-btn sbi-btn-orange' id='sbi_install_op_btn' disabled='disabled'>%s</button></div></div></div>",
				__( 'Plugin installed & activated', 'instagram-feed' )
			);
		}

		wp_send_json_success( ['output' => $output] );
		wp_die();
	}

	/**
	 * Plugins information for plugin install modal in all feeds page on select source flow
	 *
	 * @since 6.0
	 *
	 * @return array
	 */
	public static function install_plugins_popup() {
		// get the WordPress's core list of installed plugins
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$installed_plugins = get_plugins();

		$is_tiktok_installed = false;
		$tiktok_plugin       = 'feeds-for-tiktok/feeds-for-tiktok.php';
		if ( isset( $installed_plugins['tiktok-feeds-pro/tiktok-feeds-pro.php'] ) ) {
			$is_tiktok_installed = true;
			$tiktok_plugin       = 'tiktok-feeds-pro/tiktok-feeds-pro.php';
		} elseif ( isset( $installed_plugins['feeds-for-tiktok/feeds-for-tiktok.php'] ) ) {
			$is_tiktok_installed = true;
		}
		
		$is_reviews_installed = false;
		$reviews_plugin       = 'reviews-feed/sb-reviews.php';
		if ( isset( $installed_plugins['reviews-feed-pro/sb-reviews-pro.php'] ) ) {
			$is_reviews_installed = true;
			$reviews_plugin       = 'reviews-feed-pro/sb-reviews-pro.php';
		} elseif ( isset( $installed_plugins['reviews-feed/sb-reviews.php'] ) ) {
			$is_reviews_installed = true;
		}

		$is_facebook_installed = false;
		$facebook_plugin       = 'custom-facebook-feed/custom-facebook-feed.php';
		if ( isset( $installed_plugins['custom-facebook-feed-pro/custom-facebook-feed.php'] ) ) {
			$is_facebook_installed = true;
			$facebook_plugin       = 'custom-facebook-feed/custom-facebook-feed.php';
		} elseif ( isset( $installed_plugins['custom-facebook-feed/custom-facebook-feed.php'] ) ) {
			$is_facebook_installed = true;
		}

		$is_twitter_installed = false;
		$twitter_plugin       = 'custom-twitter-feeds/custom-twitter-feed.php';
		if ( isset( $installed_plugins['custom-twitter-feeds-pro/custom-twitter-feed.php'] ) ) {
			$is_twitter_installed = true;
			$twitter_plugin       = 'custom-twitter-feeds-pro/custom-twitter-feed.php';
		} elseif ( isset( $installed_plugins['custom-twitter-feeds/custom-twitter-feed.php'] ) ) {
			$is_twitter_installed = true;
		}

		$is_youtube_installed = false;
		$youtube_plugin       = 'feeds-for-youtube/youtube-feed.php';
		if ( isset( $installed_plugins['youtube-feed-pro/youtube-feed.php'] ) ) {
			$is_youtube_installed = true;
			$youtube_plugin       = 'youtube-feed-pro/youtube-feed.php';
		} elseif ( isset( $installed_plugins['feeds-for-youtube/youtube-feed.php'] ) ) {
			$is_youtube_installed = true;
		}

		$return = array(
			'reviews' => array(
				'displayName'         => __( 'Reviews', 'instagram-feed' ),
				'name'                => __( 'Reviews Feed', 'instagram-feed' ),
				'author'              => __( 'By Smash Balloon', 'instagram-feed' ),
				'description'         => __( 'To display a Reviews feed, our Reviews plugin is required. </br> Increase conversions and build positive brand trust through Google and Yelp reviews from your customers. Provide social proof needed to turn visitors into customers.', 'instagram-feed' ),
				'dashboard_permalink' => admin_url( 'admin.php?page=sbr' ),
				'svgIcon'             => '<svg width="33" height="33" viewBox="0 0 33 33" fill=none xmlns=http://www.w3.org/2000/svg xmlns:xlink=http://www.w3.org/1999/xlink><path d="M27.5674 3.146H6.23405C4.76738 3.146 3.56738 4.346 3.56738 5.81266V21.7626C3.56738 23.153 4.67729 24.2893 6.06737 24.3219L12.234 24.4793L15.6656 28.6857C16.0407 29.1456 16.7316 29.1794 17.1499 28.7583L21.4007 24.4793H27.5674C29.0341 24.4793 30.2341 23.2793 30.2341 21.8127V5.81266C30.2341 4.346 29.0341 3.146 27.5674 3.146Z" fill=url(#pattern0) /><path d="M27.5674 3.146H6.23405C4.76738 3.146 3.56738 4.346 3.56738 5.81266V21.7626C3.56738 23.153 4.67729 24.2893 6.06737 24.3219L12.234 24.4793L15.6656 28.6857C16.0407 29.1456 16.7316 29.1794 17.1499 28.7583L21.4007 24.4793H27.5674C29.0341 24.4793 30.2341 23.2793 30.2341 21.8127V5.81266C30.2341 4.346 29.0341 3.146 27.5674 3.146Z" fill=#FF611E /><path d="M16.6651 8.1315C16.7445 7.91124 17.056 7.91124 17.1354 8.1315L18.5296 11.9978C18.5644 12.0942 18.6544 12.1596 18.7568 12.1629L22.8647 12.2941C23.0988 12.3016 23.195 12.5978 23.0101 12.7414L19.7638 15.2621C19.6829 15.325 19.6485 15.4308 19.6771 15.5292L20.8217 19.4766C20.8869 19.7015 20.6349 19.8846 20.4412 19.7531L17.0407 17.4446C16.9559 17.3871 16.8446 17.3871 16.7599 17.4446L13.3594 19.7531C13.1657 19.8846 12.9136 19.7015 12.9789 19.4766L14.1235 15.5292C14.152 15.4308 14.1176 15.325 14.0367 15.2621L10.7905 12.7414C10.6055 12.5978 10.7018 12.3016 10.9358 12.2941L15.0437 12.1629C15.1461 12.1596 15.2361 12.0942 15.2709 11.9978L16.6651 8.1315Z" fill=white />
<defs><pattern id=pattern0 patternContentUnits=objectBoundingBox width=1 height=1><use xlink:href=#image0_5812_61143 transform="matrix(0.00161922 0 0 0.00166667 -0.228651 0)"/></pattern><image id=image0_5812_61143 width=900 height=600 xlink:href="data:image/png;base64,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"/></defs></svg>',
				'installed'           => $is_reviews_installed,
				'activated'           => is_plugin_active( $reviews_plugin ),
				'plugin'              => $reviews_plugin,
				'download_plugin'     => 'https://downloads.wordpress.org/plugin/reviews-feed.zip',
			),
			'facebook' => array(
				'displayName'         => __( 'Facebook', 'instagram-feed' ),
				'name'                => __( 'Facebook Feed', 'instagram-feed' ),
				'author'              => __( 'By Smash Balloon', 'instagram-feed' ),
				'description'         => __( 'To display a Facebook feed, our Facebook plugin is required. </br> It provides a clean and beautiful way to add your Facebook posts to your website. Grab your visitors attention and keep them engaged with your site longer.', 'instagram-feed' ),
				'dashboard_permalink' => admin_url( 'admin.php?page=cff-feed-builder' ),
				'svgIcon'             => '<svg viewBox="0 0 14 15"  width="36" height="36"><path d="M7.00016 0.860001C3.3335 0.860001 0.333496 3.85333 0.333496 7.54C0.333496 10.8733 2.7735 13.64 5.96016 14.14V9.47333H4.26683V7.54H5.96016V6.06667C5.96016 4.39333 6.9535 3.47333 8.48016 3.47333C9.20683 3.47333 9.96683 3.6 9.96683 3.6V5.24667H9.12683C8.30016 5.24667 8.04016 5.76 8.04016 6.28667V7.54H9.8935L9.5935 9.47333H8.04016V14.14C9.61112 13.8919 11.0416 13.0903 12.0734 11.88C13.1053 10.6697 13.6704 9.13043 13.6668 7.54C13.6668 3.85333 10.6668 0.860001 7.00016 0.860001Z" fill="rgb(0, 107, 250)"/></svg>',
				'installed'           => $is_facebook_installed,
				'activated'           => is_plugin_active( $facebook_plugin ),
				'plugin'              => $facebook_plugin,
				'download_plugin'     => 'https://downloads.wordpress.org/plugin/custom-facebook-feed.zip',
			),
			'twitter'  => array(
				'displayName'         => __( 'Twitter', 'instagram-feed' ),
				'name'                => __( 'Twitter Feed', 'instagram-feed' ),
				'author'              => __( 'By Smash Balloon', 'instagram-feed' ),
				'description'         => __( 'Custom Twitter Feeds is a highly customizable way to display tweets from your Twitter account. Promote your latest content and update your site content automatically.', 'instagram-feed' ),
				'dashboard_permalink' => admin_url( 'admin.php?page=custom-twitter-feeds' ),
				'svgIcon'             => '<svg width="36" height="36" viewBox="0 0 36 36" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M33.6905 9C32.5355 9.525 31.2905 9.87 30.0005 10.035C31.3205 9.24 32.3405 7.98 32.8205 6.465C31.5755 7.215 30.1955 7.74 28.7405 8.04C27.5555 6.75 25.8905 6 26.0005 6C20.4755 6 17.5955 8.88 17.5955 12.435C17.5955 12.945 17.6555 13.44 17.7605 13.905C12.4205 13.635 7.66555 11.07 4.50055 7.185C3.94555 8.13 3.63055 9.24 3.63055 10.41C3.63055 12.645 4.75555 14.625 6.49555 15.75C5.43055 15.75 4.44055 15.45 3.57055 15V15.045C3.57055 18.165 5.79055 20.775 8.73055 21.36C7.78664 21.6183 6.79569 21.6543 5.83555 21.465C6.24296 22.7437 7.04085 23.8626 8.11707 24.6644C9.19329 25.4662 10.4937 25.9105 11.8355 25.935C9.56099 27.7357 6.74154 28.709 3.84055 28.695C3.33055 28.695 2.82055 28.665 2.31055 28.605C5.16055 30.435 8.55055 31.5 12.1805 31.5C26.0005 31.5 30.4955 21.69 30.4955 13.185C30.4955 12.9 30.4955 12.63 30.4805 12.345C31.7405 11.445 32.8205 10.305 33.6905 9Z" fill="#1B90EF"/></svg>',
				'installed'           => $is_twitter_installed,
				'activated'           => is_plugin_active( $twitter_plugin ),
				'plugin'              => $twitter_plugin,
				'download_plugin'     => 'https://downloads.wordpress.org/plugin/custom-twitter-feeds.zip',
			),
			'youtube'  => array(
				'displayName'         => __( 'YouTube', 'instagram-feed' ),
				'name'                => __( 'Feeds for YouTube', 'instagram-feed' ),
				'author'              => __( 'By Smash Balloon', 'instagram-feed' ),
				'description'         => __( 'To display a YouTube feed, our YouTube plugin is required. It provides a simple yet powerful way to display videos from YouTube on your website, Increasing engagement with your channel while keeping visitors on your website.', 'instagram-feed' ),
				'dashboard_permalink' => admin_url( 'admin.php?page=youtube-feed' ),
				'svgIcon'             => '<svg width="36" height="36" viewBox="0 0 36 36" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M15 22.5L22.785 18L15 13.5V22.5ZM32.34 10.755C32.535 11.46 32.67 12.405 32.76 13.605C32.865 14.805 32.91 15.84 32.91 16.74L33 18C33 21.285 32.76 23.7 32.34 25.245C31.965 26.595 31.095 27.465 29.745 27.84C29.04 28.035 27.75 28.17 25.77 28.26C23.82 28.365 22.035 28.41 20.385 28.41L18 28.5C11.715 28.5 7.8 28.26 6.255 27.84C4.905 27.465 6.035 26.595 3.66 25.245C3.465 24.54 3.33 23.595 3.24 22.395C3.135 21.195 3.09 20.16 3.09 19.26L3 18C3 14.715 3.24 12.3 3.66 10.755C6.035 9.405 4.905 8.535 6.255 8.16C6.96 7.965 8.25 7.83 10.23 7.74C12.18 7.635 13.965 7.59 15.615 7.59L18 7.5C24.285 7.5 28.2 7.74 29.745 8.16C31.095 8.535 31.965 9.405 32.34 10.755Z" fill="#EB2121"/></svg>',
				'installed'           => $is_youtube_installed,
				'activated'           => is_plugin_active( $youtube_plugin ),
				'plugin'              => $youtube_plugin,
				'download_plugin'     => 'https://downloads.wordpress.org/plugin/feeds-for-youtube.zip',
			),
			'tiktok' => array(
				'displayName'         => __( 'TikTok', 'instagram-feed' ),
				'name'                => __( 'Feeds for TikTok', 'instagram-feed' ),
				'author'              => __( 'By Smash Balloon', 'instagram-feed' ),
				'description'         => __( "To display a TikTok feed, our TikTok plugin is required. It allows you to seamlessly integrate your TikTok account’s videos into your WordPress website.", 'instagram-feed' ),
				'dashboard_permalink' => admin_url( 'admin.php?page=sbtt' ),
				'svgIcon'             => '<svg width="41" height="40" fill="none" xmlns:xlink="http://www.w3.org/1999/xlink"><g clip-path="url(#a)"><path fill="url(#b)" d="M7.251 2.5h28.75v33.546H7.251z"/></g><defs><clipPath id="a"><path fill="#fff" transform="translate(.667)" d="M0 0h40v40H0z"/></clipPath><pattern id="b" patternContentUnits="objectBoundingBox" width="1" height="1"><use xlink:href="#c" transform="scale(.00091 .00078)"/></pattern><image id="c" width="1097" height="1280" xlink:href="data:image/png;base64,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"/></defs></svg>',
				'installed'           => $is_tiktok_installed,
				'activated'           => is_plugin_active( $tiktok_plugin ),
				'plugin'              => $tiktok_plugin,
				'download_plugin'     => "https://downloads.wordpress.org/plugin/feeds-for-tiktok.zip",
			)
		);

		if ( version_compare(PHP_VERSION, '7.1.0') < 0 ) {
			$incompatible_plugins = array('reviews', 'tiktok');
			foreach ( $incompatible_plugins as $plugin ) {
				if( isset( $return[$plugin] ) ) {
					unset( $return[$plugin] );
				}
			}
		}
		return $return;
	}

	/**
	 * Gets a list of info
	 * Used in multiple places in the feed creator
	 * Other Platforms + Social Links
	 * Upgrade links
	 *
	 * @return array
	 *
	 * @since 6.0
	 */
	public static function get_smashballoon_info() {
		$smash_info = array(
			'colorSchemes'   => array(
				'facebook'  => '#006BFA',
				'twitter'   => '#1B90EF',
				'instagram' => '#BA03A7',
				'youtube'   => '#EB2121',
				'linkedin'  => '#007bb6',
				'mail'      => '#666',
				'smash'     => '#EB2121'
			),
			'upgrade'        => array(
				'name' => __( 'Upgrade to Pro', 'instagram-feed' ),
				'icon' => 'instagram',
				'link' => 'https://smashballoon.com/instagram-feed/'
			),
			'platforms'      => array(
				array(
					'name' => __( 'Facebook Feed', 'instagram-feed' ),
					'icon' => 'facebook',
					'link' => 'https://smashballoon.com/custom-facebook-feed/?utm_campaign=instagram-free&utm_source=balloon&utm_medium=facebook'
				),
				array(
					'name' => __( 'Twitter Feed', 'instagram-feed' ),
					'icon' => 'twitter',
					'link' => 'https://smashballoon.com/custom-twitter-feeds/?utm_campaign=instagram-free&utm_source=balloon&utm_medium=twitter'
				),
				array(
					'name' => __( 'YouTube Feed', 'instagram-feed' ),
					'icon' => 'youtube',
					'link' => 'https://smashballoon.com/youtube-feed/?utm_campaign=instagram-free&utm_source=balloon&utm_medium=youtube'
				),
				array(
					'name' => __( 'Social Wall Plugin', 'instagram-feed' ),
					'icon' => 'smash',
					'link' => 'https://smashballoon.com/social-wall/?utm_campaign=instagram-free&utm_source=balloon&utm_medium=social-wall ',
				)
			),
			'socialProfiles' => array(
				'facebook' => 'https://www.facebook.com/SmashBalloon/',
				'twitter'  => 'https://twitter.com/smashballoon',
			),
			'morePlatforms'  => array( 'instagram', 'youtube', 'twitter' )
		);

		return $smash_info;
	}

	/**
	 * Text specific to onboarding. Will return an associative array 'active' => false
	 * if onboarding has been dismissed for the user or there aren't any legacy feeds.
	 *
	 * @return array
	 *
	 * @since 4.0
	 */
	public function get_onboarding_text() {
		// TODO: return if no legacy feeds
		$sbi_statuses_option = get_option( 'sbi_statuses', array() );

		if ( ! isset( $sbi_statuses_option['legacy_onboarding'] ) ) {
			return array( 'active' => false );
		}

		if ( $sbi_statuses_option['legacy_onboarding']['active'] === false
			 || self::onboarding_status() === 'dismissed' ) {
			return array( 'active' => false );
		}

		$type = $sbi_statuses_option['legacy_onboarding']['type'];

		$text = array(
			'active'      => true,
			'type'        => $type,
			'legacyFeeds' => array(
				'heading'     => __( 'Legacy Feed Settings', 'instagram-feed' ),
				'description' => sprintf( __( 'These settings will impact %1$s legacy feeds on your site. You can learn more about what legacy feeds are and how they differ from new feeds %2$shere%3$s.', 'instagram-feed' ), '<span class="cff-fb-count-placeholder"></span>', '<a href="https://smashballoon.com/doc/instagram-legacy-feeds/" target="_blank" rel="noopener">', '</a>' ),
			),
			'getStarted'  => __( 'You can now create and customize feeds individually. Click "Add New" to get started.', 'instagram-feed' ),
		);

		if ( $type === 'single' ) {
			$text['tooltips'] = array(
				array(
					'step'    => 1,
					'heading' => __( 'How you create a feed has changed', 'instagram-feed' ),
					'p'       => __( 'You can now create and customize feeds individually without using shortcode options.', 'instagram-feed' ) . ' ' . __( 'Click "Add New" to get started.', 'instagram-feed' ),
					'pointer' => 'top'
				),
				array(
					'step'    => 2,
					'heading' => __( 'Your existing feed is here', 'instagram-feed' ),
					'p'       => __( 'You can edit your existing feed from here, and all changes will only apply to this feed.', 'instagram-feed' ),
					'pointer' => 'top'
				)
			);
		} else {
			$text['tooltips'] = array(
				array(
					'step'    => 1,
					'heading' => __( 'How you create a feed has changed', 'instagram-feed' ),
					'p'       => __( 'You can now create and customize feeds individually without using shortcode options.', 'instagram-feed' ) . ' ' . __( 'Click "Add New" to get started.', 'instagram-feed' ),
					'pointer' => 'top'
				),
				array(
					'step'    => 2,
					'heading' => __( 'Your existing feeds are under "Legacy" feeds', 'instagram-feed' ),
					'p'       => __( 'You can edit the settings for any existing "legacy" feed (i.e. any feed created prior to this update) here.', 'instagram-feed' ) . ' ' . __( 'This works just like the old settings page and affects all legacy feeds on your site.', 'instagram-feed' )
				),
				array(
					'step'    => 3,
					'heading' => __( 'Existing feeds work as normal', 'instagram-feed' ),
					'p'       => __( 'You don\'t need to update or change any of your existing feeds. They will continue to work as usual.', 'instagram-feed' ) . ' ' . __( 'This update only affects how new feeds are created and customized.', 'instagram-feed' )
				)
			);
		}

		return $text;
	}

	public function get_customizer_onboarding_text() {

		if ( self::onboarding_status( 'customizer' ) === 'dismissed' ) {
			return array( 'active' => false );
		}

		$text = array(
			'active'   => true,
			'type'     => 'customizer',
			'tooltips' => array(
				array(
					'step'    => 1,
					'heading' => __( 'Embedding a Feed', 'instagram-feed' ),
					'p'       => __( 'After you are done customizing the feed, click here to add it to a page or a widget.', 'instagram-feed' ),
					'pointer' => 'top'
				),
				array(
					'step'    => 2,
					'heading' => __( 'Customize', 'instagram-feed' ),
					'p'       => __( 'Change your feed layout, color scheme, or customize individual feed sections here.', 'instagram-feed' ),
					'pointer' => 'top'
				),
				array(
					'step'    => 3,
					'heading' => __( 'Settings', 'instagram-feed' ),
					'p'       => __( 'Update your feed source, filter your posts, or change advanced settings here.', 'instagram-feed' ),
					'pointer' => 'top'
				)
			)
		);

		return $text;
	}

	/**
	 * Text related to the feed customizer
	 *
	 * @return array
	 *
	 * @since 6.0
	 */
	public function get_customize_screens_text() {
		$text = array(
			'common'              => array(
				'preview'       => __( 'Preview', 'instagram-feed' ),
				'help'          => __( 'Help', 'instagram-feed' ),
				'embed'         => __( 'Embed', 'instagram-feed' ),
				'save'          => __( 'Save', 'instagram-feed' ),
				'sections'      => __( 'Sections', 'instagram-feed' ),
				'enable'        => __( 'Enable', 'instagram-feed' ),
				'background'    => __( 'Background', 'instagram-feed' ),
				'text'          => __( 'Text', 'instagram-feed' ),
				'inherit'       => __( 'Inherit from Theme', 'instagram-feed' ),
				'size'          => __( 'Size', 'instagram-feed' ),
				'color'         => __( 'Color', 'instagram-feed' ),
				'height'        => __( 'Height', 'instagram-feed' ),
				'placeholder'   => __( 'Placeholder', 'instagram-feed' ),
				'select'        => __( 'Select', 'instagram-feed' ),
				'enterText'     => __( 'Enter Text', 'instagram-feed' ),
				'hoverState'    => __( 'Hover State', 'instagram-feed' ),
				'sourceCombine' => __( 'Combine sources from multiple platforms using our Social Wall plugin', 'instagram-feed' ),
			),

			'tabs'                => array(
				'customize' => __( 'Customize', 'instagram-feed' ),
				'settings'  => __( 'Settings', 'instagram-feed' ),
			),
			'overview'            => array(
				'feedLayout'  => __( 'Feed Layout', 'instagram-feed' ),
				'colorScheme' => __( 'Color Scheme', 'instagram-feed' ),
				'header'      => __( 'Header', 'instagram-feed' ),
				'posts'       => __( 'Posts', 'instagram-feed' ),
				'likeBox'     => __( 'Like Box', 'instagram-feed' ),
				'loadMore'    => __( 'Load More Button', 'instagram-feed' ),
			),
			'feedLayoutScreen'    => array(
				'layout'     => __( 'Layout', 'instagram-feed' ),
				'list'       => __( 'List', 'instagram-feed' ),
				'grid'       => __( 'Grid', 'instagram-feed' ),
				'masonry'    => __( 'Masonry', 'instagram-feed' ),
				'carousel'   => __( 'Carousel', 'instagram-feed' ),
				'feedHeight' => __( 'Feed Height', 'instagram-feed' ),
				'number'     => __( 'Number of Posts', 'instagram-feed' ),
				'columns'    => __( 'Columns', 'instagram-feed' ),
				'desktop'    => __( 'Desktop', 'instagram-feed' ),
				'tablet'     => __( 'Tablet', 'instagram-feed' ),
				'mobile'     => __( 'Mobile', 'instagram-feed' ),
				'bottomArea' => array(
					'heading'     => __( 'Tweak Post Styles', 'instagram-feed' ),
					'description' => __( 'Change post background, border radius, shadow etc.', 'instagram-feed' ),
				)
			),
			'colorSchemeScreen'   => array(
				'scheme'        => __( 'Scheme', 'instagram-feed' ),
				'light'         => __( 'Light', 'instagram-feed' ),
				'dark'          => __( 'Dark', 'instagram-feed' ),
				'custom'        => __( 'Custom', 'instagram-feed' ),
				'customPalette' => __( 'Custom Palette', 'instagram-feed' ),
				'background2'   => __( 'Background 2', 'instagram-feed' ),
				'text2'         => __( 'Text 2', 'instagram-feed' ),
				'link'          => __( 'Link', 'instagram-feed' ),
				'bottomArea'    => array(
					'heading'     => __( 'Overrides', 'instagram-feed' ),
					'description' => __( 'Colors that have been overridden from individual post element settings will not change. To change them, you will have to reset overrides.', 'instagram-feed' ),
					'ctaButton'   => __( 'Reset Overrides.', 'instagram-feed' ),
				)
			),
			'shoppableFeedScreen' => array(
				'heading1'     => __( 'Upgrade to Pro and make your Instagram Feed Shoppable', 'instagram-feed' ),
				'description1' => __( 'This feature links the post to the one specified in your caption.<br/><br/>Don’t want to add links to the caption? You can add links manually to each post.<br/><br><br>', 'instagram-feed' ),
				'heading2'     => __( 'Tap “Add” or “Update” on an<br/>image to add/update it’s URL', 'instagram-feed' ),

			)
		);

		$text['onboarding'] = $this->get_customizer_onboarding_text();

		return $text;
	}

	/**
	 * Returns an associate array of all existing sources along with their data
	 *
	 * @param int $page
	 *
	 * @return array
	 *
	 * @since 6.0
	 */

	public static function get_source_list( $page = 1 ) {
		$args['page'] = $page;
		$source_data  = SBI_Db::source_query( $args );
		$encryption   = new \SB_Instagram_Data_Encryption();

		$return = array();
		foreach ( $source_data as $source ) {
			$info                  = ! empty( $source['info'] ) ? json_decode( $encryption->decrypt( $source['info'] ), true ) : array();
			$source['header_data'] = $info;

			$settings = array( 'gdpr' => 'no' );

			$avatar = \SB_Instagram_Parse::get_avatar( $info, $settings );

			if ( \SB_Instagram_Connected_Account::local_avatar_exists( $source['username'] ) ) {
				$source['local_avatar_url'] = \SB_Instagram_Connected_Account::get_local_avatar_url( $source['username'] );
				$source['local_avatar']     = \SB_Instagram_Connected_Account::get_local_avatar_url( $source['username'] );
			} else {
				$source['local_avatar'] = false;
			}

			$source['avatar_url'] = is_bool( $avatar ) ? \SB_Instagram_Parse::get_avatar_url( $info, $settings ) : false;
			$source['just_added'] = ( ! empty( $_GET['sbi_username'] ) && isset( $info['username'] ) && $info['username'] === $_GET['sbi_username'] );

			$source['error_encryption'] = false;
			if ( isset( $source['access_token'] ) && strpos( $source['access_token'], 'IG' ) === false && strpos( $source['access_token'], 'EA' ) === false && ! $encryption->decrypt( $source['access_token'] ) ) {
				$source['error_encryption'] = true;
			}
			$return[] = $source;
		}

		return $return;
	}

	/**
	 * Get Links with UTM
	 *
	 * @return array
	 *
	 * @since 4.0
	 */
	public static function get_links_with_utm() {
		$license_key = null;
		if ( get_option( 'sbi_license_key' ) ) {
			$license_key = get_option( 'sbi_license_key' );
		}
		$all_access_bundle       = sprintf( 'https://smashballoon.com/all-access/?license_key=%s&upgrade=true&utm_campaign=instagram-free&utm_source=all-feeds&utm_medium=footer-banner&utm_content=learn-more', $license_key );
		$all_access_bundle_popup = sprintf( 'https://smashballoon.com/all-access/?license_key=%s&upgrade=true&utm_campaign=instagram-free&utm_source=balloon&utm_medium=all-access', $license_key );
		$sourceCombineCTA        = sprintf( 'https://smashballoon.com/social-wall/?license_key=%s&upgrade=true&utm_campaign=instagram-free&utm_source=customizer&utm_medium=sources&utm_content=social-wall', $license_key );

		return array(
			'allAccessBundle'  => $all_access_bundle,
			'popup'            => array(
				'allAccessBundle' => $all_access_bundle_popup,
				'fbProfile'       => 'https://www.facebook.com/SmashBalloon/',
				'twitterProfile'  => 'https://twitter.com/smashballoon',
			),
			'sourceCombineCTA' => $sourceCombineCTA,
			'multifeedCTA'     => 'https://smashballoon.com/extensions/multifeed/?utm_campaign=instagram-free&utm_source=customizer&utm_medium=sources&utm_content=multifeed',
			'doc'              => 'https://smashballoon.com/docs/instagram/?utm_campaign=instagram-free&utm_source=support&utm_medium=view-documentation-button&utm_content=view-documentation',
			'blog'             => 'https://smashballoon.com/blog/?utm_campaign=instagram-free&utm_source=support&utm_medium=view-blog-button&utm_content=view-blog',
			'gettingStarted'   => 'https://smashballoon.com/docs/getting-started/?instagram&utm_campaign=instagram-free&utm_source=support&utm_medium=getting-started-button&utm_content=getting-started',
		);
	}

	public static function get_social_wall_links() {
		return array(
			'<a href="' . esc_url( admin_url( 'admin.php?page=sbi-feed-builder' ) ) . '">' . __( 'All Feeds', 'instagram-feed' ) . '</a>',
			'<a href="' . esc_url( admin_url( 'admin.php?page=sbi-settings' ) ) . '">' . __( 'Settings', 'instagram-feed' ) . '</a>',
			'<a href="' . esc_url( admin_url( 'admin.php?page=sbi-oembeds-manager' ) ) . '">' . __( 'oEmbeds', 'instagram-feed' ) . '</a>',
			'<a href="' . esc_url( admin_url( 'admin.php?page=sbi-extensions-manager' ) ) . '">' . __( 'Extensions', 'instagram-feed' ) . '</a>',
			'<a href="' . esc_url( admin_url( 'admin.php?page=sbi-about-us' ) ) . '">' . __( 'About Us', 'instagram-feed' ) . '</a>',
			'<a href="' . esc_url( admin_url( 'admin.php?page=sbi-support' ) ) . '">' . __( 'Support', 'instagram-feed' ) . '</a>',
		);
	}

	/**
	 * Returns an associate array of all existing feeds along with their data
	 *
	 * @return array
	 *
	 * @since 6.0
	 */
	public static function get_feed_list( $feeds_args = array() ) {
		if ( ! empty( $_GET['feed_id'] ) ) {
			return array();
		}
		$feeds_data = SBI_Db::feeds_query( $feeds_args );

		$i = 0;
		foreach ( $feeds_data as $single_feed ) {
			$args  = array(
				'feed_id'       => '*' . $single_feed['id'],
				'html_location' => array( 'content' ),
			);
			$count = \SB_Instagram_Feed_Locator::count( $args );

			$content_locations = \SB_Instagram_Feed_Locator::instagram_feed_locator_query( $args );

			// if this is the last page, add in the header footer and sidebar locations
			if ( count( $content_locations ) < SBI_Db::RESULTS_PER_PAGE ) {

				$args            = array(
					'feed_id'       => '*' . $single_feed['id'],
					'html_location' => array( 'header', 'footer', 'sidebar' ),
					'group_by'      => 'html_location',
				);
				$other_locations = \SB_Instagram_Feed_Locator::instagram_feed_locator_query( $args );

				$locations = array();

				$combined_locations = array_merge( $other_locations, $content_locations );
			} else {
				$combined_locations = $content_locations;
			}

			foreach ( $combined_locations as $location ) {
				$page_text = get_the_title( $location['post_id'] );
				if ( $location['html_location'] === 'header' ) {
					$html_location = __( 'Header', 'instagram-feed' );
				} elseif ( $location['html_location'] === 'footer' ) {
					$html_location = __( 'Footer', 'instagram-feed' );
				} elseif ( $location['html_location'] === 'sidebar' ) {
					$html_location = __( 'Sidebar', 'instagram-feed' );
				} else {
					$html_location = __( 'Content', 'instagram-feed' );
				}
				$shortcode_atts = json_decode( $location['shortcode_atts'], true );
				$shortcode_atts = is_array( $shortcode_atts ) ? $shortcode_atts : array();

				$full_shortcode_string = '[instagram-feed';
				foreach ( $shortcode_atts as $key => $value ) {
					if ( ! empty( $value ) ) {
						$full_shortcode_string .= ' ' . esc_html( $key ) . '="' . esc_html( $value ) . '"';
					}
				}
				$full_shortcode_string .= ']';

				$locations[] = array(
					'link'          => esc_url( get_the_permalink( $location['post_id'] ) ),
					'page_text'     => $page_text,
					'html_location' => $html_location,
					'shortcode'     => $full_shortcode_string
				);
			}
			$feeds_data[ $i ]['instance_count']   = $count;
			$feeds_data[ $i ]['location_summary'] = $locations;
			$settings                             = json_decode( $feeds_data[ $i ]['settings'], true );

			$settings['feed'] = $single_feed['id'];

			$instagram_feed_settings = new \SB_Instagram_Settings( $settings, sbi_defaults() );

			$feeds_data[ $i ]['settings'] = $instagram_feed_settings->get_settings();

			$i++;
		}
		return $feeds_data;
	}

	/**
	 * Returns an associate array of all existing sources along with their data
	 *
	 * @return array
	 *
	 * @since 4.0
	 */
	public function get_legacy_feed_list() {
		if ( ! empty( $_GET['feed_id'] ) ) {
			return array();
		}
		$sbi_statuses = get_option( 'sbi_statuses', array() );
		$sources_list = self::get_source_list();
		if ( empty( $sbi_statuses['support_legacy_shortcode'] ) ) {
			return array();
		}

		$args       = array(
			'html_location' => array( 'header', 'footer', 'sidebar', 'content' ),
			'group_by'      => 'shortcode_atts',
			'page'          => 1
		);
		$feeds_data = \SB_Instagram_Feed_Locator::legacy_instagram_feed_locator_query( $args );
		if ( empty( $feeds_data ) ) {
			$args       = array(
				'html_location' => array( 'header', 'footer', 'sidebar', 'content' ),
				'group_by'      => 'shortcode_atts',
				'page'          => 1
			);
			$feeds_data = \SB_Instagram_Feed_Locator::legacy_instagram_feed_locator_query( $args );
		}

		$feed_saver = new SBI_Feed_Saver( 'legacy' );
		$settings   = $feed_saver->get_feed_settings();

		$default_type = 'timeline';

		if ( isset( $settings['feedtype'] ) ) {
			$default_type = $settings['feedtype'];

		} elseif ( isset( $settings['type'] ) ) {
			if ( strpos( $settings['type'], ',' ) === false ) {
				$default_type = $settings['type'];
			}
		}
		$i       = 0;
		$reindex = false;
		foreach ( $feeds_data as $single_feed ) {
			$args              = array(
				'shortcode_atts' => $single_feed['shortcode_atts'],
				'html_location'  => array( 'content' ),
			);
			$content_locations = \SB_Instagram_Feed_Locator::instagram_feed_locator_query( $args );

			$count = \SB_Instagram_Feed_Locator::count( $args );
			if ( count( $content_locations ) < SBI_Db::RESULTS_PER_PAGE ) {

				$args            = array(
					'feed_id'       => $single_feed['feed_id'],
					'html_location' => array( 'header', 'footer', 'sidebar' ),
					'group_by'      => 'html_location'
				);
				$other_locations = \SB_Instagram_Feed_Locator::instagram_feed_locator_query( $args );

				$combined_locations = array_merge( $other_locations, $content_locations );
			} else {
				$combined_locations = $content_locations;
			}

			$locations = array();
			foreach ( $combined_locations as $location ) {
				$page_text = get_the_title( $location['post_id'] );
				if ( $location['html_location'] === 'header' ) {
					$html_location = __( 'Header', 'instagram-feed' );
				} elseif ( $location['html_location'] === 'footer' ) {
					$html_location = __( 'Footer', 'instagram-feed' );
				} elseif ( $location['html_location'] === 'sidebar' ) {
					$html_location = __( 'Sidebar', 'instagram-feed' );
				} else {
					$html_location = __( 'Content', 'instagram-feed' );
				}
				$shortcode_atts = json_decode( $location['shortcode_atts'], true );
				$shortcode_atts = is_array( $shortcode_atts ) ? $shortcode_atts : array();

				$full_shortcode_string = '[instagram-feed';
				foreach ( $shortcode_atts as $key => $value ) {
					if ( ! empty( $value ) ) {
						if ( is_array( $value ) ) {
							$value = implode( ',', $value );
						}
						$full_shortcode_string .= ' ' . esc_html( $key ) . '="' . esc_html( $value ) . '"';
					}
				}
				$full_shortcode_string .= ']';

				$locations[] = array(
					'link'          => esc_url( get_the_permalink( $location['post_id'] ) ),
					'page_text'     => $page_text,
					'html_location' => $html_location,
					'shortcode'     => $full_shortcode_string
				);
			}
			$shortcode_atts = json_decode( $feeds_data[ $i ]['shortcode_atts'], true );
			$shortcode_atts = is_array( $shortcode_atts ) ? $shortcode_atts : array();

			$full_shortcode_string = '[instagram-feed';
			foreach ( $shortcode_atts as $key => $value ) {
				if ( ! empty( $value ) ) {
					if ( is_array( $value ) ) {
						$value = implode( ',', $value );
					}
					$full_shortcode_string .= ' ' . esc_html( $key ) . '="' . esc_html( $value ) . '"';
				}
			}
			$full_shortcode_string .= ']';

			$feeds_data[ $i ]['shortcode']        = $full_shortcode_string;
			$feeds_data[ $i ]['instance_count']   = $count;
			$feeds_data[ $i ]['location_summary'] = $locations;
			$feeds_data[ $i ]['feed_name']        = self::get_legacy_feed_name( $sources_list, $feeds_data[ $i ]['feed_id'] );
			$feeds_data[ $i ]['feed_type']        = $default_type;

			if ( isset( $shortcode_atts['feedtype'] ) ) {
				$feeds_data[ $i ]['feed_type'] = $shortcode_atts['feedtype'];

			} elseif ( isset( $shortcode_atts['type'] ) ) {
				if ( strpos( $shortcode_atts['type'], ',' ) === false ) {
					$feeds_data[ $i ]['feed_type'] = $shortcode_atts['type'];
				}
			}

			if ( isset( $feeds_data[ $i ]['id'] ) ) {
				unset( $feeds_data[ $i ]['id'] );
			}

			if ( isset( $feeds_data[ $i ]['html_location'] ) ) {
				unset( $feeds_data[ $i ]['html_location'] );
			}

			if ( isset( $feeds_data[ $i ]['last_update'] ) ) {
				unset( $feeds_data[ $i ]['last_update'] );
			}

			if ( isset( $feeds_data[ $i ]['post_id'] ) ) {
				unset( $feeds_data[ $i ]['post_id'] );
			}

			if ( ! empty( $shortcode_atts['feed'] ) ) {
				$reindex = true;
				unset( $feeds_data[ $i ] );
			}

			if ( isset( $feeds_data[ $i ]['shortcode_atts'] ) ) {
				unset( $feeds_data[ $i ]['shortcode_atts'] );
			}

			$i++;
		}

		if ( $reindex ) {
			$feeds_data = array_values( $feeds_data );
		}

		// if there were no feeds found in the locator table we still want the legacy settings to be available
		// if it appears as though they had used version 3.x or under at some point.
		if ( empty( $feeds_data )
			 && ! is_array( $sbi_statuses['support_legacy_shortcode'] )
			 && ( $sbi_statuses['support_legacy_shortcode'] ) ) {

			$feeds_data = array(
				array(
					'feed_id'          => __( 'Legacy Feed', 'instagram-feed' ) . ' ' . __( '(unknown location)', 'instagram-feed' ),
					'feed_name'        => __( 'Legacy Feed', 'instagram-feed' ) . ' ' . __( '(unknown location)', 'instagram-feed' ),
					'shortcode'        => '[instagram-feed]',
					'feed_type'        => '',
					'instance_count'   => false,
					'location_summary' => array()
				)
			);
		}

		return $feeds_data;
	}

	public static function get_legacy_feed_name( $sources_list, $source_id ) {
		foreach ( $sources_list as $source ) {
			if ( $source['account_id'] == $source_id ) {
				return $source['username'];
			}
		}
		return $source_id;
	}

	/**
	 * Status of the onboarding sequence for specific user
	 *
	 * @return string|boolean
	 *
	 * @since 6.0
	 */
	public static function onboarding_status( $type = 'newuser' ) {
		$onboarding_statuses = get_user_meta( get_current_user_id(), 'sbi_onboarding', true );
		$status              = false;
		if ( ! empty( $onboarding_statuses ) ) {
			$statuses = maybe_unserialize( $onboarding_statuses );
			$status   = isset( $statuses[ $type ] ) ? $statuses[ $type ] : false;
		}

		return $status;
	}

	/**
	 * Update status of onboarding sequence for specific user
	 *
	 * @return string|boolean
	 *
	 * @since 6.0
	 */
	public static function update_onboarding_meta( $value, $type = 'newuser' ) {
		$onboarding_statuses = get_user_meta( get_current_user_id(), 'sbi_onboarding', true );
		if ( ! empty( $onboarding_statuses ) ) {
			$statuses          = maybe_unserialize( $onboarding_statuses );
			$statuses[ $type ] = $value;
		} else {
			$statuses = array(
				$type => $value
			);
		}

		$statuses = maybe_serialize( $statuses );

		update_user_meta( get_current_user_id(), 'sbi_onboarding', $statuses );
	}

	/**
	 * Used to dismiss onboarding using AJAX
	 *
	 * @since 6.0
	 */
	public static function after_dismiss_onboarding() {
		check_ajax_referer( 'sbi-admin', 'nonce' );

		$cap = current_user_can( 'manage_instagram_feed_options' ) ? 'manage_instagram_feed_options' : 'manage_options';
		$cap = apply_filters( 'sbi_settings_pages_capability', $cap );

		if ( current_user_can( $cap ) ) {
			$type = 'newuser';
			if ( isset( $_POST['was_active'] ) ) {
				$type = sanitize_key( $_POST['was_active'] );
			}
			self::update_onboarding_meta( 'dismissed', $type );
		}
		wp_die();
	}

	public static function add_customizer_att( $atts ) {
		if ( ! is_array( $atts ) ) {
			$atts = array();
		}
		$atts['feedtype'] = 'customizer';
		return $atts;
	}

	/**
	 * Feed Builder Wrapper.
	 *
	 * @since 6.0
	 */
	public function feed_builder() {
		include_once SBI_BUILDER_DIR . 'templates/builder.php';
	}

	/**
	 * For types listed on the top of the select feed type screen
	 *
	 * @return array
	 *
	 * @since 6.0
	 */
	public function get_feed_types() {
		$feed_types = array(
			array(
				'type'        => 'user',
				'title'       => __( 'User Timeline', 'instagram-feed' ),
				'description' => __( 'Fetch posts from your Instagram profile', 'instagram-feed' ),
				'icon'        => 'usertimelineIcon'
			),
			array(
				'type' => 'hashtag',
				'title' => __( 'Public Hashtag', 'instagram-feed' ),
				'description' => __( 'Fetch posts from a public Instagram hashtag', 'instagram-feed' ),
				'tooltip' => __( 'Hashtag feeds require a connected Instagram business account', 'instagram-feed' ),
				'businessRequired' => true,
				'icon'	=>  'publichashtagIcon'
			),
			array(
				'type' => 'tagged',
				'title' => __( 'Tagged Posts', 'instagram-feed' ),
				'description' => __( 'Display posts your Instagram account has been tagged in', 'instagram-feed' ),
				'tooltip' => __( 'Tagged posts feeds require a connected Instagram business account', 'instagram-feed' ),
				'businessRequired' => true,
				'icon'	=>  'taggedpostsIcon'
			),
			array(
				'type' => 'socialwall',
				'title' => __( 'Social Wall', 'instagram-feed' ),
				'description' => __( 'Create a feed with sources from different social platforms', 'instagram-feed' ),
				'icon'	=>  'socialwall1Icon'
			)
		);

		return $feed_types;
	}

	/**
	 * For types listed on the bottom of the select feed type screen
	 *
	 * @return array
	 *
	 * @since 6.0
	 */
	public function get_advanced_feed_types() {
		$feed_types = array(
			array(
				'type'             => 'hashtag',
				'title'            => __( 'Public Hashtag', 'instagram-feed' ),
				'description'      => __( 'Fetch posts from a public Instagram hashtag', 'instagram-feed' ),
				'tooltip'          => __( 'Hashtag feeds require a connected Instagram business account', 'instagram-feed' ),
				'businessRequired' => true,
				'icon'             => 'publichashtagIcon'
			),
			array(
				'type'             => 'tagged',
				'title'            => __( 'Tagged Posts', 'instagram-feed' ),
				'description'      => __( 'Display posts your Instagram account has been tagged in', 'instagram-feed' ),
				'tooltip'          => __( 'Tagged posts feeds require a connected Instagram business account', 'instagram-feed' ),
				'businessRequired' => true,
				'icon'             => 'taggedpostsIcon'
			),
			array(
				'type'        => 'socialwall',
				'title'       => __( 'Social Wall', 'instagram-feed' ),
				'description' => __( 'Create a feed with sources from different social platforms', 'instagram-feed' ),
				'icon'        => 'socialwall1Icon'
			),
		);

		return $feed_types;
	}

	/**
	 * Personal Account
	 *
	 * @return array
	 *
	 * @since 6.0.8
	 */
	public static function personal_account_screen_text() {
		return array(
			'mainHeading1'    => __( 'We’re almost there...', 'instagram-feed' ),
			'mainHeading2'    => __( 'Update Personal Account', 'instagram-feed' ),
			'mainHeading3'    => __( 'Add Instagram Profile Picture and Bio', 'instagram-feed' ),
			'mainDescription' => __( 'Instagram does not provide us access to your profile picture or bio for personal accounts. Would you like to set up a custom profile photo and bio?.', 'instagram-feed' ),
			'bioLabel'        => __( 'Bio (140 Characters)', 'instagram-feed' ),
			'bioPlaceholder'  => __( 'Add your profile bio here', 'instagram-feed' ),
			'confirmBtn'      => __( 'Yes, let\'s do it', 'instagram-feed' ),
			'cancelBtn'       => __( 'No, maybe later', 'instagram-feed' ),
			'uploadBtn'       => __( 'Upload Profile Picture', 'instagram-feed' )
		);
	}

	/**
	 * Get Smahballoon Plugins Info
	 *
	 * @since 6.2.9
	 */
	public static function get_smashballoon_plugins_info() {
		$active_sb_plugins = Util::get_sb_active_plugins_info();

        return [
            'facebook' => [
                'installed' => $active_sb_plugins['is_facebook_installed'],
                'class' => 'CFF_Elementor_Widget',
				'link' => 'https://smashballoon.com/custom-facebook-feed/',
                'icon' => '<svg width="40" height="40" viewBox="0 0 40 40" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M19.198 4.4374C10.615 4.88722 3.95971 12.2621 4.41198 20.8919C4.82091 28.6946 10.8719 34.8716 18.3927 35.651L17.8202 24.7272L13.8564 24.935L13.6192 20.4094L17.583 20.2017L17.4022 16.7528C17.197 12.8359 19.4093 10.5605 22.983 10.3732C24.684 10.284 26.4785 10.4873 26.4785 10.4873L26.6805 14.3418L24.7142 14.4449C22.7792 14.5463 22.2335 15.7798 22.2981 17.0127L22.4519 19.9465L26.7902 19.7191L26.3251 24.2815L22.6891 24.4721L23.2616 35.3959C26.9085 34.6224 30.1587 32.5706 32.4255 29.6109C34.6923 26.6513 35.8264 22.9787 35.6229 19.2562C35.1706 10.6264 27.781 3.98759 19.198 4.4374Z" fill="#006BFA"/></svg>',
                'description' => __('Custom Facebook Feeds is a highly customizable way to display tweets from your Facebook account. Promote your latest content and update your site content automatically.', 'instagram-feed'),
				'download_plugin' => 'https://downloads.wordpress.org/plugin/custom-facebook-feed.zip',
            ],
            'instagram' => [
                'installed' => $active_sb_plugins['is_instagram_installed'],
                'class' => 'SBI_Elementor_Widget',
				'link' => 'https://smashballoon.com/instagram-feed/',
                'icon' => '<svg width="36" height="36" viewBox="0 0 36 36" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M18 9.91406C13.5 9.91406 9.91406 13.5703 9.91406 18C9.91406 22.5 13.5 26.0859 18 26.0859C22.4297 26.0859 26.0859 22.5 26.0859 18C26.0859 13.5703 22.4297 9.91406 18 9.91406ZM18 23.2734C15.1172 23.2734 12.7266 20.9531 12.7266 18C12.7266 15.1172 15.0469 12.7969 18 12.7969C20.8828 12.7969 23.2031 15.1172 23.2031 18C23.2031 20.9531 20.8828 23.2734 18 23.2734ZM28.2656 9.63281C28.2656 8.57812 27.4219 7.73438 26.3672 7.73438C25.3125 7.73438 24.4688 8.57812 24.4688 9.63281C24.4688 10.6875 25.3125 11.5312 26.3672 11.5312C27.4219 11.5312 28.2656 10.6875 28.2656 9.63281ZM33.6094 11.5312C33.4688 9 32.9062 6.75 31.0781 4.92188C29.25 3.09375 27 2.53125 24.4688 2.39062C21.8672 2.25 14.0625 2.25 11.4609 2.39062C8.92969 2.53125 6.75 3.09375 4.85156 4.92188C3.02344 6.75 2.46094 9 2.32031 11.5312C2.17969 14.1328 2.17969 21.9375 2.32031 24.5391C2.46094 27.0703 3.02344 29.25 4.85156 31.1484C6.75 32.9766 8.92969 33.5391 11.4609 33.6797C14.0625 33.8203 21.8672 33.8203 24.4688 33.6797C27 33.5391 29.25 32.9766 31.0781 31.1484C32.9062 29.25 33.4688 27.0703 33.6094 24.5391C33.75 21.9375 33.75 14.1328 33.6094 11.5312ZM30.2344 27.2812C29.7422 28.6875 28.6172 29.7422 27.2812 30.3047C25.1719 31.1484 20.25 30.9375 18 30.9375C15.6797 30.9375 10.7578 31.1484 8.71875 30.3047C7.3125 29.7422 6.25781 28.6875 5.69531 27.2812C4.85156 25.2422 5.0625 20.3203 5.0625 18C5.0625 15.75 4.85156 10.8281 5.69531 8.71875C6.25781 7.38281 7.3125 6.32812 8.71875 5.76562C10.7578 4.92188 15.6797 5.13281 18 5.13281C20.25 5.13281 25.1719 4.92188 27.2812 5.76562C28.6172 6.25781 29.6719 7.38281 30.2344 8.71875C31.0781 10.8281 30.8672 15.75 30.8672 18C30.8672 20.3203 31.0781 25.2422 30.2344 27.2812Z" fill="url(#paint0_linear)"></path><defs><linearGradient id="paint0_linear" x1="13.4367" y1="62.5289" x2="79.7836" y2="-5.19609" gradientUnits="userSpaceOnUse"><stop stop-color="white"></stop><stop offset="0.147864" stop-color="#F6640E"></stop><stop offset="0.443974" stop-color="#BA03A7"></stop><stop offset="0.733337" stop-color="#6A01B9"></stop><stop offset="1" stop-color="#6B01B9"></stop></linearGradient></defs></svg>',
                'description' => __('Instagram Feeds is a highly customizable way to display tweets from your Instagram account. Promote your latest content and update your site content automatically.', 'instagram-feed'),
				'download_plugin' => 'https://downloads.wordpress.org/plugin/instagram-feed.zip',
            ],
            'twitter' => [
                'installed' => $active_sb_plugins['is_twitter_installed'],
                'class' => 'CTF_Elementor_Widget',
				'link' => 'https://smashballoon.com/custom-twitter-feeds/',
                'icon' => '<svg width="31" height="27" viewBox="0 0 31 27" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M30.2343 3.19694C29.1196 3.77582 27.9066 4.18141 26.6398 4.41138C27.9036 3.55701 28.8468 2.25846 29.2428 0.735778C28.0508 1.54178 26.7137 2.13232 25.2907 2.5043C24.0523 1.29034 22.3674 0.63512 20.4988 0.733046C17.0138 0.915688 14.3157 3.91223 14.4999 7.42689C14.5263 7.9311 14.6113 8.41738 14.7392 8.87166C9.4458 8.88141 4.61186 6.59188 1.28148 2.91496C0.781741 3.87799 0.527828 4.99171 0.588449 6.14844C0.704251 8.35808 1.91908 10.2573 3.69762 11.2794C2.64471 11.3346 1.6504 11.0893 0.766952 10.6895L0.769284 10.734C0.930941 13.8185 3.26098 16.2839 6.19793 16.7099C5.27812 17.0142 4.30028 17.1011 3.34122 16.9637C3.81026 18.2068 4.65708 19.2717 5.76263 20.0086C6.86818 20.7455 8.17687 21.1175 9.50474 21.0721C7.3493 22.9702 4.61227 24.0786 1.74347 24.2151C1.23926 24.2415 0.733488 24.2383 0.226167 24.2054C3.13864 25.8669 6.54536 26.7442 10.1342 26.5561C21.82 25.9437 27.7331 15.9085 27.2924 7.49999C27.2776 7.21822 27.2636 6.95129 27.234 6.6703C28.4331 5.71523 29.4418 4.5322 30.2343 3.19694Z" fill="#1B90EF"/></svg>',
                'description' => __('Custom Twitter Feeds is a highly customizable way to display tweets from your Twitter account. Promote your latest content and update your site content automatically.', 'instagram-feed'),
				'download_plugin' => 'https://downloads.wordpress.org/plugin/custom-twitter-feeds.zip',
            ],
            'youtube' => [
                'installed' => $active_sb_plugins['is_youtube_installed'],
                'class' => 'SBY_Elementor_Widget',
				'link' => 'https://smashballoon.com/youtube-feed/',
                'icon' => '<svg width="36" height="36" viewBox="0 0 36 36" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M15 22.5L22.785 18L15 13.5V22.5ZM32.34 10.755C32.535 11.46 32.67 12.405 32.76 13.605C32.865 14.805 32.91 15.84 32.91 16.74L33 18C33 21.285 32.76 23.7 32.34 25.245C31.965 26.595 31.095 27.465 29.745 27.84C29.04 28.035 27.75 28.17 25.77 28.26C23.82 28.365 22.035 28.41 20.385 28.41L18 28.5C11.715 28.5 7.8 28.26 6.255 27.84C4.905 27.465 6.035 26.595 3.66 25.245C3.465 24.54 3.33 23.595 3.24 22.395C3.135 21.195 3.09 20.16 3.09 19.26L3 18C3 14.715 3.24 12.3 3.66 10.755C6.035 9.405 4.905 8.535 6.255 8.16C6.96 7.965 8.25 7.83 10.23 7.74C12.18 7.635 13.965 7.59 15.615 7.59L18 7.5C24.285 7.5 28.2 7.74 29.745 8.16C31.095 8.535 31.965 9.405 32.34 10.755Z" fill="#EB2121"></path></svg>',
                'description' => __('YouTube Feeds is a highly customizable way to display tweets from your YouTube account. Promote your latest content and update your site content automatically.', 'instagram-feed'),
				'download_plugin' => 'https://downloads.wordpress.org/plugin/feeds-for-youtube.zip',
            ]
        ];
	}

}

